<?php

declare(strict_types=1);

namespace Crossroads\Magento\Test\Integration;

use Mage;
use Mage_Catalog_Helper_Category_Flat;
use Mage_Catalog_Helper_Product_Flat;
use Mage_Core_Model_Locale;
use Mage_Core_Model_Resource_Setup;
use Mage_Core_Model_Store;
use Mage_Directory_Model_Currency;
use Zend_Controller_Request_Abstract;
use Varien_Profiler;

class MagentoManager {
    const TESTING_STORE = "testing";
    const HOST_VALUE = "https://example.com";

    protected static $cryptKey = "9c0fb682-ef68-4842-8ed8-0cde0a683610";

    public static function setCryptKey($key) {
        self:$cryptKey = $key;
    }

    public static function getDatabaseConfig(): array {
        return [
            [
                "name" => "profiler",
                "value" => $_ENV["DB_PROFILER"] ??  "0",
            ],
            [
                "name" => "host",
                "value" => $_ENV["DB_HOST"] ??  "127.0.0.1",
            ],
            [
                "name" => "username",
                "value" => $_ENV["DB_USER"] ?? "root",
            ],
            [
                "name" => "password",
                "value" => $_ENV["DB_PASSWORD"] ?? "",
            ],
            [
                "name" => "dbname",
                "value" => $_ENV["DB_NAME"] ?? "testing_magento",
            ],
            [
                "name" => "active",
                "value" => "1",
            ],
        ];
    }

    public static function getConfig(): array {
        return [
            [
                "name" => "global",
                "children" => [
                    [
                        "name" => "install",
                        "children" => [
                            [
                                "name" => "date",
                                "value" => date(DATE_RFC822),
                            ],
                        ],
                    ],
                    [
                        "name" => "crypt",
                        "children" => [
                            [
                                "name" => "key",
                                "value" => self::$cryptKey,
                            ],
                        ],
                    ],
                    [
                        "name" => "disable_local_modules",
                        "value" => "false",
                    ],
                    [
                        "name" => "resources",
                        "children" => [
                            [
                                "name" => "db",
                                "children" => [
                                    [
                                        "name" => "table_prefix",
                                        "value" => "",
                                    ],
                                ],
                            ],
                            [
                                "name" => "default_setup",
                                "children" => [
                                    [
                                        "name" => "connection",
                                        "children" => self::getDatabaseConfig(),
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ]
        ];
    }

    public static function setUp() {
        // A fix for PHP serialize issues, see https://wiki.php.net/rfc/precise_float_value
        ini_set("serialize_precision", "-1");

        require_once __DIR__ . "/../../app/Mage.php";

        umask(0);

        Config::setConfig(self::getConfig());

        if( ! class_exists("Mage_Core_Controller_Response_Http", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Response", "Mage_Core_Controller_Response_Http");
        }

        if( ! class_exists("Mage_Core_Model_Cookie", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Cookie", "Mage_Core_Model_Cookie");
        }

        if( ! class_exists("Mage_Catalog_Model_Product_Image", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Catalog\\Image", "Mage_Catalog_Model_Product_Image");
        }

        if( ! class_exists("Mage_Catalog_Model_Product_Attribute_Backend_Media", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Catalog\\Product\\Attribute\\Media", "Mage_Catalog_Model_Product_Attribute_Backend_Media");
        }

        if( ! class_exists("Mage_Core_Model_Session_Abstract_Varien", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Session", "Mage_Core_Model_Session_Abstract_Varien");
        }

        if( ! class_exists("Varien_Profiler", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Profiler", "Varien_Profiler");
        }

        Mage::reset();
        gc_collect_cycles();
        Mage::setIsDeveloperMode(true);
        Mage::register("isSecureArea", true);
        Mage::app("admin", "store", [
            "is_installed" => false,
            "config_model" => __NAMESPACE__ . "\\Config",
        ])->getCacheInstance()->flush();

        self::createVarFolder();

        Mage_Core_Model_Resource_Setup::applyAllUpdates();

        self::configure();

        // TODO: Use a DB-cache here if possible?

        // Reinit to run data updates since they require store
        self::initAdmin();

        Mage_Core_Model_Resource_Setup::applyAllDataUpdates();

        self::createStores();

        // TODO: Load test data

        // TODO: Reindex

        // self::logQueries();

        self::reset();
    }

    public static function logQueries() {
        if( ! Mage::getConfig()) {
            return;
        }

        $var = Mage::getBaseDir("var");
        $profiler = Mage::getSingleton("core/resource")->getConnection("core_write")->getProfiler();
        $queries = $profiler->getQueryProfiles();

        if( ! empty($queries)) {
            $queries = array_map(function($q) {
                $t = $q->getStartedMicrotime();
                $micro = sprintf("%06d",($t - floor($t)) * 1000000);
                $date = gmdate("Y-m-d\TH:i:s.".$micro."Z", (int)$t);
                $params = implode(", ", array_map(function($v, $k = "") {
                    return ($k ? $k.": " : "").$v;
                }, $q->getQueryParams()));

                return sprintf("[%s] %f: %s (%s)", $date, $q->hasEnded() ? $q->getElapsedSecs() : -1, $q->getQuery(), $params);
            }, $queries);
        }

        if( ! file_exists($var.DS."log")) {
            mkdir($var.DS."log", 0777, true);
        }

        file_put_contents($var.DS."log".DS."query.log", implode("\n", $queries), FILE_APPEND);
    }

    public static function createVarFolder() {
        $var = Mage::getBaseDir("var");

        if( ! file_exists($var.DS."log")) {
            mkdir($var.DS."log", 0777, true);
        }

        self::createEmptyFile($var.DS."log".DS."system.log");
        self::createEmptyFile($var.DS."log".DS."exception.log");
    }

    public static function createEmptyFile($file) {
        if(file_exists($file)) {
            $f = fopen($file, "w");

            if( ! $f) {
                throw new Exception("Failed to open file '$f'.");
            }

            if( ! ftruncate($f, 0)) {
                throw new Exception("Failed to truncate file '$f'.");
            }

            fclose($f);
        }
        else {
            touch($file);
        }
    }

    public static function getSystemLog(): string {
        return self::readLogFile("system.log");
    }

    public static function getExceptionLog(): string {
        return self::readLogFile("exception.log");
    }

    public static function readLogFile(string $file): string {
        if( ! Mage::getConfig()) {
            throw new Exception("Magento has not been initialized");
        }

        return file_get_contents(Mage::getBaseDir("var").DS."log".DS.$file);
    }

    public static function configure() {
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        // Clear store-specific config
        $setupModel->deleteConfigData(Mage_Core_Model_Store::XML_PATH_STORE_STORE_NAME);
        $setupModel->deleteConfigData(Mage_Core_Model_Store::XML_PATH_UNSECURE_BASE_URL);
        $setupModel->deleteConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_BASE_URL);

        // General configuration
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_USE_REWRITES, 1);
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_IN_FRONTEND, 0);
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_OFFLOADER_HEADER, "SSL_OFFLOADED");
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_STORE_STORE_NAME, "Default Magento");
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_UNSECURE_BASE_URL, "https://default.test/");
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_BASE_URL, "https://default.test/");
        $setupModel->setConfigData(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE, "sv_SE");
        $setupModel->setConfigData(Mage_Core_Model_Locale::XML_PATH_DEFAULT_COUNTRY, "SE");
        $setupModel->setConfigData(Mage_Core_Model_Locale::XML_PATH_DEFAULT_TIMEZONE, "CET");
        $setupModel->setConfigData(Mage_Directory_Model_Currency::XML_PATH_CURRENCY_BASE, "SEK");
        $setupModel->setConfigData(Mage_Directory_Model_Currency::XML_PATH_CURRENCY_DEFAULT, "SEK");
        $setupModel->setConfigData(Mage_Directory_Model_Currency::XML_PATH_CURRENCY_ALLOW, "SEK");
        $setupModel->setConfigData("general/country/allow", "DE,DK,SE,NO,FI,GB");
        $setupModel->setConfigData("design/head/default_title", "Default Title");
        $setupModel->setConfigData("design/head/title_prefix", "Default Magento |");
        $setupModel->setConfigData("shipping/origin/country_id", "SE");
        $setupModel->setConfigData("web/default/no_route", "cms/index/noRoute");
        $setupModel->setConfigData("web/url/redirect_to_base", 0);
        $setupModel->setConfigData("dev/log/file", "system.log");
        $setupModel->setConfigData("dev/log/exception_file", "exception.log");
    }

    public static function createStores() {
        $website = Mage::getModel("core/website");
        $group = Mage::getModel("core/store_group");
        $store = Mage::getModel("core/store");

        $website->load(static::TESTING_STORE);

        if($website->getDefaultGroupId()) {
            $group->load($website->getDefaultGroupId());
        }

        $store->load(static::TESTING_STORE);

        $website->addData([
            "code" => static::TESTING_STORE,
            "is_default" => 0,
            "name" => "Test Website",
            "sort_order" => 1,
        ])->save();

        $group->addData([
            "name" => "Test Store Group",
            "website_id" => $website->getId(),
        ])->save();

        $store->addData([
            "code" => static::TESTING_STORE,
            "group_id" => $group->getId(),
            "is_active" => 1,
            "name" => "Test Store",
            "sort_order" => 1,
            "website_id" => $website->getId(),
        ])->save();

        self::configureStore($store);

        self::createCategories($store, $group);
    }

    private static function configureStore(Mage_Core_Model_Store $store) {
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_STORE_STORE_NAME, "Testing Magento", "stores", $store->getId());
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_UNSECURE_BASE_URL, "https://example.com/", "stores", $store->getId());
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_BASE_URL, "https://example.com/", "stores", $store->getId());
        $setupModel->setConfigData("design/head/default_title", "Testing Title", "stores", $store->getId());
        $setupModel->setConfigData("design/head/title_prefix", "Testing Magento |", "stores", $store->getId());

        /*
        $taxClass = Mage::getModel("tax/class");

        $taxClass->load("Test Rate", "class_name");

        $taxClass->addData([
            "class_name" => "Test Rate",
            "class_type" => Mage_Tax_Model_Class::TAX_CLASS_TYPE_PRODUCT,
        ]);

        $taxClass->save();
         */
    }

    /**
     * Since we are not actually reindexing after every small modification we
     * have to load data by id indirectly, and use custom queries to obtain them.
     */
    public static function loadCategoryByAttribute($attrType, $attr, $value) {
        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        $id = $conn->query("SELECT v.entity_id FROM catalog_category_entity_$attrType v
            JOIN eav_attribute e ON e.attribute_id = v.attribute_id AND e.entity_type_id = v.entity_type_id
            WHERE e.attribute_code = ? AND v.value = ?", [$attr, $value])->fetchColumn();


        $category = Mage::getModel("catalog/category");

        if($id) {
            $category->load($id);
        }

        return $category;
    }

    public static function createCategories($store, $group) {
        $root = self::loadCategoryByAttribute("varchar", "name", "The Test Root");

        $root->addData([
            "url_key" => "",
            "name" => "The Test Root",
            "atttribute_set_id" => $root->getDefaultAttributeSetId(),
            "description" => "Testing root category",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for Root Category",
            "meta_description" => "Meta description for Root Category",
            "meta_keywords" => "Meta keywords for Root Category",
            "image" => "category_test_root_image.jpg",
            "is_active" => 1,
            "is_anchor" => 0,
            "store_id" => $store->getId(),
            "path" => $root->getPath() ?: "1",
            "parent_id" => 1,
        ]);

        $root->save();

        $group->setRootCategoryId($root->getId())->save();

        $category = self::loadCategoryByAttribute("varchar", "url_key", "test-category");

        $category->addData([
            "url_key" => "test-category",
            "name" => "The Test Category",
            "atttribute_set_id" => $category->getDefaultAttributeSetId(),
            "description" => "Testing A test category",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for Category",
            "meta_description" => "Meta description for Category",
            "meta_keywords" => "Meta keywords for Category",
            "image" => "category_test_image.jpg",
            "is_active" => 1,
            "is_anchor" => 0,
            "path" => $category->getPath() ?: $root->getPath(),
            "parent_id" => $root->getId(),
            "store_id" => $store->getId(),
        ]);

        $category->save();

        $inactiveCategory = self::loadCategoryByAttribute("varchar", "url_key", "test-inactive-category");

        $inactiveCategory->addData([
            "url_key" => "test-inactive-category",
            "name" => "The Inactive Test Category",
            "atttribute_set_id" => $inactiveCategory->getDefaultAttributeSetId(),
            "description" => "Testing an inactive test category",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for Inactive Category",
            "meta_description" => "Meta description for Inactive Category",
            "meta_keywords" => "Meta keywords for Inactive Category",
            "image" => "category_test_inactive_image.jpg",
            "is_active" => 0,
            "is_anchor" => 0,
            "path" => $inactiveCategory->getPath() ?: $root->getPath(),
            "parent_id" => $root->getId(),
            "store_id" => $store->getId(),
        ]);

        $inactiveCategory->save();

        $childCategory = self::loadCategoryByAttribute("varchar", "url_key", "test-child-category");

        $childCategory->addData([
            "url_key" => "test-child-category",
            "name" => "A Child Category",
            "atttribute_set_id" => $childCategory->getDefaultAttributeSetId(),
            "description" => "Test a child category type",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for a Child Category",
            "meta_description" => "Meta description for a Child Category",
            "meta_keywords" => "Meta keywords for a Child Category",
            "image" => "category_child_test_image.jpg",
            "is_active" => 1,
            "is_anchor" => 0,
            "path" => $childCategory->getPath() ?: $category->getPath(),
            "parent_id" => $category->getId(),
            "store_id" => $store->getId(),
        ]);

        $childCategory->save();
    }

    public static function reindex() {
        self::initAdmin();

        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/indexer_url")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();
        Mage::getSingleton("catalog/category_indexer_flat")->reindexAll();
        Mage::getSingleton("catalog/category_indexer_product")->reindexAll();
        Mage::getResourceSingleton("cataloginventory/indexer_stock")->reindexAll();
        Mage::getSingleton("catalogsearch/indexer_fulltext")->reindexAll();
    }

    public static function initAdmin() {
        $options = [
            "is_installed" => true,
            "config_model" => __NAMESPACE__ . "\\Config",
        ];
        Mage::reset();
        gc_collect_cycles();
        Mage::setIsDeveloperMode(true);
        Mage::register("isSecureArea", true);
        Mage::init("admin", "store", $options);
    }

    public static function init() {
        $options = [
            "is_installed" => true,
            "config_model" => __NAMESPACE__ . "\\Config",
        ];
        Mage::app(self::TESTING_STORE, "store", $options);
    }

    public static function reset() {
        $options = [
            "is_installed" => true,
            "config_model" => __NAMESPACE__ . "\\Config",
        ];
        Mage::app("admin", "store", $options)->getCacheInstance()->flush();
        self::createVarFolder();
        Mage::reset();
        Varien_Profiler::clear();
        gc_collect_cycles();
        Mage::setIsDeveloperMode(true);
    }

    public static function runRequest(Zend_Controller_Request_Abstract $request) {
        if( ! $request->getHeader("Host")) {
            $request->setHeader("Host", self::HOST_VALUE);
        }

        $options = [
            "is_installed" => true,
            "config_model" => __NAMESPACE__ . "\\Config",
        ];
        $app = Mage::app(self::TESTING_STORE, "store", $options);

        $app->setRequest($request);
        $app->run([
            "scope_code" => self::TESTING_STORE,
            "scope_type" => "store",
            "options"    => $options,
        ]);

        return $app->getResponse();
    }
}
