<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use Stripe\Exception\ExceptionInterface;

class Crossroads_Stripe_Model_Graphql_Payment extends MageQL_Sales_Model_Payment_Abstract {
    const SUCCESS = "success";
    const REQUIRES_ACTION = "requiresAction";
    const RECREATE_INTENT = "recreateIntent";
    const ERROR_MESSAGE = "errorMessage";
    const ERROR_INVALID_INTENT_STATUS = "errorInvalidIntentStatus";

    public function getCode(): string {
        return "Crossroads_Stripe_PaymentIntents";
    }

    public function getMethodTypeName(): string {
        return "Stripe";
    }

    public function getPaymentMethodFields(Registry $registry): array {
        return [];
    }

    public function getQuotePaymentFields(Registry $registry): array {
        return [
            "publishableKey" => $this->field("ID!", "The publishable key for stripe SDK")
                ->setResolver([$this, "resolvePublishableKey"]),
        ];
    }

    public function getMutateSetQuotePaymentFields(Registry $registry): array {
        return [
            "result" => $this->field(
                $this->getMutationResultType()."!",
                "Result of attempting to set the payment method to Stripe"
            )
                // Simple result, we either succeed or fail
                ->setResolver("MageQL\\forwardResolver"),
            "publishableKey" => $this->field("ID!", "The publishable key for stripe SDK")
                ->setResolver([$this, "resolvePublishableKey"]),
        ];
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "StripePaymentIntentResult":
            return $this->interface("The result from trying to create a payment intent with stripe.")
                ->setTypeResolver(function($value) {
                    switch($value->getResult()) {
                    case self::REQUIRES_ACTION:
                        return "StripePaymentIntentResultRequiresAction";
                    case self::ERROR_MESSAGE:
                        return "StripePaymentIntentResultError";
                    default:
                        return "StripePaymentIntentResultSimple";
                    }
                });

        case "StripePaymentIntentResultSimple":
            return $this->object("A single value result for creating a payment intent with stripe.")
                ->setInterfaces(["StripePaymentIntentResult"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "StripePaymentIntentResultRequiresAction":
            return $this->object("The Payment intent requires an action from the customer.")
                ->setInterfaces(["StripePaymentIntentResult"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "StripePaymentIntentResultError":
            return $this->object("There was an error creating/confirming the payment intent.")
                ->setInterfaces(["StripePaymentIntentResult"])
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "StripePaymentIntentResultType":
            return $this->enum("Type of result from createStripePaymentIntent", [
                self::SUCCESS => [
                    "description" => "Payment intent was successfully created",
                ],
                self::REQUIRES_ACTION => [
                    "description" => "Payment intent requires an action from frontend",
                ],
                self::RECREATE_INTENT => [
                    "description" => "The client needs to recreate the intent since new information has been received.",
                ],
                self::ERROR_MESSAGE => [
                    "description" => "Error with a message for the user",
                ],
                self::ERROR_INVALID_INTENT_STATUS => [
                    "description" => "The intent status is invalid or not recognized",
                ],
            ]);

        default:
            return parent::getTypeBuilder($typeName, $registry);
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "StripePaymentIntentResult":
            return [
                "result" => $this->field("StripePaymentIntentResultType!", "The result type"),
            ];

        case "StripePaymentIntentResultSimple":
            return $registry->getFieldBuilders("StripePaymentIntentResult");

        case "StripePaymentIntentResultRequiresAction":
            return array_merge($registry->getFieldBuilders("StripePaymentIntentResult"), [
                "clientSecret" => $this->field("ID!", "The client secret for the payment intent if it requires an action"),
            ]);

        case "StripePaymentIntentResultError":
            return array_merge($registry->getFieldBuilders("StripePaymentIntentResult"), [
                "error" => $this->field("String!", "The client visible error."),
            ]);

        case "Mutation":
            return [
                "createStripePaymentIntent" => $this->field("StripePaymentIntentResult!", "Creates a stripe payment intent")
                    ->addArgument("paymentMethod", $this->argument("ID!", "Stripe payment method id from stripe client API"))
                    ->setResolver([$this, "mutateCreateIntent"]),
                "confirmStripePaymentIntent" => $this->field("StripePaymentIntentResult!", "Confirms a stripe payment intent")
                    ->addArgument("intent", $this->argument("ID!", "Stripe payment intent id from stripe client API"))
                    ->setResolver([$this, "mutateConfirmIntent"]),
            ];
        default:
            return parent::getTypeFields($typeName, $registry);
        }
    }

    public function resolvePublishableKey($src, array $args, MageQL_Core_Model_Context $ctx) {
        $helper = Mage::helper("Crossroads_Stripe/config");

        return $helper->publishableKey($ctx->getStore());
    }

    public function mutateCreateIntent($src, array $args, $ctx) {
        $helper = Mage::helper("Crossroads_Stripe/intent");
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();

        try {
            $intent = $helper->create($quote, $args["paymentMethod"]);

            $helper->assignIntentToPayment($quote->getPayment(), $intent);

            return $this->getIntentResult($quote, $intent);
        }
        catch(ExceptionInterface $e) {
            Mage::logException($e);

            return new Varien_Object([
                "result" => self::ERROR_MESSAGE,
                "client_secret" => null,
                "error" => $e->getMessage(),
            ]);
        }
    }

    public function mutateConfirmIntent($src, array $args, $ctx) {
        $helper = Mage::helper("Crossroads_Stripe/intent");
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();

        try {
            $intent = $helper->retrieve($quote->getStore(), $args["intent"]);

            if((int)($quote->getGrandTotal() * 100) !== (int)$intent->amount) {
                $intent->cancel();
                $helper->removePaymentIntentFromPayment($quote->getPayment());

                return new Varien_Object([
                    "result" => self::RECREATE_INTENT,
                    "client_secret" => null,
                    "error" => null,
                ]);
            }

            $intent->confirm();

            $helper->assignIntentToPayment($quote->getPayment(), $intent);

            return $this->getIntentResult($quote, $intent);
        }
        catch(ExceptionInterface $e) {
            Mage::logException($e);

            return new Varien_Object([
                "result" => self::ERROR_MESSAGE,
                "client_secret" => null,
                "error" => $e->getMessage(),
            ]);
        }
    }

    public function mutateSetPaymentMethod(Mage_Sales_Model_Quote $quote, array $args, $ctx, ResolveInfo $info) {
        $model = Mage::getSingleton("mageql_sales/quote");

        try {
            $model->importPaymentData($quote, "Crossroads_Stripe_PaymentIntents", []);
            $model->saveSessionQuote();

            return self::SUCCESS;
        }
        catch(Mage_Core_Exception $e) {
            return $this->translatePaymentError($e);
        }
    }

    public function getIntentResult(Mage_Sales_Model_Quote $quote, $intent) {
        if($intent->status === "requires_action"  &&
            $intent->next_action->type === "use_stripe_sdk") {
            return new Varien_Object([
                "result" => self::REQUIRES_ACTION,
                "client_secret" => $intent->client_secret,
                "error" => null,
            ]);
        }

        if($intent->status === "requires_capture") {
            // We are supposed to caputure this later, we have now authorized the intent
            $payment = $quote->getPayment();

            $payment->setIsTransactionApproved(true);
            $payment->save();

            return new Varien_Object([
                "result" => self::SUCCESS,
                "client_secret" => null,
                "error" => null,
            ]);
        }

        if($intent->status === "succeeded") {
            Mage::throwException(sprintf("Stripe payment captured before checkout on quote '%d'.", $quote->getId()));
        }

        Mage::throwException(sprintf("Invalid intent status '%s' for intent on quote '%d'.", $intent->status, $quote->getId()));
    }

    public function getUnreachableTypes(): array {
        return array_merge(parent::getUnreachableTypes(), [
            "StripePaymentIntentResultRequiresAction",
            "StripePaymentIntentResultError",
            "StripePaymentIntentResultSimple",
        ]);
    }
}
