<?php

class Crossroads_Stripe_Model_Export_Csv
{
    /**
     * @param Mage_Sales_Model_Order[] $orders
     * @return string
     * @throws Mage_Core_Exception
     */
    public function generate($orders)
    {
        $tmpName = tempnam(sys_get_temp_dir(), 'stripeexportcsv');
        $fp = fopen($tmpName, 'w');

        $headers = [
            'Ordernr',
            'Transaktionsnr',
            'Belopp i SEK ',
            'Belopp',
            'Valuta',
            'Avgift',
            'Korttyp',
            'Debiteringstidpunkt',
            'Tidpunkt för återbetalning',
            'Orderinformation',
        ];

        fputcsv($fp, $headers);

        foreach ($orders as $order) {
            fputcsv($fp, $this->orderToCsvRow($order));
        }

        fclose($fp);

        return $tmpName;
    }

    /**
     * @param Mage_Sales_Model_Order $order
     * @return array
     * @throws Mage_Core_Exception
     */
    private function orderToCsvRow($order)
    {
        $order->getPayment()->getAdditionalInformation('intent_id');
        $intent = Mage::helper('Crossroads_Stripe/intent')->getPaymentIntentFromPayment($order->getPayment());

        if (!$intent) {
            Mage::throwException('Unable to fetch payment intent on order '.$order->getIncrementId());
        }

        $currency = $order->getBaseCurrency()->getCode();

        return [
            $order->getIncrementId(),                    // Ordernr
            $intent->id,                                 // Transaktionsnr
            '',                                          // Belopp i SEK
            sprintf('%.2f', $order->getGrandTotal()),    // Belopp
            $this->iso4217NumericCode($currency),        // Valuta
            $this->fee($order, $intent),                 // Avgift
            $this->cardType($intent),                    // Korttyp
            $this->captureDate($order),                  // Debiteringstidpunkt
            '',                                          // Tidpunkt för återbetalning
            '',                                          // Orderinformation
        ];
    }

    /**
     * @param Mage_Sales_Model_Order $order
     * @return string
     */
    private function captureDate($order)
    {
        /** @var Mage_Sales_Model_Entity_Order_Invoice_Collection $invoiceCollection */
        $invoiceCollection = $order->getInvoiceCollection();

        /** @var Mage_Sales_Model_Order_Invoice $first */
        $invoice = $invoiceCollection->getFirstItem();
        if (!$invoice) {
            $this->logWarning('No invoice found on order '.$order->getIncrementId());

            return '';
        }

        $date = $invoice->getCreatedAt();

        return date('d/m-Y \k\l. H:i:s', strtotime($date));
    }

    /**
     * @param \Stripe\PaymentIntent $intent
     * @return string
     */
    private function cardType($intent)
    {
        if (!isset($intent->charges->data[0])) {
            $this->logWarning('No charge found on intent '.$intent->id);

            return '';
        }

        /** @var \Stripe\Charge $charge */
        $charge = $intent->charges->data[0];

        if ('card' !== $charge->payment_method_details->type) {
            return '';
        }

        $brand = $charge->payment_method_details->card->brand;

        switch ($brand) {
            case 'visa':
                return 'VISA';
            case 'mastercard':
                return 'MC';
            case 'amex':
                return 'AMEX';
            default:
                return strtoupper($brand);
        }
    }

    /**
     * 0.8% + 0.8kr / 0.10€ for EEA Visa/Mastercard
     * 2.9% + 1.8kr / 0.25€ for Amex and international Visa/Mastercard
     *
     * @param Mage_Sales_Model_Order $order
     * @param \Stripe\PaymentIntent $intent
     * @return string
     */
    private function fee($order, $intent)
    {
        if (!isset($intent->charges->data[0])) {
            $this->logWarning('No charge found on intent '.$intent->id);

            return '';
        }

        /** @var \Stripe\Charge $charge */
        $charge = $intent->charges->data[0];

        if ('card' !== $charge->payment_method_details->type) {
            return '';
        }

        $isEEA = in_array($charge->payment_method_details->card->country, ['SE', 'DK', 'FI']);
        $brand = $charge->payment_method_details->card->brand;

        if ($isEEA && in_array($brand, ['visa', 'mastercard'])) {
            $partial = $order->getGrandTotal() * 0.008;
            $fixed = 'eur' === $intent->currency ? 0.1 : 0.8;
        } else {
            $partial = $order->getGrandTotal() * 0.018;
            $fixed = 'eur' === $intent->currency ? 0.25 : 1.8;
        }

        return sprintf('%.2f', $partial + $fixed);
    }

    /**
     * @param string $message
     */
    private function logWarning($message)
    {
        Mage::log('Problem when creating stripe csv export: '.$message, Zend_Log::WARN);
    }

    /**
     * @param string $string
     * @return string
     */
    private function iso4217NumericCode($string)
    {
        $codes = [
            'AFN' => '971',
            'EUR' => '978',
            'ALL' => '008',
            'DZD' => '012',
            'USD' => '840',
            'AOA' => '973',
            'XCD' => '951',
            'ARS' => '032',
            'AMD' => '051',
            'AWG' => '533',
            'AUD' => '036',
            'AZN' => '944',
            'BSD' => '044',
            'BHD' => '048',
            'BDT' => '050',
            'BBD' => '052',
            'BYN' => '933',
            'BZD' => '084',
            'XOF' => '952',
            'BMD' => '060',
            'INR' => '356',
            'BTN' => '064',
            'BOB' => '068',
            'BOV' => '984',
            'BAM' => '977',
            'BWP' => '072',
            'NOK' => '578',
            'BRL' => '986',
            'BND' => '096',
            'BGN' => '975',
            'BIF' => '108',
            'CVE' => '132',
            'KHR' => '116',
            'XAF' => '950',
            'CAD' => '124',
            'KYD' => '136',
            'CLP' => '152',
            'CLF' => '990',
            'CNY' => '156',
            'COP' => '170',
            'COU' => '970',
            'KMF' => '174',
            'CDF' => '976',
            'NZD' => '554',
            'CRC' => '188',
            'HRK' => '191',
            'CUP' => '192',
            'CUC' => '931',
            'ANG' => '532',
            'CZK' => '203',
            'DKK' => '208',
            'DJF' => '262',
            'DOP' => '214',
            'EGP' => '818',
            'SVC' => '222',
            'ERN' => '232',
            'ETB' => '230',
            'FKP' => '238',
            'FJD' => '242',
            'XPF' => '953',
            'GMD' => '270',
            'GEL' => '981',
            'GHS' => '936',
            'GIP' => '292',
            'GTQ' => '320',
            'GBP' => '826',
            'GNF' => '324',
            'GYD' => '328',
            'HTG' => '332',
            'HNL' => '340',
            'HKD' => '344',
            'HUF' => '348',
            'ISK' => '352',
            'IDR' => '360',
            'XDR' => '960',
            'IRR' => '364',
            'IQD' => '368',
            'ILS' => '376',
            'JMD' => '388',
            'JPY' => '392',
            'JOD' => '400',
            'KZT' => '398',
            'KES' => '404',
            'KPW' => '408',
            'KRW' => '410',
            'KWD' => '414',
            'KGS' => '417',
            'LAK' => '418',
            'LBP' => '422',
            'LSL' => '426',
            'ZAR' => '710',
            'LRD' => '430',
            'LYD' => '434',
            'CHF' => '756',
            'MOP' => '446',
            'MKD' => '807',
            'MGA' => '969',
            'MWK' => '454',
            'MYR' => '458',
            'MVR' => '462',
            'MRU' => '929',
            'MUR' => '480',
            'XUA' => '965',
            'MXN' => '484',
            'MXV' => '979',
            'MDL' => '498',
            'MNT' => '496',
            'MAD' => '504',
            'MZN' => '943',
            'MMK' => '104',
            'NAD' => '516',
            'NPR' => '524',
            'NIO' => '558',
            'NGN' => '566',
            'OMR' => '512',
            'PKR' => '586',
            'PAB' => '590',
            'PGK' => '598',
            'PYG' => '600',
            'PEN' => '604',
            'PHP' => '608',
            'PLN' => '985',
            'QAR' => '634',
            'RON' => '946',
            'RUB' => '643',
            'RWF' => '646',
            'SHP' => '654',
            'WST' => '882',
            'STN' => '930',
            'SAR' => '682',
            'RSD' => '941',
            'SCR' => '690',
            'SLL' => '694',
            'SGD' => '702',
            'XSU' => '994',
            'SBD' => '090',
            'SOS' => '706',
            'SSP' => '728',
            'LKR' => '144',
            'SDG' => '938',
            'SRD' => '968',
            'SZL' => '748',
            'SEK' => '752',
            'CHE' => '947',
            'CHW' => '948',
            'SYP' => '760',
            'TWD' => '901',
            'TJS' => '972',
            'TZS' => '834',
            'THB' => '764',
            'TOP' => '776',
            'TTD' => '780',
            'TND' => '788',
            'TRY' => '949',
            'TMT' => '934',
            'UGX' => '800',
            'UAH' => '980',
            'AED' => '784',
            'USN' => '997',
            'UYU' => '858',
            'UYI' => '940',
            'UYW' => '927',
            'UZS' => '860',
            'VUV' => '548',
            'VES' => '928',
            'VND' => '704',
            'YER' => '886',
            'ZMW' => '967',
            'ZWL' => '932',
            'XBA' => '955',
            'XBB' => '956',
            'XBC' => '957',
            'XBD' => '958',
            'XTS' => '963',
            'XXX' => '999',
            'XAU' => '959',
            'XPD' => '964',
            'XPT' => '962',
            'XAG' => '961',
        ];

        return isset($codes[$string]) ? $codes[$string] : '';
    }
}
