<?php

class Crossroads_Stripe_Model_Method_PaymentIntents extends Mage_Payment_Model_Method_Abstract
{
    const CODE = "Crossroads_Stripe_PaymentIntents";

    protected $_code = self::CODE;

    protected $_isGateway = true;

    protected $_isOnline = true;

    protected $_canUseForMultishipping = false;

    protected $_canUseInternal = true;

    protected $_canAuthorize = true;

    protected $_canRefund = false;

    protected $_canCapture = true;

    protected $_canVoid = true;

    protected $_canCancel = true;

    protected $_canFetchTransactionInfo = true;

    /**
     * Check whether payment method can be used
     *
     * @param  Mage_Sales_Model_Quote|null $quote
     * @return bool
     */
    public function isAvailable($quote = null)
    {
        $config = $this->config();
        $store = $quote ? $quote->getStore() : Mage::app()->getStore();

        $minimumOrderTotal = $config->minOrderTotal($store);
        $publishableKey = $config->publishableKey($store);

        return (!$quote || $quote->getBaseGrandTotal() >= $minimumOrderTotal) && $publishableKey && parent::isAvailable($quote);
    }

    /**
     * Called through `Mage_Sales_Model_Quote_Payment::importData` to set data for the payment method.
     *
     * @param  Varien_Object $data
     * @return Mage_Payment_Model_Info
     */
    public function assignData($data)
    {
        return parent::assignData($data);
    }

    /**
     * Authorization function for the payment, executed before the order is approved.
     *
     * @param  Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment|Varien_Object $payment
     * @param  float $amount
     * @return $this
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function authorize(Varien_Object $payment, $amount)
    {
        // https://stripe.com/docs/payments/payment-intents/status

        $intent = $this->intentHelper()->getPaymentIntentFromPayment($payment);

        if (!$intent) {
            throw Crossroads_API_ResponseException::create(500, 'Intent no longer available');
        }

        if ('canceled' === $intent->status) {
            throw Crossroads_API_ResponseException::create(403, 'Payment has been cancelled');
        }

        if ('requires_confirmation' === $intent->status) {
            // Payment should have already been reserved in custom controller action.
            throw Crossroads_API_ResponseException::create(403, 'Payment has not been confirmed.');
        }

        if (!$this->intentHelper()->hasSameTotal($payment, $intent)) {
            // Magento payment and Stripe payment totals do no match.
            $intent->cancel();
            $this->intentHelper()->removePaymentIntentFromPayment($payment);

            throw Crossroads_API_ResponseException::create(400, 'Quote has changed since payment intent was created. Create a new intent.');
        }

        if ('succeeded' === $intent->status) {
            // Should not have happened when capture_method is manual
            throw Crossroads_API_ResponseException::create(500, 'Payment captured too early');
        }

        $payment->setTransactionId($intent->id);
        // Leave the transaction opened so it can be captured later
        $payment->setIsTransactionClosed(false);
        $payment->setIsTransactionApproved(true);
        $payment->save();

        return $this;
    }

    /**
     * Captures amount from Stripe from a previous reservation made using authorize.
     *
     * @param  Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment|Varien_Object $payment
     * @param  float $amount
     * @return $this
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function capture(Varien_Object $payment, $amount)
    {
        $intent = $this->intentHelper()->getPaymentIntentFromPayment($payment);

        if (!$intent) {
            throw new Mage_Core_Exception('Payment intent is missing');
        }

        if (!$this->intentHelper()->hasSameTotal($payment, $intent)) {
            throw new Mage_Core_Exception('Amount on payment and intent does not match');
        }

        if ('canceled' === $intent->status) {
            $payment->setIsTransactionDenied(true);
            $payment->setIsTransactionClosed(true);
            $payment->save();

            throw new Mage_Core_Exception('This PaymentIntent could not be captured because it has a status of canceled.');
        }

        try {
            $intent->capture();
        } catch (Exception $exception) {
            throw new Mage_Core_Exception($exception->getMessage(), 0, $exception);
        }

        $payment->setTransactionId($intent->id);
        $payment->setIsTransactionPending(false);
        $payment->setIsTransactionClosed(true);

        return $this;
    }

    /**
     * Cancels a reservation with Stripe.
     *
     * @param  Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     * @throws Mage_Core_Exception
     */
    public function cancel(Varien_Object $payment)
    {
        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        $intent = $this->intentHelper()->getPaymentIntentFromPayment($payment);

        if (!$intent) {
            return $this;
        }

        try {
            $intent->cancel();
        } catch (Exception $exception) {
            throw new Mage_Core_Exception($exception->getMessage(), 0, $exception);
        }

        return $this;
    }

    /**
     * @param  Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     * @throws Mage_Core_Exception
     */
    public function void(Varien_Object $payment)
    {
        return $this->cancel($payment);
    }

    /**
     * Updates transaction information from Stripe.
     *
     * @param  Mage_Payment_Model_Info $payment
     * @param  int $transactionId
     * @return array
     */
    public function fetchTransactionInfo(Mage_Payment_Model_Info $payment, $transactionId)
    {
        return parent::fetchTransactionInfo($payment, $transactionId);
    }

    /**
     * @return Crossroads_Stripe_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_Stripe/config');
    }

    /**
     * @return Crossroads_Stripe_Helper_Intent
     */
    protected function intentHelper()
    {
        return Mage::helper('Crossroads_Stripe/intent');
    }
}
