<?php

use Stripe\PaymentIntent;
use Stripe\Stripe;

class Crossroads_Stripe_Helper_Intent extends Mage_Core_Helper_Abstract
{
    /**
     * @param Mage_Payment_Model_Info $payment
     * @param PaymentIntent $intent
     * @throws Exception
     */
    public function assignIntentToPayment(Mage_Payment_Model_Info $payment, PaymentIntent $intent)
    {
        $payment->setAdditionalInformation('intent_id', $intent->id);
        $payment->setAdditionalInformation('intent_amount', $intent->amount);
        $payment->setAdditionalInformation('intent', json_encode($intent));

        $payment->save();
    }

    /**
     * @param  Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment $payment
     * @return PaymentIntent|null
     */
    public function getPaymentIntentFromPayment(Mage_Payment_Model_Info $payment)
    {
        $store = null;
        $intentId = $payment->getAdditionalInformation('intent_id');

        if (!$intentId) {
            return null;
        }

        if($payment instanceof Mage_Sales_Model_Quote_Payment) {
            $store = $payment->getQuote()->getStore();
        }
        elseif($payment instanceof Mage_Sales_Model_Order_Payment) {
            $store = $payment->getOrder()->getStore();
        }

        Stripe::setApiKey($this->config()->secretKey($store));

        return PaymentIntent::retrieve($intentId);
    }

    /**
     * @param PaymentIntent $intent
     * @return bool
     */
    public function hasSameTotal(Mage_Sales_Model_Order_Payment $payment, $intent)
    {
        return (int)($payment->getAmountOrdered() * 100) === (int)$intent->amount;
    }

    /**
     * @return Mage_Sales_Model_Quote
     */
    public function getQuote()
    {
        return Mage::getSingleton('checkout/session')->getQuote();
    }

    /**
     * @return Crossroads_Stripe_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_Stripe/config');
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @param string $paymentMethodId
     * @return PaymentIntent
     */
    public function create($quote, $paymentMethodId)
    {
        Stripe::setApiKey($this->config()->secretKey($quote->getStore()));

        return PaymentIntent::create([
            'payment_method'      => $paymentMethodId,
            'amount'              => (int) ($quote->getGrandTotal() * 100),
            'currency'            => $quote->getQuoteCurrencyCode(),
            'confirmation_method' => 'manual',

            // Payment is captured when order is created or when invoice is created, depending on the configuration
            'capture_method'      => 'manual',

            // We need to confirm now to to see if additional action is required.
            'confirm'             => true,
        ]);
    }

    /**
     * @param string $paymentIntentId
     * @return PaymentIntent
     */
    public function retrieve(Mage_Core_Model_Store $store, $paymentIntentId)
    {
        Stripe::setApiKey($this->config()->secretKey($store));

        return PaymentIntent::retrieve($paymentIntentId);
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @throws Mage_Core_Exception
     */
    public function removePaymentIntentFromPayment(Mage_Payment_Model_Info $payment)
    {
        $payment->setAdditionalInformation('intent_id', null);
        $payment->setAdditionalInformation('intent_amount', null);
        $payment->setAdditionalInformation('intent', null);

        $payment->save();
    }
}
