<?php

use Stripe\PaymentIntent;

class Crossroads_Stripe_Model_Observer_CheckoutObserver
{
    const CANCELABLE_PAYMENT_INTENT_STATUSES = [
        PaymentIntent::STATUS_REQUIRES_PAYMENT_METHOD,
        PaymentIntent::STATUS_REQUIRES_CAPTURE,
        PaymentIntent::STATUS_REQUIRES_CONFIRMATION,
        PaymentIntent::STATUS_REQUIRES_ACTION,
    ];

    /**
     * @param Varien_Event_Observer $observer
     * @throws Exception
     */
    public function preMerge(Varien_Event_Observer $observer)
    {
        /** @var Mage_Sales_Model_Quote $quote */
        $quote = $observer->getEvent()->getData('quote');

        $method = Mage::getModel('Crossroads_Stripe/payment_paymentIntents')->getCode();

        if ($method !== $quote->getPayment()->getMethod()) {
            return;
        }

        $intent = $this->intentHelper()->getPaymentIntentFromPayment($quote->getPayment());

        if (!$intent) {
            return;
        }

        if((int)($quote->getGrandTotal() * 100) === (int)$intent->amount) {
            return;
        }

        try {
            $intent->cancel();
            $this->intentHelper()->removePaymentIntentFromPayment($quote->getPayment());
        } catch (\Stripe\Error\Base $e) {
            throw Crossroads_API_ResponseException::create(500, $e->getMessage());
        }
    }

    public function quoteSaveBefore($event) {
        $quote = $event->getQuote();
        $payment = $quote->getPayment();
        $helper = $this->intentHelper();

        if($payment->getMethod() !== "Crossroads_Stripe_PaymentIntents") {
            return;
        }

        $intentAmount = $payment->getAdditionalInformation("intent_amount");

        if( ! $intentAmount) {
            return;
        }

        if((int)($quote->getGrandTotal() * 100) !== (int)$intentAmount) {
            Mage::log(sprintf(
                "%s: Mismatched amount ordered (%d) and intent amount (%d) on quote %d",
                __METHOD__,
                (int)($quote->getGrandTotal() * 100),
                (int)$intentAmount,
                $quote->getId()
            ));

            try {
                $intent = $helper->getPaymentIntentFromPayment($quote->getPayment());

                if($intent && in_array($intent, self::CANCELABLE_PAYMENT_INTENT_STATUSES)) {
                    $intent->cancel();
                }
            }
            finally {
                $helper->removePaymentIntentFromPayment($payment);
            }
        }
    }

    /**
     * @return Crossroads_Stripe_Helper_Intent
     */
    protected function intentHelper()
    {
        return Mage::helper('Crossroads_Stripe/intent');
    }
}
