<?php

class Crossroads_Stripe_Controller_Payment extends Crossroads_API_Controller_Super
{
    /**
     * /api/payment/Crossroads_Stripe_PaymentIntents/confirmPayment
     *
     * @see https://stripe.com/docs/payments/payment-intents/quickstart#creating-with-manual-confirmation
     * @throws Exception
     */
    public function confirmPaymentAction()
    {
        $quote = $this->getQuote();

        if (!$quote || !$quote->getGrandTotal()) {
            throw Crossroads_API_ResponseException::create(400, 'Quote not set');
        }

        $stripeResponse = json_decode($this->getRequest()->getRawBody());

        $intent = null;
        try {
            if (isset($stripeResponse->paymentMethod) && isset($stripeResponse->paymentMethod->id)) {
                $intent = $this->intentHelper()->create($quote, $stripeResponse->paymentMethod->id);
            }

            if (isset($stripeResponse->paymentIntent) && isset($stripeResponse->paymentIntent->id)) {
                $intent = $this->intentHelper()->retrieve($quote->getStore(), $stripeResponse->paymentIntent->id);

                $intent->confirm();
            }
        } catch (\Stripe\Error\Base $e) {
            throw Crossroads_API_ResponseException::create(500, 'Stripe error: '.$e->getMessage());
        }

        if (!$intent) {
            throw Crossroads_API_ResponseException::create(400, 'Missing input paymentMethod.id or paymentIntent.id');
        }

        $this->intentHelper()->assignIntentToPayment($quote->getPayment(), $intent);

        if ('requires_action' === $intent->status &&
            'use_stripe_sdk' === $intent->next_action->type) {
            // Tell the client to handle the action

            return $this->sendData([
                200,
                [
                    'requires_action'              => true,
                    'payment_intent_client_secret' => $intent->client_secret,
                ],
            ]);
        } elseif ('requires_capture' === $intent->status) {
            $payment = $quote->getPayment();
            $payment->setIsTransactionApproved(true);
            $payment->save();

            return $this->sendData([
                200,
                [
                    'success' => true,
                    'status'  => $intent->status,
                ],
            ]);
        } elseif ('succeeded' === $intent->status) {
            throw Crossroads_API_ResponseException::create(500, 'Payment captured before checkout');
        } else {
            throw Crossroads_API_ResponseException::create(500, 'Invalid PaymentIntent status: '.$intent->status);
        }
    }

    /**
     * @return Crossroads_Stripe_Helper_Intent
     */
    protected function intentHelper()
    {
        return Mage::helper('Crossroads_Stripe/intent');
    }

    /**
     * @return Mage_Sales_Model_Quote
     */
    public function getQuote()
    {
        return Mage::getSingleton('checkout/session')->getQuote();
    }
}
