<?php

class Crossroads_Stripe_Model_Export_Orders
{
    const INTEGRATION_TABLE                    = 'crossroads_integration_orders';
    const INTEGRATION_STATUS_IMPORTED_TO_VISMA = 5;
    const STRIPE_STATUS_EXPORT_PROCESSING      = 1;
    const STRIPE_STATUS_EXPORT_FAILED          = 2;
    const STRIPE_STATUS_EXPORT_EXPORTED        = 3;

    /**
     * @return bool
     */
    public function integrationTableExists()
    {
        $resource = Mage::getSingleton('core/resource');

        $readConnection = $resource->getConnection('core_read');

        $results = $readConnection->fetchAll('SHOW TABLES LIKE "'.static::INTEGRATION_TABLE.'"');

        return 0 !== count($results);
    }

    /**
     * @return Mage_Sales_Model_Order[]|Mage_Sales_Model_Resource_Order_Collection
     */
    public function getOrdersForExport()
    {
        $paymentsTable = Mage::getSingleton('core/resource')->getTableName('sales/order_payment');
        $statusTable = Mage::getSingleton('core/resource')->getTableName('Crossroads_Stripe/order_export_status');
        $integrationTable = static::INTEGRATION_TABLE;

        /** @var Mage_Sales_Model_Resource_Order_Collection|Mage_Sales_Model_Order[] $collection */
        $collection = Mage::getModel('sales/order')->getCollection();

        $collection->getSelect()
            ->join($paymentsTable, "main_table.entity_id = $paymentsTable.parent_id", [])
            ->join($integrationTable, "main_table.entity_id = $integrationTable.magento_order_id", [])
            ->joinLeft($statusTable, "main_table.entity_id = $statusTable.magento_order_id", []);

        // Only stripe orders
        $method = Mage::getModel('Crossroads_Stripe/payment_paymentIntents')->getCode();
        $collection->addFieldToFilter('method', $method);

        // Only orders placed yesterday or earlier
        $collection->addFieldToFilter('main_table.created_at', [
            'to'   => date('Y-m-d H:i:s', strtotime('today midnight')),
            'date' => true,
        ]);

        // Only orders when the correct integration status
        $collection->addFieldToFilter("$integrationTable.export_status", static::INTEGRATION_STATUS_IMPORTED_TO_VISMA);

        // Only orders not yet processed
        $collection->addFieldToFilter("$statusTable.status", ['null' => true]);

        return $collection;
    }

    /**
     * @param Mage_Sales_Model_Order[] $orders
     */
    public function setStatusesToProcessing($orders)
    {
        foreach ($orders as $order) {
            $this->setStatus($order, static::STRIPE_STATUS_EXPORT_PROCESSING);
        }
    }

    /**
     * @param Mage_Sales_Model_Order[] $orders
     */
    public function setStatusesToFailed($orders)
    {
        foreach ($orders as $order) {
            $this->setStatus($order, static::STRIPE_STATUS_EXPORT_FAILED);
        }
    }

    /**
     * @param Mage_Sales_Model_Order[] $orders
     */
    public function setStatusesToExported($orders)
    {
        foreach ($orders as $order) {
            $this->setStatus($order, static::STRIPE_STATUS_EXPORT_EXPORTED);
        }
    }

    /**
     * @param Mage_Sales_Model_Order $order
     * @param int $status
     */
    private function setStatus($order, $status)
    {
        $statusModel = Mage::getModel('Crossroads_Stripe/orderExportStatus')
            ->load($order->getId(), 'magento_order_id');

        // in case it does not exists:
        $statusModel->setData('magento_order_id', $order->getId());

        $statusModel->setData('status', $status);

        $statusModel->save();
    }
}
