<?php

use Stripe\PaymentIntent;

class Crossroads_Stripe_Helper_Intent extends Mage_Core_Helper_Abstract
{
    /**
     * @param Mage_Payment_Model_Info $payment
     * @param PaymentIntent $intent
     * @throws Exception
     */
    public function assignIntentToPayment(Mage_Payment_Model_Info $payment, PaymentIntent $intent)
    {
        $payment->setAdditionalInformation('intent_id', $intent->id);
        $payment->setAdditionalInformation('intent', json_encode($intent));

        $payment->save();
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @return PaymentIntent|null
     */
    public function getPaymentIntentFromPayment(Mage_Payment_Model_Info $payment)
    {
        $intentId = $payment->getAdditionalInformation('intent_id');

        if (!$intentId) {
            return null;
        }

        \Stripe\Stripe::setApiKey($this->config()->secretKey());

        return \Stripe\PaymentIntent::retrieve($intentId);
    }

    /**
     * @param Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment $payment
     * @param PaymentIntent $intent
     * @return bool
     */
    public function hasSameTotal($payment, $intent)
    {
        return ((float) $payment->getAmountOrdered()) * 100 === (float) $intent->amount;
    }

    /**
     * @return Mage_Sales_Model_Quote
     */
    public function getQuote()
    {
        return Mage::getSingleton('checkout/session')->getQuote();
    }

    /**
     * @return Crossroads_Stripe_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_Stripe/config');
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @param string $paymentMethodId
     * @return PaymentIntent
     */
    public function create($quote, $paymentMethodId)
    {
        \Stripe\Stripe::setApiKey($this->config()->secretKey());

        return \Stripe\PaymentIntent::create([
            'payment_method'      => $paymentMethodId,
            'amount'              => (int) ($quote->getGrandTotal() * 100),
            'currency'            => $quote->getQuoteCurrencyCode(),
            'confirmation_method' => 'manual',

            // Payment is captured when order is created or when invoice is created, depending on the configuration
            'capture_method'      => 'manual',

            // We need to confirm now to to see if additional action is required.
            'confirm'             => true,
        ]);
    }

    /**
     * @param string $paymentIntentId
     * @return PaymentIntent
     */
    public function retrieve($paymentIntentId)
    {
        \Stripe\Stripe::setApiKey($this->config()->secretKey());

        return \Stripe\PaymentIntent::retrieve($paymentIntentId);
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @throws Mage_Core_Exception
     */
    public function removePaymentIntentFromPayment(Mage_Payment_Model_Info $payment)
    {
        $payment->setAdditionalInformation('intent_id', null);
        $payment->setAdditionalInformation('intent', null);

        $payment->save();
    }
}
