<?php

class Crossroads_Serialcodes_Helper_Data extends Mage_Core_Helper_Abstract
{
    const TIMEOUT = 5;
    const SERIALCODES_IMAGE_NONE = 0;
    const SERIALCODES_IMAGE_BARCODE = 1;
    const SERIALCODES_IMAGE_QRCODE = 2;
    const SERIALCODES_IMAGE_AZTECCODE = 3;
    const SERIALCODES_IMAGE_BARCODE128C = 4;

    // Email setup
    const USE_EMAIL_QUEUE = true; // Magento >= 1.9 uses this
    const EMAIL_EVENT_TYPE = "serialcodes";

    const SERIALCODES_ACL_LIST             = "admin/catalog/serialcodes";
    const SERIALCODES_ACL_EDIT             = "admin/catalog/serialcodes/edit";
    const SERIALCODES_ACL_ORDERS           = "admin/sales/serialcodes_orders";
    const SERIALCODES_ACL_ORDER_ITEM_VIEW  = "admin/sales/order/actions/serialcodes_view";
    const SERIALCODES_ACL_ORDER_ITEM_EDIT  = "admin/sales/order/actions/serialcodes_edit";
    const SERIALCODES_ACL_ORDER_ITEM_ISSUE = "admin/sales/order/actions/serialcodes_issue";
    const SERIALCODES_ACL_ORDER_ITEM_EMAIL = "admin/sales/order/actions/serialcodes_email";

    const SERIALCODES_IMAGE_URL = "_auc" . DS . "serialcodes" . DS . "get_image" . DS . "hash" . DS;

    public static $allowedOrderStatuses = array('processing', 'complete');
    public static $imageTypes = array(self::SERIALCODES_IMAGE_BARCODE, self::SERIALCODES_IMAGE_BARCODE128C, self::SERIALCODES_IMAGE_QRCODE, self::SERIALCODES_IMAGE_AZTECCODE);

    protected $_logFile = null;

    public function __construct()
    {
        $this->_logFile = 'serialcodes.log';
    }

    public function getImgHTML($values, $code, $storeId = 0)
    {
        $imgHTML = "";
        $serialCodeImageType = $values["SerialCodeImageType"];
        $sku = $values["sku"];
        $imageParams = $values["SerialCodeImageParams"];

        // Only try to create images if code type is set to something else than 'none'.
        if (in_array($serialCodeImageType, self::$imageTypes)) {
            $sqlQuery = "SELECT code, image_hash FROM serialcodes WHERE sku = :sku AND code = :code";
            $params = [
                "sku" => $sku,
                "code" => $code
            ];
            $codeData = Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchRow($sqlQuery, $params);
            $baseUrl = Mage::app()->getStore($storeId)->getBaseUrl();

            if (empty($codeData["image_hash"])) {
                switch ($serialCodeImageType) {
                    case self::SERIALCODES_IMAGE_BARCODE:
                        $codeData["image_hash"] = $this->createBARcode($sku, $code, $imageParams);
                        break;

                    case self::SERIALCODES_IMAGE_BARCODE128C:
                        $codeData["image_hash"] = $this->createBARcode128c($sku, $code, $imageParams);
                        break;

                    case self::SERIALCODES_IMAGE_QRCODE:
                        $codeData["image_hash"] = $this->createQRcode($sku, $code, $imageParams);
                        break;

                    case self::SERIALCODES_IMAGE_AZTECCODE:
                        $codeData["image_hash"] = $this->createAZTECcode($sku, $code, $imageParams);
                        break;

                    default:
                        break;

                }
            }

            $hash = !empty($codeData["image_hash"]) ? $codeData["image_hash"] : "missing";
            $controllerUrl = self::SERIALCODES_IMAGE_URL;
            $imgHTML .= "<table width=\"100%\" class=\"callout\" style=\"Margin-bottom:14px;border-collapse:collapse;border-spacing:0;margin-bottom:14px;padding:0;text-align:left;vertical-align:top\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"callout-inner\" style=\"Margin:0;background:#fff;border:none;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:14px;text-align:left;width:100%\"><table class=\"row collapse\" style=\"border-collapse:collapse;border-spacing:0;display:table;padding:0;position:relative;text-align:left;vertical-align:top;width:100%\"><tbody><tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"small-12 large-4 columns first\" style=\"Margin:0 auto;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0 auto;padding:0;padding-bottom:0;padding-left:0;padding-right:0;text-align:left;width:190.33px\"><table style=\"border-collapse:collapse;border-spacing:0;padding:0;text-align:left;vertical-align:top;width:100%\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th style=\"Margin:0;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:0;text-align:left\"><center><img align=\"center\" class=\"float-center code__image\" style=\"-ms-interpolation-mode:bicubic;Margin:0 auto;Margin-top:32px;clear:both;display:block;float:none;height:auto!important;margin:0 auto;max-width:100%;outline:0;text-align:center;text-decoration:none;width:auto;margin-top:32px;\" data-f=\"{$baseUrl}\" src=\"{$baseUrl}{$controllerUrl}{$hash}.png\" /></center></th></tr></table></th></tr>\n";
        }

        return $imgHTML;
    }

    public function createBARcode($sku, $code, $imageParams = array())
    {
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGFontFile.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGColor.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGDrawing.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGcode39.barcode.php');

        try {
            $renderCode = $code;
            if (!empty($imageParams['code_prefix'])) {
                $renderCode = $imageParams['code_prefix'] . $renderCode;
            }
            if (!empty($imageParams['code_suffix'])) {
                $renderCode = $renderCode . $imageParams['code_suffix'];
            }

            $this->log(LOG_DEBUG, "About to create BAR image for [{$sku}] with data '{$renderCode}'");

            // Loading Font
            $font = new BCGFontFile(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/font/Arial.ttf', 18);

            // The arguments are R, G, B for color.
            $color_black = new BCGColor(0, 0, 0);
            $color_white = new BCGColor(255, 255, 255);

            $drawException = null;
            try {
                $codeImage = new BCGcode39();
                $codeImage->setScale(2); // Resolution
                $codeImage->setThickness(30); // Thickness
                $codeImage->setForegroundColor($color_black); // Color of bars
                $codeImage->setBackgroundColor($color_white); // Color of spaces
                $codeImage->setFont($font); // Font (or 0)
                $codeImage->parse($renderCode); // Text
            } catch (Exception $exception) {
                $drawException = $exception;
            }

            $insertData = array(
                'hash' => md5("SomeStartSalt{$sku}{$code}SomeEndSalt"),
                'imagedata' => null
            );

            $path = Mage::getBaseDir("media") . "/barcodes";
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = "{$path}/{$insertData["hash"]}.png";

            $drawing = new BCGDrawing($fileName, $color_white);

            if ($drawException) {
                $drawing->drawException($drawException);
            } else {
                $drawing->setBarcode($codeImage);
                $drawing->draw();
            }

            $drawing->finish(BCGDrawing::IMG_FORMAT_PNG);
            $insertData['imagedata'] = file_get_contents($fileName);
            unlink($fileName);

            $sqlQuery1 = "INSERT INTO serialcode_images (hash, imagedata) VALUES (:hash, :imagedata) ON DUPLICATE KEY UPDATE imagedata = VALUES(imagedata)";
            $stm1 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery1);
            $stm1->execute($insertData);
            $affectedRows1 = $stm1->rowCount();

            if ($affectedRows1 > 1) {
                // Updated
                $this->log(LOG_DEBUG, "Already have BAR image for hash {$insertData['hash']}");
                return $insertData['hash'];
            } elseif ($affectedRows1 > 0) {
                // Saved, also save hash in serialcodes table
                $this->log(LOG_DEBUG, "Successfully created BAR image with hash {$insertData['hash']}");
                $updateData = array(
                    'hash' => $insertData['hash'],
                    'sku' => $sku,
                    'code' => $code
                );
                $sqlQuery2 = "UPDATE serialcodes SET image_hash = :hash WHERE sku = :sku AND code = :code";
                $stm2 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery2);
                $stm2->execute($updateData);
                $affectedRows2 = $stm2->rowCount();
                if ($affectedRows2 < 1) {
                    $this->log(LOG_ERR, 'Unable to update image_hash in serialcodes table.');
                }
                return $insertData['hash'];
            } else {
                // Not saved
                $this->log(LOG_ERR, "Unable to insert/update hash/imagedata.");
            }
        } catch (Exception $exception) {
            // Just silently ignore any error...
            $this->logException($exception, 'Exception while creating BAR code');
        }

        return null;
    }

    public function createBARcode128c($sku, $code, $imageParams = array())
    {
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGFontFile.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGColor.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGDrawing.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGcode128.barcode.php');

        try {
            $renderCode = $code;
            if (!empty($imageParams['code_prefix'])) {
                $renderCode = $imageParams['code_prefix'] . $renderCode;
            }
            if (!empty($imageParams['code_suffix'])) {
                $renderCode = $renderCode . $imageParams['code_suffix'];
            }

            $this->log(LOG_DEBUG, "About to create BAR image for [{$sku}] with data '{$renderCode}'");

            // Loading Font
            $font = new BCGFontFile(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/font/Arial.ttf', 18);

            // The arguments are R, G, B for color.
            $color_black = new BCGColor(0, 0, 0);
            $color_white = new BCGColor(255, 255, 255);

            $drawException = null;
            try {
                $codeImage = new BCGcode128("C");
                $codeImage->setScale(2); // Resolution
                $codeImage->setThickness(30); // Thickness
                $codeImage->setForegroundColor($color_black); // Color of bars
                $codeImage->setBackgroundColor($color_white); // Color of spaces
                $codeImage->setFont($font); // Font (or 0)
                $codeImage->parse($renderCode); // Text
            } catch (Exception $exception) {
                $drawException = $exception;
            }

            $insertData = array(
                'hash' => md5("SomeStartSalt{$sku}{$code}SomeEndSalt"),
                'imagedata' => null
            );

            $path = Mage::getBaseDir("media") . "/barcodes";
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = "{$path}/{$insertData["hash"]}.png";

            $drawing = new BCGDrawing($fileName, $color_white);

            if ($drawException) {
                $drawing->drawException($drawException);
            } else {
                $drawing->setBarcode($codeImage);
                $drawing->draw();
            }

            $drawing->finish(BCGDrawing::IMG_FORMAT_PNG);
            $insertData['imagedata'] = file_get_contents($fileName);
            unlink($fileName);

            $sqlQuery1 = "INSERT INTO serialcode_images (hash, imagedata) VALUES (:hash, :imagedata) ON DUPLICATE KEY UPDATE imagedata = VALUES(imagedata)";
            $stm1 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery1);
            $stm1->execute($insertData);
            $affectedRows1 = $stm1->rowCount();

            if ($affectedRows1 > 1) {
                // Updated
                $this->log(LOG_DEBUG, "Already have BAR image for hash {$insertData['hash']}");
                return $insertData['hash'];
            } elseif ($affectedRows1 > 0) {
                // Saved, also save hash in serialcodes table
                $this->log(LOG_DEBUG, "Successfully created BAR image with hash {$insertData['hash']}");
                $updateData = array(
                    'hash' => $insertData['hash'],
                    'sku' => $sku,
                    'code' => $code
                );
                $sqlQuery2 = "UPDATE serialcodes SET image_hash = :hash WHERE sku = :sku AND code = :code";
                $stm2 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery2);
                $stm2->execute($updateData);
                $affectedRows2 = $stm2->rowCount();
                if ($affectedRows2 < 1) {
                    $this->log(LOG_ERR, 'Unable to update image_hash in serialcodes table.');
                }
                return $insertData['hash'];
            } else {
                // Not saved
                $this->log(LOG_ERR, "Unable to insert/update hash/imagedata.");
            }
        } catch (Exception $exception) {
            // Just silently ignore any error...
            $this->logException($exception, 'Exception while creating BAR code');
        }

        return null;
    }

    public function createQRcode($sku, $code, $imageParams = array())
    {
        require_once(Mage::getBaseDir('lib') . '/phpqrcode/qrlib.php');

        try {
            $insertData = array(
                'hash' => md5("SomeStartSalt{$sku}{$code}SomeEndSalt"),
                'imagedata' => null
            );

            $renderCode = $code;
            if (!empty($imageParams['code_prefix'])) {
                $renderCode = $imageParams['code_prefix'] . $renderCode;
            }
            if (!empty($imageParams['code_suffix'])) {
                $renderCode = $renderCode . $imageParams['code_suffix'];
            }

            $this->log(LOG_DEBUG, "About to create QR image for [{$sku}] with data '{$renderCode}'");

            $path = Mage::getBaseDir("media") . "/barcodes";
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = "{$path}/{$insertData["hash"]}.png";

            QRcode::png($renderCode, $fileName, QR_ECLEVEL_L, 8);
            $insertData['imagedata'] = file_get_contents($fileName);
            unlink($fileName);

            $sqlQuery1 = "INSERT INTO serialcode_images (hash, imagedata) VALUES (:hash, :imagedata) ON DUPLICATE KEY UPDATE imagedata = VALUES(imagedata), created_at = now()";
            $stm1 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery1);
            $stm1->execute($insertData);
            $affectedRows1 = $stm1->rowCount();

            if ($affectedRows1 > 1) {
                // Updated
                $this->log(LOG_DEBUG, "Already have QR image for hash {$insertData['hash']}");
                return $insertData['hash'];
            } elseif ($affectedRows1 > 0) {
                // Saved, also save hash in serialcodes table
                $this->log(LOG_DEBUG, "Successfully created QR image with hash {$insertData['hash']}");
                $updateData = array(
                    'hash' => $insertData['hash'],
                    'sku' => $sku,
                    'code' => $code
                );
                $sqlQuery2 = "UPDATE serialcodes SET image_hash = :hash WHERE sku = :sku AND code = :code";
                $stm2 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery2);
                $stm2->execute($updateData);
                $affectedRows2 = $stm2->rowCount();
                if ($affectedRows2 < 1) {
                    $this->log(LOG_ERR, 'Unable to update image_hash in serialcodes table.');
                }
                return $insertData['hash'];
            } else {
                // Not saved
                $this->log(LOG_ERR, "Unable to insert/update hash/imagedata.");
            }
        } catch (Exception $exception) {
            // Just silently ignore any error...
            $this->logException($exception, 'Exception while creating QR code');
        }

        return null;
    }

    public function createAZTECcode($sku, $code, $imageParams = array())
    {
        require_once Mage::getBaseDir('lib') . DS . 'Metzli' . DS . 'Metzli.php';
        $renderFactor = 8; // 8 = 152px x 152px when aztec code is compact

        try {
            $insertData = array(
                'hash' => md5("SomeStartSalt{$sku}{$code}SomeEndSalt"),
                'imagedata' => null
            );

            $renderCode = $code;
            if (!empty($imageParams['code_prefix'])) {
                $renderCode = $imageParams['code_prefix'] . $renderCode;
            }
            if (!empty($imageParams['code_suffix'])) {
                $renderCode = $renderCode . $imageParams['code_suffix'];
            }
            $this->log(LOG_DEBUG, "About to create AZTEC image for [{$sku}] with data '{$renderCode}'");
            $aztecCode = MetzliEncoder::encode($renderCode);
            $renderer = new MetzliPngRenderer($renderFactor);
            $insertData['imagedata'] = $renderer->render($aztecCode);

            $sqlQuery1 = "INSERT INTO serialcode_images (hash, imagedata) VALUES (:hash, :imagedata) ON DUPLICATE KEY UPDATE imagedata = VALUES(imagedata), created_at = now()";
            $stm1 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery1);
            $stm1->execute($insertData);
            $affectedRows1 = $stm1->rowCount();

            if ($affectedRows1 > 1) {
                // Updated
                $this->log(LOG_DEBUG, "Already have AZTEC image for hash {$insertData['hash']}");
                return $insertData['hash'];
            } elseif ($affectedRows1 > 0) {
                // Saved, also save hash in serialcodes table
                $this->log(LOG_DEBUG, "Successfully created AZTEC image with hash {$insertData['hash']}");
                $updateData = array(
                    'hash' => $insertData['hash'],
                    'sku' => $sku,
                    'code' => $code
                );
                $sqlQuery2 = "UPDATE serialcodes SET image_hash = :hash WHERE sku = :sku AND code = :code";
                $stm2 = Mage::getSingleton('core/resource')->getConnection('integration_write')->prepare($sqlQuery2);
                $stm2->execute($updateData);
                $affectedRows2 = $stm2->rowCount();
                if ($affectedRows2 < 1) {
                    $this->log(LOG_ERR, 'Unable to update image_hash in serialcodes table.');
                }
                return $insertData['hash'];
            } else {
                // Not saved
                $this->log(LOG_ERR, "Unable to insert/update hash/imagedata. hash: {{$insertData['hash']}}");
            }
        } catch (Exception $exception) {
            // Just silently ignore any error...
            $this->logException($exception, 'Exception while creating AZTEC code');
        }

        return null;
    }

    public function getImagesForEmail($itemId, $storeId = 0)
    {
        $retval = array('top' => null, 'bottom' => null);
        $db = Mage::getSingleton('core/resource')->getConnection('core_read');
        $images = $db->fetchAll("SELECT T1.value, T2.store_id, T2.label FROM catalog_product_entity_media_gallery T1 JOIN catalog_product_entity_media_gallery_value T2 ON T2.value_id = T1.value_id WHERE T1.entity_id = ? AND T2.label != ''", $itemId);
        if ($images) {
            foreach ($images as $image) {
                $labelData = explode('_', $image['label']);
                if (!empty($labelData)) {
                    $label = $labelData[0];
                    if (array_key_exists($label, $retval)) {
                        if ($image['store_id'] == $storeId || ($image['store_id'] == 0 && empty($retval[$label]))) {
                            if ($label == 'top') {
                                $retval['background-color'] = empty($labelData[1]) ? 'ffffff' : $labelData[1];
                                $retval['border-color'] = empty($labelData[2]) ? $retval['background-color'] : $labelData[2];
                                $retval['font-color'] = empty($labelData[3]) ? '000000' : $labelData[3];
                            } elseif ($label == 'bottom') {
                                $retval['font-color'] = empty($labelData[1]) ? '000000' : $labelData[1];
                            }
                            $retval[$label] = Mage::getStoreConfig('web/unsecure/base_media_url', $storeId) . 'catalog/product' . $image['value'];
                        }
                    }
                }
            }
        }

        return $retval;
    }

    public function getValidTo($serialcode)
    {
        $db = Mage::getSingleton('core/resource')->getConnection('integration_read');

        return $db->fetchOne("SELECT valid_to FROM serialcodes WHERE code = ?", $serialcode);
    }

    public function loadCodesAndExpiryDates($codes)
    {
        if (empty($codes)) {
            return [];
        }

        $conn = Mage::getSingleton("core/resource")->getConnection("integration_read");

        $query = $conn->select()
            ->from(["s" => "serialcodes"], ["code", "valid_to"])
            ->where("code IN (?)", $codes)
            ->where("valid_to IS NOT NULL");

        return $conn->query($query)->fetchAll();
    }

    public function issueCodesForOrder($order, $sendCustomerEmail = true)
    {
        $codeStatus = array();

        try {

            $orderStatus = $order->getStatus();
            if (!in_array($orderStatus, self::$allowedOrderStatuses)) {
                // $this->log(LOG_DEBUG, "Wrong order status '{$orderStatus}' for serialcodes.");
                // $codeStatus[] = "Wrong order state '{$orderState}' for serialcodes.";
                // Do not add this warning to codeStatus. We don't need to spam order history with non essential information.
                return $codeStatus;
            }

            $storeId = $order->getStoreId();
            $incrementId = $order->getIncrementId();
            $orderItems = $order->getItemsCollection()->getItems();

            foreach ($orderItems as $item) {
                $productId = $item->getProductId();
                $qty = round($item->getQtyOrdered());
                $product = Mage::getModel('catalog/product')->setStoreId($storeId)->load($productId);
                $productSku = $product->getSku();
                $sku = trim($product->getData('serial_code_pool'));
                if (empty($sku)) {
                    $sku = trim($product->getSku());
                }

                if ($productId != $product->getId()) {
                    $this->_sendAdminEmail('Unable to load serialcode product', "Order {$incrementId} was unable to load product [{$productSku}]");
                    $this->log(LOG_DEBUG, "Order {$incrementId} was unable to load product [{$productSku}]");
                    $codeStatus[] = "Was unable to load product #{$productId}";
                }

                if ($product->getSerialCodeSerialized()) {

                    $codes = $this->issueCodesForItem($order, $item, $product, true);

                    if ($codes === false) {
                        $this->log(LOG_ERR, "Something went wrong when issuing serialcodes for order {$incrementId}");
                        $codeStatus[] = "Something went wrong when issuing serialcodes for product [{$productSku}]";
                    } elseif ($codes === true) {
                        $this->log(LOG_DEBUG, "Order {$incrementId} already have {$qty} issued serialcodes for [{$productSku}]");
                        $codeStatus[] = "Serialcodes already issued for product [{$productSku}]";
                    } elseif (is_numeric($codes)) {
                        $this->_sendAdminEmail('Unable to get all wanted serialcodes', "Order {$incrementId} was supposed to get {$qty} codes for [{$productSku}] but have {$codes}.");
                        $this->log(LOG_ERR, "Order {$incrementId} was supposed to get {$qty} codes for [{$productSku}] but have {$codes}.");
                        $codeStatus[] = "was supposed to get {$qty} codes for [{$productSku}] but have {$codes}.";
                    } else {

                        // Send email to customer with the issued serialcodes
                        if ($sendCustomerEmail) {
                            $this->sendCustomerEmail($order, $item, $product, $codes);
                            $codeStatus[] = "Successfully sent email to customer with {$qty} codes for [{$productSku}].";
                        }
                    }

                    // Check to see if we need to send email about too few available serialcodes
                    $this->sendWarningEmail($order, $product);
                } else {
                    // $this->log(LOG_DEBUG, "Skipping product [{$productSku}].");
                    // Do not add this warning to codeStatus. We don't need to spam order history with non essential information.
                }
            }

        } catch (Exception $exception) {
            $this->_sendAdminEmail('Exception while issuing serialcode', "Message:\n" . $exception->getMessage() . "\n\nTrace:\n" . $exception->getTraceAsString());
            $this->logException($exception, 'Exception while issuing serialcodes');
            $codeStatus[] = "Exception while issuing serialcodes.";
        }

        return $codeStatus;
    }

    public function issueCodesForItem($order, $item, $product, $updateItem = true)
    {
        $incrementId  = $order->getIncrementId();
        $note         = $this->getNote($order, $item);
        $totalQty     = (int) round($item->getQtyOrdered());
        $sku          = trim($product->getData('serial_code_pool'));
        $db           = Mage::getSingleton("core/resource")->getConnection("integration_write");
        $affectedRows = 0;

        if (empty($sku)) {
            $sku = trim($product->getSku());
        }

        // Wait at most 5 seconds on locks
        $db->exec("SET lock_wait_timeout = " . self::TIMEOUT);

        try {
            $db->exec("LOCK TABLES serialcodes WRITE");

            // Try to load codes, to se if they already have been assigned
            $codes = $db->query("SELECT * FROM `serialcodes` WHERE `sku` = ? AND `note` = ?", [$sku, $note])->fetchAll();
            $codeQty = count($codes);

            // Check to se if we have all codes
            if ($codeQty < $totalQty) {
                // Nope, we do not have all codes. Try to reserve the missing codes.
                // This query can possibly generate warnings in older MySQL-servers, but is (cluster)safe when using ORDER BY <primary key>
                $wantedCodes  = $totalQty - $codeQty;
                // Use sprintf() since prepared statement breaks on this query
                $q = sprintf("UPDATE `serialcodes` SET `note` = %s, `status` = '1', `update_time` = NOW() WHERE `sku` = %s AND `status` = 0 AND (`valid_to` IS NULL OR `valid_to` > DATE(DATE_ADD(NOW(), INTERVAL 14 DAY))) ORDER BY `serialcodes_id` LIMIT %s", $db->quote($note), $db->quote($sku), $db->quote($wantedCodes));
                $affectedRows = $db->query($q)->rowCount();

                // Check to se if we were able to reserve the requested amount of codes
                if ($affectedRows != $wantedCodes) {
                    // No, we were unable to reserve the requested amount of codes. Log this here but handle error later.
                    $this->log(LOG_ERR, "Order {$incrementId} have serialcodes product {$sku} and wanted to reserve {$totalQty} codes but only got {$affectedRows}.");
                }

                $codes = $db->query("SELECT * FROM `serialcodes` WHERE `sku` = ? AND `note` = ?", [$sku, $note])->fetchAll();
                $codeQty = count($codes);
            }

            $db->exec("UNLOCK TABLES");

            $this->log(LOG_DEBUG, "Done processing {$note}:{$sku}, has {$codeQty} of requested {$totalQty}, added {$affectedRows} codes.");

            // Check to se if we got all codes
            if ($codeQty == $totalQty) {
                // Serialcodes already issued
                if ($updateItem) {
                    $this->_updateItem($item, $product, $codes);
                }

                // Return true if nothing was changed
                return $affectedRows > 0 ? $codes : true;
            }

            return $codeQty;

        } catch (Exception $e) {
            $db->exec("UNLOCK TABLES");

            Mage::logException($e);

            $this->_sendAdminEmail('Exception while issuing serialcode', "Message:\n" . $e->getMessage() . "\n\nTrace:\n" . $e->getTraceAsString());
            $this->logException($e, 'Exception while issuing serialcode');

            return false;
        }
    }

    private function _updateItem($item, $product, $codes)
    {
        try {
            $previousSerialCodes = $item->getSerialCodes();
            if (empty($previousSerialCodes)) {
                $serialCodes = array();
                foreach ($codes as $code) {
                    $serialCodes[] = $code["code"];
                }

                $codetype = trim($product->getData('serial_code_type'));
                if (empty($codetype)) {
                    $codetype = 'Serial Code';
                }

                $item->setSerialCodeType($codetype);
                $item->setSerialCodes(implode("\n", $serialCodes));
                $item->save();
            }
        } catch (Exception $exception) {
            $this->_sendAdminEmail('Exception while updating item with serialcods', "Message:\n" . $exception->getMessage() . "\n\nTrace:\n" . $exception->getTraceAsString());
            $this->logException($exception, 'Exception while updating item with serialcods');
        }
    }

    public function getCodes($order, $item, $product)
    {
        try {
            $note = $this->getNote($order, $item);
            $qty = round($item->getQtyOrdered());
            $sku = trim($product->getData('serial_code_pool'));
            if (empty($sku)) {
                $sku = trim($product->getSku());
            }

            $sqlQuery1 = "SELECT * FROM serialcodes WHERE sku = '{$sku}' AND note = '{$note}'";
            $codes = Mage::getSingleton('core/resource')->getConnection('integration_read')->fetchAll($sqlQuery1);
            $codeQty = count($codes);

            // If we did not get any codes, try fetching from codes stored on item
            if ($codeQty == 0) {
                $serialCodes = explode("\n", $item->getSerialCodes());
                if (!empty($serialCodes)) {
                    $safeSerialCodes = array();
                    foreach ($serialCodes as $code) {
                        $safeSerialCodes[] = trim($code);
                    }
                    if (!empty($safeSerialCodes)) {
                        $safeList = implode("','", $safeSerialCodes);
                        $sqlQuery2 = "SELECT * FROM serialcodes WHERE code IN ('{$safeList}')";
                        $codes = Mage::getSingleton('core/resource')->getConnection('integration_read')->fetchAll($sqlQuery2);
                        $codeQty = count($codes);
                    }

                }
            }

            if ($codeQty != $qty) {
                return $codeQty;
            }

            return $codes;

        } catch (Exception $exception) {
            $this->_sendAdminEmail('Exception while fetching serialcodes', "Message:\n" . $exception->getMessage() . "\n\nTrace:\n" . $exception->getTraceAsString());
            $this->logException($exception, 'Exception while fetching serialcodes');
            return false;
        }
    }

    public function sendWarningEmail($order, $product)
    {
        // Check to se if we need to warn about low level of codes
        $email = $product->getData('serial_code_send_warning');
        if (!empty($email)) {
            $sku = trim($product->getData('serial_code_pool'));
            if (empty($sku)) {
                $sku = trim($product->getSku());
            }

            $codetype = trim($product->getData('serial_code_type'));
            if (empty($codetype)) {
                $codetype = 'Serial Code';
            }

            try {
                $db = Mage::getSingleton('core/resource')->getConnection('integration_read');

                // Wait at most 5 seconds on locks
                $db->exec("SET lock_wait_timeout = " . self::TIMEOUT);

                $sqlQuery4 = "SELECT count(serialcodes_id) AS qty_free FROM serialcodes WHERE sku = '{$sku}' AND `status` = 0";
                $available = $db->fetchOne($sqlQuery4);

            } catch(Exception $e) {
                Mage::logException($e);

                return;
            }

            $level = $product->getData('serial_code_warning_level');

            if ($available <= $level) {
                $emailvars = array(
                    'product' => $product->getName(),
                    'available' => $available,
                    'none' => !$available,
                    'codetype' => $codetype,
                    'pool' => $sku,
                    'order' => $order
                );
                if (is_numeric($template = $product->getData('serial_code_warning_template'))) {
                    $emailTemplate = Mage::getModel('core/email_template')->load($template);
                } else {
                    $emailTemplate = Mage::getModel('core/email_template')->loadDefault($template);
                }
                $emails = array_filter(array_map('trim', explode(' ', $email)));
                $email = $emails[0];
                unset($emails[0]);
                $emailTemplate->setSenderName(Mage::getStoreConfig('trans_email/ident_sales/name'));
                $emailTemplate->setSenderEmail(Mage::getStoreConfig('trans_email/ident_sales/email'));
                $bccEmails = array_values($emails);
                if (!empty($bccEmails)) {
                    $emailTemplate->addBcc($bccEmails);
                }
                $emailTemplate->send($email, 'Administrator', $emailvars);
            }
        }
    }

    private function _sendAdminEmail($subject, $body)
    {
        $adminEmailAddresses = array(
            'it@crossroads.se'
        );

        foreach ($adminEmailAddresses as $adminEmailAddress) {
            mail($adminEmailAddress, $subject, $body);
        }
    }

    public function sendCustomerEmailForOrder($order)
    {
        $orderStatus = $order->getStatus();
        if (!in_array($orderStatus, self::$allowedOrderStatuses)) {
            return;
        }

        $storeId = $order->getStoreId();
        $appEmulation = Mage::getSingleton('core/app_emulation');
        $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);
        $orderItems = $order->getItemsCollection()->getItems();
        foreach ($orderItems as $item) {
            $productId = $item->getProductId();
            $product = Mage::getModel('catalog/product')->setStoreId($storeId)->load($productId);
            $sku = trim($product->getData('serial_code_pool'));
            if (empty($sku)) {
                $sku = trim($product->getSku());
            }

            if ($product->getSerialCodeSerialized()) {
                $codes = $this->getCodes($order, $item, $product);
                if (is_array($codes)) {
                    $this->sendCustomerEmail($order, $item, $product, $codes);
                }
            }
        }
        $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
    }

    public function sendCustomerEmail($order, $item, $product, $codes)
    {
        try {
            $productId = $product->getId();
            $storeId = $order->getStoreId();
            $incrementId = $order->getIncrementId();

            $sku = trim($product->getData('serial_code_pool'));
            if (empty($sku)) {
                $sku = trim($product->getSku());
            }

            // Prepare all fields for each code
            $serialCodes = array();
            $codeIds = array();
            $valids = array();
            $pin = array();
            $extra = array();
            foreach ($codes as $code) {
                $serialCodes[] = $code['code'];
                $codeIds[] = $code['serialcodes_id'];
                $valids[$code['code']] = $code['valid_to'];
                $pin[$code['code']] = $code['pin'];
                if (!empty($code['extra'])) {
                    try {
                        $extraValues = json_decode($code['extra'], true);
                        if (!empty($extraValues) && is_array($extraValues)) {
                            $extra[$code['code']] = $extraValues;
                        }
                    } catch (Exception $ex) {
                        // Silently ignore json errors
                    }
                }
                // Pin is supposed to be an extra field in the future.
                if (!empty($code['pin'])) {
                    $extra[$code['code']]['pin'] = $code['pin'];
                }
            }

            $mailarray = array();
            $mailarray['sku'] = $sku;
            $mailarray['template'] = $product->getSerialCodeEmailTemplate();
            $mailarray['emailtype'] = $product->getSerialCodeEmailType();
            $mailarray['images'] = $this->getImagesForEmail($productId, $storeId);
            $mailarray['gift_card_text'] = $product->getData('gift_card_text');
            $mailarray['gift_card_terms'] = $product->getData('gift_card_terms');
            $mailarray['store_name'] = Mage::getStoreConfig('general/store_information/name', $storeId);
            $mailarray['store_logo'] = Mage::getStoreConfig('web/unsecure/base_url', $storeId);
            $mailArray['store_secure_base_url'] = Mage::getStoreConfig('web/secure/base_url', $storeId);

            $logoConfig = Mage::getStoreConfig('design/email/logo', $storeId);
            if (!empty($logoConfig)) {
                $mailarray['store_logo'] .= "media/email/logo/{$logoConfig}";
            } else {
                $mailarray['store_logo'] .= 'skin/frontend/';
                $mailarray['store_logo'] .= Mage::getStoreConfig('design/package/name', $storeId) . '/';
                $mailarray['store_logo'] .= Mage::getStoreConfig('design/theme/skin', $storeId) . '/';
                $mailarray['store_logo'] .= 'images/logo_email.gif';
            }

            $mailarray['codes'] = array();
            $mailarray['valids'] = array();
            $mailarray['validshtml'] = '';
            $mailarray['bcc'] = $product->getSerialCodeSendCopy();
            $rawUrl = $product->getData('gift_card_url');

            $mailarray['SerialCodeImageType'] = $product->getSerialCodeImageType();
            $mailarray['SerialCodeImageParams'] = $product->getSerialCodeImageParams();
            if (empty($mailarray['SerialCodeImageParams'])) {
                $mailarray['SerialCodeImageParams'] = array();
            } else {
                $mailarray['SerialCodeImageParams'] = json_decode($mailarray['SerialCodeImageParams'], true);
            }
            $mailarray['generated_codes'] = '';

            $mailarray['codetype'] = $item->getSerialCodeType();
            if (empty($mailarray['codetype'])) {
                $mailarray['codetype'] = 'Serial Code';
            }

            if (empty($mailarray['html'])) {
                $mailarray['html'] = '<div class="sc_items">';
            }
            if ($mailarray['html'] != '<div class="sc_items">') {
                $mailarray['html'] .= '<br /><br />';
            }
            $mailarray['html'] .= '<span class="sc_product">' . $product->getName() . '</span>';

            $mailarray['codes'] = $serialCodes;
            foreach ($serialCodes as $code) {
                $buttonHtml = "";

                // Fetch image data if code is supposed to have it.
                $mailarray['generated_codes'] .= $this->getImgHTML($mailarray, $code, $storeId);

                if (empty($rawUrl)) {
                    $mailarray['html'] .= "<br /><span class=\"sc_type\">{$mailarray['codetype']}:</span> <span class=\"sc_code\">{$code}</span>";
                    $mailarray['validshtml'] .= "<table width=\"100%\" class=\"callout\" style=\"Margin-bottom:14px;border-collapse:collapse;border-spacing:0;margin-bottom:14px;padding:0;text-align:left;vertical-align:top\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"callout-inner\" style=\"Margin:0;background:#fff;border:none;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:14px;text-align:left;width:100%\"><table class=\"row collapse\" style=\"border-collapse:collapse;border-spacing:0;display:table;padding:0;position:relative;text-align:left;vertical-align:top;width:100%\"><tbody><tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"small-12 large-12 columns first last\" valign=\"middle\" style=\"Margin:0 auto;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0 auto;padding:0;padding-bottom:0;padding-left:0;padding-right:0;text-align:left;width:557px\"><table style=\"border-collapse:collapse;border-spacing:0;padding:0;text-align:left;vertical-align:top;width:100%\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th style=\"Margin:0;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:0;text-align:left\"><h4 class=\"code__title\" style=\"Margin:0;Margin-bottom:14px;color:#9d9d9d;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:20px;margin:0;margin-bottom:14px;padding:0;text-align:center;word-wrap:normal\">" . $this->__('Value code') . "</h4><h4 class=\"code__code\" style=\"Margin:0;Margin-bottom:14px;color:#212121 !important;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:28px;margin:0;margin-bottom:14px;padding:0;text-align:center;word-wrap:normal;text-decoration: none !important;\">{$code}</h4>";

                    if (!empty($mailarray['generated_codes'])) {
                        // if code have generated image (bar, aztec, QR)
                        $mailarray['generated_codes'] .= "<tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"small-12 large-8 columns last\" valign=\"middle\" style=\"Margin:0 auto;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0 auto;padding:0;padding-bottom:0;padding-left:0;padding-right:0;text-align:left;width:373.67px\"><table style=\"border-collapse:collapse;border-spacing:0;padding:0;text-align:left;vertical-align:top;width:100%\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th style=\"Margin:0;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:0;text-align:left\"><h4 class=\"code__title\" style=\"Margin:0;Margin-bottom:14px;Margin-top:20px;color:#9d9d9d;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:20px;margin:0;margin-bottom:14px;margin-top:20px;padding:0;text-align:center;word-wrap:normal\">" . $this->__('Value code') . "</h4><h4 class=\"code__code\" style=\"Margin:0;Margin-bottom:14px;color:#212121 !important;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:28px;margin:0;margin-bottom:14px;padding:0;text-align:center;word-wrap:normal;text-decoration: none !important;\">{$code}</h4>";
                    }
                } else {
                    // Old way to replace variables in URL was %1 for code and %key for other values.
                    $codeUrl = str_replace("%1", $code, $rawUrl);
                    $codeUrl = str_replace("{code}", $code, $codeUrl);

                    // Check to see if any other extra data is supposed to be added to url
                    if (!empty($extra[$code])) {
                        foreach ($extra[$code] as $key => $val) {
                            $codeUrl = str_replace("%".$key, $val, $codeUrl);
                            $codeUrl = str_replace("{".$key."}", $val, $codeUrl);
                        }
                    }
                    // Remove any unknown or empty vars, like {unknown} and {}
                    $codeUrl = preg_replace("/\{[^\{]*\}/", "", $codeUrl);

                    $buttonHtml = "<center style=\"width:100%\"><table style=\"Margin:14px auto 14px auto;border-collapse:collapse;border-spacing:0;float:none;margin:14px auto 14px auto;padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;text-align:center;vertical-align:top;width:auto\"><tr style=\"padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;text-align:left;vertical-align:top\"><td style=\"-moz-hyphens:auto;-webkit-hyphens:auto;Margin:0;border-collapse:collapse!important;color:#5c5c5c;font-family:Helvetica,Arial,sans-serif;font-size:16px;font-weight:400;hyphens:auto;line-height:26px;margin:0;padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;text-align:left;vertical-align:top;word-wrap:break-word\"><table style=\"border-collapse:collapse;border-spacing:0;padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;text-align:left;vertical-align:top\"><tbody><tr style=\"padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;text-align:left;vertical-align:top\"><td style=\"-moz-hyphens:auto;-webkit-hyphens:auto;Margin:0;background:" . Mage::getStoreConfig('Email/translations/theme_color', $storeId) . ";border:0 solid #fff;border-collapse:collapse!important;border-radius:3px;color:#fff;font-family:Helvetica,Arial,sans-serif;font-size:16px;font-weight:400;hyphens:auto;line-height:26px;margin:0;opacity:.9;padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;text-align:left;transition:color .25s,background .25s,border-color .25s,opacity .25s,transform .25s,box-shadow .25s;vertical-align:top;word-wrap:break-word\"><a align=\"center\" href=\"" . $codeUrl . "\" style=\"border:0 solid #fff;color:" . Mage::getStoreConfig('Email/translations/text_on_background_color', $storeId) . ";display:inline-block;font-family:Helvetica,Arial,sans-serif;font-size:20px;font-weight:700;line-height:1.6;padding:10px 20px 10px 20px;text-align:left;text-decoration:none\">" . $this->__('Redeem voucher') . "</a></td></tr></tbody></table></td></tr></table></center>";
                    $mailarray['html'] .= "<br /><span class=\"sc_type\">{$mailarray['codetype']}:</span> <span class=\"sc_code\"><a style=\"color:#212121;\" href=\"{$codeUrl}\">{$code}</a></span>";
                    $mailarray['validshtml'] .= "<table width=\"100%\" class=\"callout\" style=\"Margin-bottom:14px;border-collapse:collapse;border-spacing:0;margin-bottom:14px;padding:0;text-align:left;vertical-align:top\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"callout-inner\" style=\"Margin:0;background:#fff;border:none;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:14px;text-align:left;width:100%\"><table class=\"row collapse\" style=\"border-collapse:collapse;border-spacing:0;display:table;padding:0;position:relative;text-align:left;vertical-align:top;width:100%\"><tbody><tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"small-12 large-12 columns first last\" valign=\"middle\" style=\"Margin:0 auto;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0 auto;padding:0;padding-bottom:0;padding-left:0;padding-right:0;text-align:left;width:557px\"><table style=\"border-collapse:collapse;border-spacing:0;padding:0;text-align:left;vertical-align:top;width:100%\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th style=\"Margin:0;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:0;text-align:left\"><h4 class=\"code__title\" style=\"Margin:0;Margin-bottom:14px;color:#9d9d9d;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:20px;margin:0;margin-bottom:14px;padding:0;text-align:center;word-wrap:normal\">" . $this->__('Value code') . "</h4><h4 class=\"code__code\" style=\"Margin:0;Margin-bottom:14px;color:#212121 !important;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:28px;margin:0;margin-bottom:14px;padding:0;text-align:center;word-wrap:normal;text-decoration: none !important;\"><a style=\"color:#212121;\" href=\"{$codeUrl}\">{$code}</a></h4>";

                    if (!empty($mailarray['generated_codes'])) {
                        // if code have generated image (bar, aztec, QR)
                        $mailarray['generated_codes'] .= "<tr style=\"padding:0;text-align:left;vertical-align:top\"><th class=\"small-12 large-8 columns last\" valign=\"middle\" style=\"Margin:0 auto;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0 auto;padding:0;padding-bottom:0;padding-left:0;padding-right:0;text-align:left;width:373.67px\"><table style=\"border-collapse:collapse;border-spacing:0;padding:0;text-align:left;vertical-align:top;width:100%\"><tr style=\"padding:0;text-align:left;vertical-align:top\"><th style=\"Margin:0;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:400;line-height:20px;margin:0;padding:0;text-align:left\"><h4 class=\"code__title\" style=\"Margin:0;Margin-bottom:14px;Margin-top:20px;color:#9d9d9d;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:20px;margin:0;margin-bottom:14px;margin-top:20px;padding:0;text-align:center;word-wrap:normal\">" . $this->__('Value code') . "</h4><h4 class=\"code__code\" style=\"Margin:0;Margin-bottom:14px;color:#212121 !important;font-family:Helvetica,Arial,sans-serif;font-size:24px;font-weight:700!important;line-height:28px;margin:0;margin-bottom:14px;padding:0;text-align:center;word-wrap:normal;text-decoration: none !important;\"><a style=\"color:#212121 !important\" href=\"{$codeUrl}\">{$code}</a></h4>";
                    }
                }

                if (!empty($valids[$code])) {
                    $mailarray['html'] .= " <span class=\"sc_valid\">{$valids[$code]}</span>";
                    $mailarray['valids'][] = "{$code}&nbsp;({$valids[$code]})";
                    $mailarray['validshtml'] .= "<p class=\"code__validity\" style=\"Margin:0!important;Margin-bottom:14px;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:300!important;line-height:20px;margin:0!important;margin-bottom:14px;padding:0;text-align:center\">" . $this->__('Valid until') . "&nbsp;{$valids[$code]}</p>";

                    if (!empty($mailarray['generated_codes'])) {
                        // if code have generated image (bar, aztec, QR)
                        $mailarray['generated_codes'] .= "<p class=\"code__validity\" style=\"Margin:0!important;Margin-bottom:14px;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:300!important;line-height:20px;margin:0!important;margin-bottom:14px;padding:0;text-align:center\">" . $this->__('Valid until') . "&nbsp;{$valids[$code]}</p>";
                    }
                }

                if (!empty($extra[$code])) {
                    foreach ($extra[$code] as $key => $val) {
                        switch (strtolower($key)) {
                            case 'pin':
                                $mailarray['html'] .= " <span class=\"sc_valid\">{$val}</span>";
                                $mailarray['validshtml'] .= "<p class=\"code__validity\" style=\"Margin:0!important;Margin-bottom:14px;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:300!important;line-height:20px;margin:0!important;margin-bottom:14px;padding:0;text-align:center\">" . $this->__('PIN code') . "&nbsp;{$val}</p>";

                                if (!empty($mailarray['generated_codes'])) {
                                    // if code have generated image (bar, aztec, QR)
                                    $mailarray['generated_codes'] .= "<p class=\"code__validity\" style=\"Margin:0!important;Margin-bottom:14px;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:300!important;line-height:20px;margin:0!important;margin-bottom:14px;padding:0;text-align:center\">" . $this->__('PIN code') . "&nbsp;{$val}</p>";
                                }
                                break;

                            case 'hash':
                                // Do not add anything for this extra field
                                break;

                            default:
                                $mailarray['html'] .= " <span class=\"sc_valid\">{$val}</span>";
                                $mailarray['validshtml'] .= "<p class=\"code__validity\" style=\"Margin:0!important;Margin-bottom:14px;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:300!important;line-height:20px;margin:0!important;margin-bottom:14px;padding:0;text-align:center\">" . $this->__($key) . "&nbsp;{$val}</p>";

                                if (!empty($mailarray['generated_codes'])) {
                                    // if code have generated image (bar, aztec, QR)
                                    $mailarray['generated_codes'] .= "<p class=\"code__validity\" style=\"Margin:0!important;Margin-bottom:14px;color:#616161;font-family:Helvetica,Arial,sans-serif;font-size:14px;font-weight:300!important;line-height:20px;margin:0!important;margin-bottom:14px;padding:0;text-align:center\">" . $this->__($key) . "&nbsp;{$val}</p>";
                                }
                        }
                    }
                }

                if (!empty($mailarray['generated_codes'])) {
                    // if code have generated image (bar, aztec, QR)

                    if (!empty($buttonHtml)) {
                        $mailarray['generated_codes'] .= $buttonHtml;
                    }

                    $mailarray['generated_codes'] .= '</th></tr></table></th></tr></tbody></table></th></tr></table>';
                }

                if (!empty($mailarray['validshtml'])) {

                    if (!empty($buttonHtml)) {
                        $mailarray['validshtml'] .= $buttonHtml;
                    }

                    $mailarray['validshtml'] .= '</th></tr></table></th></tr></tbody></table></th></tr></table>';
                }
            }

            if (is_numeric($mailarray['template'])) {
                $emailTemplate = Mage::getModel('core/email_template')->load($mailarray['template']);
            } else {
                $emailTemplate = Mage::getModel('core/email_template')->loadDefault($mailarray['template']);
            }
            $itemstext = strip_tags(str_replace('<br />', "\n", $mailarray['html']));
            $emailvars = array(
                'itemstext' => $itemstext,
                'itemshtml' => $mailarray['html'] . '</div>',
                'codetype' => $mailarray['codetype'],
                'emailtype' => $mailarray['emailtype'],
                'store_name' => $mailarray['store_name'],
                'store_logo' => $mailarray['store_logo'],
                'image_top' => $mailarray['images']['top'],
                'image_bottom' => $mailarray['images']['bottom'],
                'background-color' => $mailarray['images']['background-color'],
                'border-color' => $mailarray['images']['border-color'],
                'font-color' => $mailarray['images']['font-color'],
                'codes' => implode(', ', $mailarray['codes']),
                'validcodes' => implode(', ', $mailarray['valids']),
                'validshtml' => $mailarray['validshtml'],
                'gift_card_text' => $mailarray['gift_card_text'],
                'gift_card_terms' => $mailarray['gift_card_terms'],
                'generated_codes' => $mailarray['generated_codes'],
                'customer_name' => $order->getBillingAddress()->getName(),
                'order' => $order,
                'product' => $product,
                'msrp' => Mage::helper('core')->currency($product->getMsrp(),false,false),
                'hasImages' => !empty($mailarray['generated_codes']),
                'hasValidity' => !empty($mailarray['valids']),
                'year' => date('Y'),
                'secure_base_url' => $mailArray['store_secure_base_url'],
            );
            $emailTemplate->setSenderName(Mage::getStoreConfig('trans_email/ident_sales/name', $storeId));
            $emailTemplate->setSenderEmail(Mage::getStoreConfig('trans_email/ident_sales/email', $storeId));
            $bcc = $mailarray['bcc'];
            if ($bcc) {
                $emails = array_filter(array_map('trim', explode(' ', $bcc)));
                $emailTemplate->addBcc($emails);
            }

            $customerEmail = $order->getCustomerEmail();
            if (empty($customerEmail)) {
                $this->_sendAdminEmail('Serialcodes order missing email address', "Order {$incrementId} is missing customer email address.\nThe following codes for product [{$mailarray['sku']}] should have been sent: {$emailvars['codes']}\n");
                $this->log(LOG_ERR, "Order {$incrementId} is missing customer email address.");
            } else {

                if (self::USE_EMAIL_QUEUE) {
                    /** @var $emailQueue Mage_Core_Model_Email_Queue */
                    $emailQueue = Mage::getModel('core/email_queue');
                    $emailQueue->setEventType(self::EMAIL_EVENT_TYPE)
                        ->setEntityType(Mage_Sales_Model_Order::ENTITY)
                        ->setEntityId($order->getId())
                        ->setIsForceCheck(false);
                    $emailTemplate->setQueue($emailQueue);
                }

                $emailTemplate->send($customerEmail, $order->getBillingAddress()->getName(), $emailvars);
                $this->log(LOG_DEBUG, "Email sent to {$customerEmail}.");
            }

        } catch (Exception $exception) {
            $this->_sendAdminEmail('Exception while sending customer email with serialcodes', "Message:\n" . $exception->getMessage() . "\n\nTrace:\n" . $exception->getTraceAsString());
            $this->logException($exception, 'Exception while sending customer email with serialcodes');
        }

    }

    public function getDataForOrder($order)
    {
        $sqlQuery = "
            SELECT
                tmp.entity_id,
                tmp.increment_id,
                tmp.state,
                tmp.`status`,
                tmp.payment_method,
                tmp.product_id,
                tmp.sku,
                tmp.serial_code_type,
                tmp.serial_codes,
                tmp.attribute_code,
                COALESCE(max(tmp.local_attribute_value), max(tmp.global_attribute_value), 0) AS attribute_value
            FROM (
                SELECT
                    o.entity_id,
                    o.increment_id,
                    o.state,
                    o.`status`,
                    p.method AS payment_method,
                    i.product_id,
                    e.sku,
                    i.serial_code_type,
                    i.serial_codes,
                    a.attribute_code,
                    IF(ei.store_id = 0, ei.`value`, 0) AS global_attribute_value,
                    IF(ei.store_id > 0, ei.`value`, 0) AS local_attribute_value
                FROM sales_flat_order o
                JOIN sales_flat_order_payment p ON p.parent_id = o.entity_id
                JOIN sales_flat_order_item i ON i.order_id = o.entity_id
                JOIN catalog_product_entity e ON e.entity_id = i.product_id
                JOIN catalog_product_entity_int ei ON ei.entity_id = e.entity_id AND (ei.store_id = 0 OR ei.store_id = :storeId)
                JOIN eav_attribute a ON a.attribute_id = ei.attribute_id
                WHERE o.entity_id = :entityId AND a.attribute_code = :attributeCode
            ) tmp
            GROUP BY tmp.entity_id
        ";
        $params = [
            'entityId' => $order->getId(),
            'attributeCode' => 'serial_code_serialized',
            'storeId' => $order->getStoreId()
        ];
        return Mage::getSingleton('core/resource')->getConnection('core_read')->fetchAll($sqlQuery, $params);

    }

    public function getCodesFromCheckoutSession()
    {
        $items  = [];
        $sess   = Mage::getSingleton("checkout/session");
        $order  = Mage::getModel("sales/order")->load($sess->getLastOrderId());

        if(empty($order) || ! $order->getId()) {
            return null;
        }

        foreach($order->getAllItems() as $item) {
            if($item->getProduct()->getSerialCodeSerialized()) {
                $codes          = array_filter(array_map("trim", explode("\n", $item->getSerialCodes() ?: "")));
                $codesWithDates = array_combine($codes, array_map(function($code) {
                    return [
                        "code"       => $code,
                        "expiryDate" => null,
                    ];
                }, $codes));

                foreach($this->loadCodesAndExpiryDates($codes) as $c) {
                    $codesWithDates[$c["code"]]["expiryDate"] = date("Y-m-d\TH:i:s\Z", strtotime($c["valid_to"]));
                }

                $items[] = [
                    "sku"          => $item->getSku(),
                    "parentSku"    => $item->getParentItem() ? $item->getParentItem()->getSku() : null,
                    "name"         => $item->getName(),
                    "requestedQty" => (int)$item->getQtyOrdered(),
                    "codes"        => array_values($codesWithDates),
                ];
            }
        }

        return $items;
    }

    public function getNote($order, $item)
    {
        $incrementId = $order->getIncrementId();
        $orderId = $order->getId();
        $itemId = $item->getId();
        $magentoId = Mage::getStoreConfig('integration/general/magento_instance') - 0;
        $note = "{$incrementId}:{$orderId}:{$itemId}:{$magentoId}";

        return $note;
    }

    public function log($logType, $message)
    {
        // $logType: LOG_ERR, LOG_WARNING, LOG_DEBUG
        // $this->log(LOG_DEBUG, "Message");
        Mage::log("{$message}", $logType, $this->_logFile, true);
    }

    public function logException($exception, $messages = array())
    {
        if (!is_array($messages)) {
            $messages = array($messages);
        }

        foreach ($messages as $message) {
            $this->log(LOG_ERR, $message);
        }

        $this->log(LOG_ERR, 'Message: ' . $exception->getMessage());
        $this->log(LOG_ERR, "Trace:\n" . $exception->getTraceAsString());
    }

}
