<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class InfoTest extends TestCase {
    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testFixedDiscountFreeshipping() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setCouponCode("thetestcoupon");
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate")->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $firstItem = $quote->getAllItems()[0];

        $this->assertEquals(12.34, $firstItem->getBaseRowTotal());
        $this->assertEquals(2.0, $firstItem->getBaseDiscountAmount());
        $this->assertEquals(23.71, $quote->getBaseGrandTotal());
        $this->assertEquals(10.34, $quote->getBaseSubtotalWithDiscount());
    }

    public function testFreeproduct() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setCouponCode("freeproduct");
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate")->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $firstItem = $quote->getAllItems()[0];

        $this->assertEquals(12.34, $firstItem->getBaseRowTotal());
        $this->assertEquals(12.34, $firstItem->getBaseDiscountAmount());
        $this->assertEquals(13.37, $quote->getBaseGrandTotal());
        $this->assertEquals(0, $quote->getBaseSubtotalWithDiscount());
    }

    public function testPercentAndShipping() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setCouponCode("free");
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate")->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $firstItem = $quote->getAllItems()[0];

        $this->assertEquals(12.34, $firstItem->getBaseRowTotal());
        $this->assertEquals(12.34, $firstItem->getBaseDiscountAmount());
        $this->assertEquals(0, $quote->getBaseGrandTotal());
        $this->assertEquals(-13.37, $quote->getBaseSubtotalWithDiscount());
    }

    public function testPercentFreeshipping() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setCouponCode("thepercent");
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate")->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $firstItem = $quote->getAllItems()[0];

        $this->assertEquals(12.34, $firstItem->getBaseRowTotal());
        $this->assertEquals(4.94, $firstItem->getBaseDiscountAmount());
        $this->assertEquals(20.77, $quote->getBaseGrandTotal());
        $this->assertEquals(7.4, $quote->getBaseSubtotalWithDiscount());
    }

    public function testFixedWithShipping() {
        global $_SESSION;

        unset($_SESSION["checkout"]);

        MagentoManager::init();

        // We have to use load, cannot use loadByAttribute or similar
        $product = Mage::getModel("catalog/product")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $request = new Varien_Object([
            "qty" => 1,
        ]);
        $quote = Mage::getSingleton("checkout/session")->getQuote();

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->addProduct($product, $request);
        $quote->setCouponCode("20coupon");
        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate")->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $firstItem = $quote->getAllItems()[0];

        $this->assertEquals(12.34, $firstItem->getBaseRowTotal());
        $this->assertEquals(12.34, $firstItem->getBaseDiscountAmount());
        $this->assertEquals(5.71, $quote->getBaseGrandTotal());
        $this->assertEquals(-7.66, $quote->getBaseSubtotalWithDiscount());
    }
}
