<?php

class Crossroads_Retain24_Helper_Data extends Mage_Core_Helper_Abstract
{
    const CONFIG_ENABLED                   = "retain24/general/enabled";
    const CONFIG_TEST                      = "retain24/general/test";
    const CONFIG_COUNTRY_NUMBER            = "retain24/general/country_number";
    const CONFIG_SSL_KEY_NAME              = "retain24/general/ssl_key_name";
    const CONFIG_RECAPTCHA_SECRET          = "retain24/recaptcha/secret";
    const CONFIG_RECAPTCHA_SCORE_THRESHOLD = "retain24/recaptcha/score_threshold";

    // Email setup
    const USE_EMAIL_QUEUE = true; // Magento >= 1.9 uses this
    const EMAIL_EVENT_TYPE = 'retain24';

    // URL's for the Retain24 endpoint.
    const BASE_URL_TEST        = "https://pilot.mvoucher.se/ta/servlet/TA_XML?TA_ACTION=";
    const BASE_URL_LIVE        = "https://live.mvoucher.se/ta/servlet/TA_XML?TA_ACTION=";
    const TEMPLATELIST         = "5-45103";
    const VALIDATE             = "5-43101";
    const RESERVE_INSTANCE     = "5-43109";
    const REDEEM_RESERVATION   = "5-43110";
    const VALIDATE_RESERVATION = "5-43112";
    const CANCEL_RESERVATION   = "5-43111";
    const SEND                 = "5-45102";
    const TRANSACTION_REPORT   = "5-49081";

    // Product attributes
    const VOUCHER_PRODUCT        = "is_retain24_product";
    const VOUCHER_TEMPLATE       = "retain24_template_id";
    const VOUCHER_DYNAMIC        = "retain24_dynamic";
    const VOUCHER_REQUIRE_PIN    = "retain24_require_pin";
    const VOUCHER_MIN            = "retain24_min_value";
    const VOUCHER_MAX            = "retain24_max_value";
    const VOUCHER_VALUE          = "retain24_value";
    const VOUCHER_STEPS          = "retain24_value_steps";
    const VOUCHER_MULTIPLIER     = "retain24_value_multiplier";
    const VOUCHER_VALID_TO       = "retain24_valid_to";
    const VOUCHER_SALES_END      = "retain24_sales_end";
    const VOUCHER_USAGE_START    = "retain24_usage_start";
    const VOUCHER_DISTRIBUTION   = "retain24_distribution";
    const VOUCHER_EMAIL_TEMPLATE = "retain24_email_template";
    const VOUCHER_EMAIL_BCC      = "retain24_email_bcc";
    const VOUCHER_EMAIL_TEXT     = "gift_card_text";
    const VOUCHER_EMAIL_TERMS    = "gift_card_terms";
    const VOUCHER_IMAGE_TYPE     = "serial_code_image_type";
    const VOUCHER_IMAGE_PARAMS   = "serial_code_image_params";

    // Retain24 distribution methods (CHANNEL_CODE)
    const DISTRIBUTION_SMS      = 1;
    const DISTRIBUTION_EMAIL    = 2;
    const DISTRIBUTION_EXTERNAL = 4;
    const DISTRIBUTION_POST     = 5;
    const DISTRIBUTION_PRINTOUT = 6;

    // Retain24 cnptype
    const CPNTYPE_GIFT_CARD          = 1;
    const CPNTYPE_COUPON             = 2;
    const CPNTYPE_BONUS_VOUCHER      = 3;
    const CPNTYPE_CREDIT_NOTE        = 4;
    const CPNTYPE_BONUS_CARD         = 5;
    const CPNTYPE_COMPENSATION_CHECK = 6;
    const CPNTYPE_EXTERNAL           = 7;
    const CPNTYPE_DICOUNT_COUPON     = 8;

    protected $cpntypes = [
        self::CPNTYPE_GIFT_CARD,
        self::CPNTYPE_COUPON,
        self::CPNTYPE_BONUS_VOUCHER,
        self::CPNTYPE_CREDIT_NOTE,
        self::CPNTYPE_BONUS_CARD,
        self::CPNTYPE_COMPENSATION_CHECK,
        self::CPNTYPE_EXTERNAL,
        self::CPNTYPE_DICOUNT_COUPON
    ];

    // Magento Access Control
    const RETAIN24_ACL_ORDER_ITEM_VIEW  = "admin/sales/order/actions/retain24_view";
    const RETAIN24_ACL_ORDER_ITEM_ISSUE = "admin/sales/order/actions/retain24_issue";
    const RETAIN24_ACL_ORDER_ITEM_EMAIL = "admin/sales/order/actions/retain24_email";

    // URL to controller that fetches code image by hash (barcode, QRcode, etc)
    const RETAIN24_IMAGE_URL   = "_auc" . DS . "retain24" . DS . "get_image" . DS . "hash" . DS;

    protected $voucherItemListCache = [];
    protected $currentStoreId = null;

    public static $issueVoucherOrderStatuses = [
        Mage_Sales_Model_Order::STATE_PROCESSING,
        Mage_Sales_Model_Order::STATE_COMPLETE
    ];

    public function getXMLDebugLog() {
        return Mage::getBaseDir("var").DS.'log'.DS.'retain24.xmldebug.log';
    }

    // Retain24 ssl certificates
    public function getCrtFile() {
        return Mage::getBaseDir("etc").DS."retain24".DS."test_crt.pem";
    }

    public function getKeyFile() {
        return Mage::getBaseDir("etc").DS."retain24".DS."test_key.pem";
    }

    // Endpoint URL.
    protected function get_url($store, $url)
    {
        // Save store id for later use
        $storeId = $store->getId();
        if ($this->currentStoreId != $storeId) {
            $this->currentStoreId = $storeId;
        }

        if($store->getConfig(self::CONFIG_TEST)) {
            return self::BASE_URL_TEST.rawurlencode($url);
        }

        return self::BASE_URL_LIVE . rawurlencode($url);
    }

    // Initialize XML request with common data.
    protected function init_xml()
    {
        $xml = new SimpleXMLElement('<?xml version="1.0" encoding="ISO-8859-1"?><TICKETANYWHERE></TICKETANYWHERE>');
        $coupon = $xml->addChild('COUPON');
        $coupon->addAttribute('VER', '1.0');

        return $xml;
    }

    // Post data to Retain24
    protected function post_data($url, $xml)
    {
        // Find out which ssl key to use
        $crtFile = $this->getCrtFile();
        $keyFile = $this->getKeyFile();

        if ($this->currentStoreId !== null && !Mage::getStoreConfig(self::CONFIG_TEST, $this->currentStoreId)) {
            $keyName = Mage::getStoreConfig(self::CONFIG_SSL_KEY_NAME, $this->currentStoreId);
            if (!empty($keyName)) {
                $tmpFile1 = DS.'etc'.DS.'ssl'.DS.'retain24'.DS.$keyName."_crt.pem";
                $tmpFile2 = DS.'etc'.DS.'ssl'.DS.'retain24'.DS.$keyName."_key.pem";
                if (file_exists($tmpFile1) && file_exists($tmpFile2)) {
                    $crtFile = $tmpFile1;
                    $keyFile = $tmpFile2;
                }
            }
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_VERBOSE, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_SSLKEY, $keyFile);
        curl_setopt($ch, CURLOPT_SSLCERT, $crtFile);
        curl_setopt($ch, CURLOPT_SSLCERTTYPE, 'PEM');
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_SSLVERSION, 6); // Force TLS 1.2
        curl_setopt($ch, CURLOPT_POSTFIELDS, $xml->asXML());
        $response = curl_exec($ch);

        if(curl_errno($ch)) {
            throw new Exception(sprintf("%s: Curl Error (%d) %s", __METHOD__, curl_errno($ch), curl_error($ch)));
        }

        return $response;
    }

    public function formatPhone($store, $phone)
    {
        $phone = preg_replace("/[^0-9+]/", "", $phone);

        if(strlen($phone) > 0 && $phone[0] === '+') {
            return "00".substr($phone, 1);
        }
        else if(substr($phone, 0, 2) === "00") {
            return $phone;
        }
        else if(strlen($phone) > 0 && $phone[0] === '0') {
            $phone = substr($phone, 1);
        }

        return "00".$store->getConfig(self::CONFIG_COUNTRY_NUMBER).$phone;
    }

    /**
     * List the various templates we can use.
     *
     * @param  Mage_Core_Store
     * @return array|null
     */
    public function templatelist($store)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('TEMPLATELIST');


        $result = $this->post_data($this->get_url($store, self::TEMPLATELIST), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->TEMPLATELIST)) {
            return json_decode(json_encode($xmlData->COUPON->RESPONSE->TEMPLATELIST), true);
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        return null;
    }

    /**
     * Validate retain24 code.
     *
     * @param  Mage_Core_Store
     * @param  string
     * @param  boolean
     * @param  string|null
     * @return array|null
     */
    public function validate($store, $code, $isPhone, $pin = null)
    {
        if (is_numeric($store)) {
            $store = Mage::app()->getStore($store);
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('VALIDATE');
        $xml->COUPON->VALIDATE->addAttribute('TYPE', 'STANDARD');

        $phone = $this->formatPhone($store, $code);
        if($isPhone) {
            $xml->COUPON->VALIDATE->addChild('MSISDN', $phone);
        } else {
            $xml->COUPON->VALIDATE->addChild('MULTICODE', $code);
        }

        if($pin) {
            $xml->COUPON->VALIDATE->addChild('PIN', $pin);
        }

        $result = $this->post_data($this->get_url($store, self::VALIDATE), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE)) {
            $data = json_decode(json_encode($xmlData->COUPON->RESPONSE), true);
            Mage::log(sprintf("Retain24::VALIDATE(msisdn: %s, multicode: %s): %s", $phone, $code, json_encode($data, JSON_UNESCAPED_SLASHES)));

            if (!empty($data['CPNINFO']['@attributes'])) {
                // If result only contains 1 voucher, make that the only result in $cpninfo
                $cpninfo = [$data['CPNINFO']];
            } else {
                $cpninfo = $data['CPNINFO'];
            }
            foreach ($cpninfo as $voucher) {
                // Possible flaw: We only use first usable voucher
                if ($voucher['ERRORCODE'] == 0) {
                    $voucher["ID"] = $voucher["@attributes"]["ID"];
                    unset($voucher["@attributes"]);
                    return $voucher;
                }
            }

            if ($test) {
                file_put_contents($this->getXMLDebugLog(), $xmlData->asXML(), FILE_APPEND);
            }

            return null;
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        return null;

    }

    /**
     * Validate retain24 code with TYPE=PHONE to obtain USERDEFINED.
     *
     * @param  Mage_Core_Store
     * @param  string
     * @param  boolean
     * @param  string|null
     * @return array|null
     */
    public function validate_phone($store, $code, $isPhone, $pin = null)
    {
        if (is_numeric($store)) {
            $store = Mage::app()->getStore($store);
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('VALIDATE');
        $xml->COUPON->VALIDATE->addAttribute('TYPE', 'PHONE');

        $phone = $this->formatPhone($store, $code);
        if($isPhone) {
            $xml->COUPON->VALIDATE->addChild('MSISDN', $phone);
        } else {
            $xml->COUPON->VALIDATE->addChild('MULTICODE', $code);
        }

        if($pin) {
            $xml->COUPON->VALIDATE->addChild('PIN', $pin);
        }

        $result = $this->post_data($this->get_url($store, self::VALIDATE), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE)) {
            $data = json_decode(json_encode($xmlData->COUPON->RESPONSE), true);
            Mage::log(sprintf("Retain24::VALIDATE_PHONE(msisdn: %s, multicode: %s): %s", $phone, $code, json_encode($data, JSON_UNESCAPED_SLASHES)));

            if (!empty($data['CPNINFO']['@attributes'])) {
                // If result only contains 1 voucher, make that the only result in $cpninfo
                $cpninfo = [$data['CPNINFO']];
            } else {
                $cpninfo = $data['CPNINFO'];
            }
            foreach ($cpninfo as $voucher) {
                // Possible flaw: We only use first usable voucher
                if (empty($voucher['ERRORCODE'])) {
                    $voucher["ID"] = $voucher["@attributes"]["ID"];
                    unset($voucher["@attributes"]);
                    return $voucher;
                }
            }

            if ($test) {
                file_put_contents($this->getXMLDebugLog(), $xmlData->asXML(), FILE_APPEND);
            }

            return null;
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        return null;

    }

    /**
     * Reserve specific amount (qty) on a retain24 code.
     *
     * @param  Mage_Core_Store
     * @param  string
     * @param  int
     * @param  string
     * @return array
     */
    public function reserve_instance($store, $id, $qty, $pin_code)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('RESERVE_INSTANCE');
        $xml->COUPON->RESERVE_INSTANCE->addChild('CPNSELECTION');
        $xml->COUPON->RESERVE_INSTANCE->CPNSELECTION->addChild('ID', $id);
        $xml->COUPON->RESERVE_INSTANCE->CPNSELECTION->addChild('QTY', $qty);
        $xml->COUPON->RESERVE_INSTANCE->CPNSELECTION->addChild('PINCODE', $pin_code);

        $result = $this->post_data($this->get_url($store, self::RESERVE_INSTANCE), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->RESERVATION)) {
            $data = json_decode(json_encode($xmlData->COUPON->RESPONSE), true);
            Mage::log(sprintf("Retain24::RESERVE_INSTANCE(id: %s, qty: %s): %s", $id, $qty, json_encode($data, JSON_UNESCAPED_SLASHES)));

            return $data["RESERVATION"];
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        Mage::throwException('Missing response: RESERVATION');
    }

    /**
     * Redeem reservation based on values from validation and reservation. Also takes amount (qty).
     *
     * @param  Mage_Core_Store
     * @param  string
     * @param  string
     * @param  int
     * @return array
     */
    public function redeem_reservation($store, $reference_id, $cpnselection_id, $qty)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('REDEEM_RESERVATION');
        $xml->COUPON->REDEEM_RESERVATION->addChild('REFERENCE');
        $xml->COUPON->REDEEM_RESERVATION->REFERENCE->addAttribute("ID", $reference_id);
        $xml->COUPON->REDEEM_RESERVATION->REFERENCE->addChild('CPNSELECTION');
        $xml->COUPON->REDEEM_RESERVATION->REFERENCE->CPNSELECTION->addAttribute("ID", $cpnselection_id);
        $xml->COUPON->REDEEM_RESERVATION->REFERENCE->CPNSELECTION->addChild('QTY', $qty);

        $result = $this->post_data($this->get_url($store, self::REDEEM_RESERVATION), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->RESERVATION)) {
            $data = json_decode(json_encode($xmlData->COUPON->RESPONSE), true);
            if (!empty($data["RESERVATION"]["@attributes"]["REFERENCE"])) {
                $return_data = $data["RESERVATION"];
                $return_data["REFERENCE"] = $data["RESERVATION"]["@attributes"]["REFERENCE"];
                unset($return_data["@attributes"]);

                return $return_data;
            }
            if ($test) {
                file_put_contents($this->getXMLDebugLog(), $xmlData->asXML(), FILE_APPEND);
            }

            Mage::throwException('Missing REFERENCE in response');
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        Mage::throwException('Missing response: RESERVATION');
    }

    /**
     * Validate reservation. Based on value from reservation.
     *
     * @param  Mage_Core_Store
     * @param  string
     * @return array
     */
    public function validate_reservation($store, $reference_id)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('VALIDATE_RESERVATION');
        $xml->COUPON->VALIDATE_RESERVATION->addChild('REFERENCE', $reference_id);

        $result = $this->post_data($this->get_url($store, self::VALIDATE_RESERVATION), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->RESERVATION)) {
            $data = json_decode(json_encode($xmlData->COUPON->RESPONSE), true);
            if (!empty($data["RESERVATION"]["@attributes"]["REFERENCE"])) {
                $return_data = $data["RESERVATION"];
                $return_data["REFERENCE"] = $data["RESERVATION"]["@attributes"]["REFERENCE"];
                unset($return_data["@attributes"]);

                Mage::log(sprintf("Retain24::VALIDATE_RESERVATION(reference: %s): %s", $reference_id, json_encode($return_data, JSON_UNESCAPED_SLASHES)));
                return $data;
            }

            if ($test) {
                file_put_contents($this->getXMLDebugLog(), $xmlData->asXML(), FILE_APPEND);
            }

            Mage::throwException('Missing REFERENCE in response');
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        Mage::throwException('Missing response: RESERVATION');

    }

    /**
     * Cancel reservation. Based on value from reservation.
     *
     * @param  Mage_Core_Store
     * @param  string
     * @return array
     */
    public function cancel_reservation($store, $reference_id)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $xml = $this->init_xml();
        $xml->COUPON->addChild('CANCEL_RESERVATION');
        $xml->COUPON->CANCEL_RESERVATION->addChild('REFERENCE', $reference_id);

        $result = $this->post_data($this->get_url($store, self::CANCEL_RESERVATION), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->RESERVATION_REFERENCE)) {
            $data = json_decode(json_encode($xmlData->COUPON->RESPONSE), true);
            Mage::log(sprintf("Retain24::CANCEL_RESERVATION(reference: %s): %s", $reference_id, json_encode($data, JSON_UNESCAPED_SLASHES)));
            return $data["RESERVATION_REFERENCE"];
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }

        Mage::throwException('Missing response: RESERVATION_REFERENCE');

    }

    /**
     * Purchase code/voucher for specific item or all items in order.
     *
     * @param  Mage_Sales_Model_Order
     * @param  int
     * @param  int
     * @return void
     */
    public function purchase_voucher($order, $specificItemId = null, $qty = 0)
    {
        // Let's find out if order contains any voucher products.
        $itemList = $this->getVoucherItemList($order);

        // Loop through all voucher products, if we have any.
        if (!empty($itemList)) {

            Mage::log(
                sprintf(
                    "Retain24::purchase_voucher(order: %s): %s",
                    $order->getIncrementId(),
                    json_encode($itemList, JSON_UNESCAPED_SLASHES)
                )
            );

            foreach ($itemList as $itemId => $itemData) {

                // Only process vouchers for specified item, if supplied
                if (!empty($specificItemId) && $itemId != $specificItemId) {
                    continue;
                }

                // Check to se if we are supposed to purchase specified amount instead of all
                if (!empty($specificItemId) && !empty($qty)) {
                    $itemData['qty'] = $qty;
                }

                // Make sure item has Retain24 template id
                if (!empty($itemData['product_config'][self::VOUCHER_TEMPLATE])) {
                    $issuedQty = $this->send_voucher($order, $itemData);
                } else {
                    Mage::throwException("Missing template id");
                }

                // Prepare comment for order with info about purchase
                if ($issuedQty > 0) {
                    $codeWord = $issuedQty > 1 ? "codes" : "code";
                    $sentBy = Mage::registry("retain24_sent_by") ?: "system";
                    $previousComment = Mage::registry("retain24_order_comment");
                    $newComment = "{$previousComment}User '{$sentBy}' successfully issued {$issuedQty} {$codeWord} for [{$itemData["product_config"]["sku"]}] {$itemData['product_config']["name"]}.<br>\n";
                    try {
                        Mage::register("retain24_order_comment", $newComment);
                    } catch (Exception $ex) {
                        Mage::unregister("retain24_order_comment");
                        Mage::register("retain24_order_comment", $newComment);
                    }
                }
            }
        }

    }

    /**
     * send_voucher (purchase) retain24 code.
     *
     * @param  Mage_Sales_Model_Order
     * @param  array, see getVoucherItemList()
     * @return int
     */
    public function send_voucher($order, $itemData)
    {

        // Check to see of we have previously assigned vouchers
        $voucherCollection = Mage::getModel("Crossroads_Retain24/voucher")->getCollection()->addItemFilter($itemData["item_id"]);
        $existingVoucherQty = count($voucherCollection);
        if ($existingVoucherQty >= $itemData["qty"]) {
            return 0;
        }

        // Make sure we only get the missing amount of vouchers
        $wantedQty = $itemData["qty"] - $existingVoucherQty;
        $issuedQty = 0;

        $store = $order->getStore();
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $billingAddressId = $order->getBillingAddressId();
        $shippingAddressId = $order->getShippingAddressId();
        if (empty($billingAddressId) && empty($shippingAddressId)) {
            // Allow order without shipping AND billing address? Don't think so!
            Mage::throwException("Missing both billing address and shipping address");
        } elseif (empty($billingAddressId) && !empty($shippingAddressId)) {
            // Free order get same billing address as shipping address.
            $shippingAddress = Mage::getModel('sales/order_address')->load($shippingAddressId);
            $billingAddress = $shippingAddress;
        } elseif (!empty($billingAddressId) && empty($shippingAddressId)) {
            // Virtual order get same shipping address as billing address.
            $billingAddress = Mage::getModel('sales/order_address')->load($billingAddressId);
            $shippingAddress = $billingAddress;
        } else {
            // Normal order with both billing and shipping address.
            $billingAddress = Mage::getModel('sales/order_address')->load($billingAddressId);
            $shippingAddress = Mage::getModel('sales/order_address')->load($shippingAddressId);
        }

        $xml = $this->init_xml();
        $xml->COUPON->addChild("SEND");
        $xml->COUPON->SEND->addChild("TEMPLATE", $itemData['product_config'][self::VOUCHER_TEMPLATE]);

        // QTY is only used for dynamic vouchers
        if ($itemData['product_config'][self::VOUCHER_DYNAMIC]) {
            $actualQty = $itemData['value'];
            if (!empty($itemData['product_config'][self::VOUCHER_MULTIPLIER])) {
                $actualQty *= $itemData['product_config'][self::VOUCHER_MULTIPLIER];
            }
            $xml->COUPON->SEND->addChild("QTY", $actualQty);
        }

        switch ($itemData['product_config'][self::VOUCHER_DISTRIBUTION]) {
            case self::DISTRIBUTION_SMS:
                $phone = $shippingAddress->getTelephone() ?: ($billingAddress->getTelephone() ?: 0);
                if (empty($phone)) {
                    Mage::throwException("Missing phone number");
                }
                $xml->COUPON->SEND->addChild("MSISDN", $this->formatPhone($store, $phone));
                $xml->COUPON->SEND->addChild("SMS_TEXT", "ToDo: set message");
                $xml->COUPON->SEND->addChild("SEND_DATE", "ToDo: set date");
                break;

            case self::DISTRIBUTION_EMAIL:
                $customerEmail = $order->getCustomerEmail();
                if (empty($customerEmail)) {
                    Mage::throwException("Missing email address");
                }
                $xml->COUPON->SEND->addChild("EMAIL_ADDRESS", $customerEmail);
                $xml->COUPON->SEND->addChild("EMAIL_TEXT", "ToDo: set message");
                $xml->COUPON->SEND->addChild("SEND_DATE", "ToDo: set date");
                break;

            case self::DISTRIBUTION_EXTERNAL:
                // No additional information needed
                break;

            case self::DISTRIBUTION_POST:
                $address = $shippingAddress->getStreet() ?:  ($billingAddress->getStreet() ?: []);
                if (!is_array($address)) { // Sometimes the street seems to be a string, not an array (should have 2 rows)
                    $address = [$address];
                }
                if (empty($address[0])) {
                    Mage::throwException("Missing address (street)");
                }
                $zipcode = $shippingAddress->getPostcode() ?: ($billingAddress->getPostcode() ?: 0);
                if (empty($zipcode)) {
                    Mage::throwException("Missing address (zipcode)");
                }

                $city = $shippingAddress->getCity() ?: ($billingAddress->getCity() ?: 0);
                if (empty($city)) {
                    Mage::throwException("Missing address (city)");
                }

                $countryCode = $shippingAddress->getCountryId() ?: ($billingAddress->getCountryId() ?: "SE");
                $countryName = $this->getCountryName($countryCode);
                if (empty($countryName)) {
                    Mage::throwException("Missing country name for: {$countryCode}");
                }
                $xml->COUPON->SEND->addChild("POSTADDRESS");
                $xml->COUPON->SEND->POSTADDRESS->addChild("DATA");
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("RECEIVE_DATE", "ToDo: set date");
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("FIRSTNAME", $order->getCustomerFirstname());
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("LASTNAME", $order->getCustomerLastname());
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("ADDRESS", $address[0]);
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("ADDRESS2", $address[1] ?? "");
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("ZIPCODE", $zipcode);
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("CITY", $city);
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("COUNTRY", $countryName);
                $xml->COUPON->SEND->addChild("SEND_DATE", "ToDo: set date");
                break;

            case self::DISTRIBUTION_PRINTOUT:
                // No additional information needed
                break;

        }

        $url = $this->get_url($store, self::SEND);
        for ($iterations = 0; $iterations < $wantedQty; $iterations++) {

            // Pause for a while if we are supposed to buy more than 1 voucher, so we don't trigger any hacking or DDOS filter
            if ($iterations > 0 && $wantedQty > 1) {
                usleep(100000);
            }

            $result = $this->post_data($url, $xml);
            Mage::log(
                sprintf(
                    "Retain24::SEND_VOUCHER(order: %s, item: %s, qty: %s of %s): %s",
                    $order->getIncrementId(),
                    $itemData['item_id'],
                    $iterations + 1,
                    $wantedQty,
                    json_encode($result, JSON_UNESCAPED_SLASHES)
                )
            );

            $xmlData = simplexml_load_string($result);
            if (!empty($xmlData->COUPON->RESPONSE->RECEIPT)) {
                $receipt = json_decode(json_encode($xmlData->COUPON->RESPONSE->RECEIPT), true);
                if (!empty($receipt['STATUS'])) {
                    if ($receipt['STATUS'] == "OK") {
                        $newVoucherId = $this->saveVoucher($itemData, $receipt);
                        if (!empty($itemData['product_config'][self::VOUCHER_IMAGE_TYPE])) {
                            $this->createCodeImage($newVoucherId, $itemData, $receipt);
                        }
                        $issuedQty++;
                    } else {
                        $message = "Status = {$receipt['STATUS']}";
                        $message .= !empty($receipt['MESSAGE']) ? ", message = {$receipt['MESSAGE']}" : "";
                        if ($test) {
                            file_put_contents($this->getXMLDebugLog(), $xml->asXML(), FILE_APPEND);
                        }
                        Mage::throwException($message);
                    }
                } else {
                    if ($test) {
                        file_put_contents($this->getXMLDebugLog(), $xml->asXML(), FILE_APPEND);
                    }
                    Mage::throwException('Empty status');
                }
            } else {
                if ($test) {
                    file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
                }
                Mage::throwException('Missing receipt');
            }
        }

        return $issuedQty;
    }

    /**
     * Get information about transactions of specified type.
     * Default date interval: from now to one month back.
     *
     * @param  Mage_Core_Store
     * @param  int, see this->cpntypes
     * @param  array
     * @param  boolean
     * @return void
     */
    public function transaction_report($store, $cpntype, $dateInterval = [], $useronly = true)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        $from = $dateInterval['from'] ?? date('Y-m-d 00:00', strtotime('-1 month'));
        $to = $dateInterval['to'] ?? date('Y-m-d H:i');

        if (!in_array($cpntype, $this->cpntypes)) {
            Mage::throwException("Unknown cpntype: {$cpntype}");
        }

        $xml = $this->init_xml();
        $xml->COUPON->addChild("TRANSACTION_REPORT");
        $xml->COUPON->TRANSACTION_REPORT->addChild("STARTDATE", $from);
        $xml->COUPON->TRANSACTION_REPORT->addChild("ENDDATE", $to);
        $xml->COUPON->TRANSACTION_REPORT->addChild("CPNTYPE", $cpntype);
        if ($useronly) {
            $xml->COUPON->TRANSACTION_REPORT->addChild("USERONLY", "1");
        }

        $result = $this->post_data($this->get_url($store, self::TRANSACTION_REPORT), $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->TRANSACTION_REPORT)) {
            $report = json_decode(json_encode($xmlData->COUPON->RESPONSE->TRANSACTION_REPORT), true);
            if (!empty($report['TRANSACTIONDATA'])) {
                return $report['TRANSACTIONDATA'];
            } else {
                return []; // No transactions found in given interval
            }
        }

        if ($test) {
            file_put_contents($this->getXMLDebugLog(), $result, FILE_APPEND);
        }
        Mage::throwException('Missing result: TRANSACTION_REPORT');

    }

    /**
     * Get data for all Retain24 items in specified order
     *
     * @param  Mage_Sales_Model_Order->id
     * @return array
     */
    public function getOrderData($orderId)
    {
        $sqlQuery = "
            SELECT
                o.entity_id,
                o.increment_id,
                o.state,
                o.`status`,
                p.method AS payment_method,
                i.product_id,
                e.sku
            FROM sales_flat_order o
            JOIN sales_flat_order_payment p ON p.parent_id = o.entity_id
            JOIN sales_flat_order_item i ON i.order_id = o.entity_id
            JOIN catalog_product_entity e ON e.entity_id = i.product_id
            JOIN catalog_product_entity_int ei ON ei.entity_id = e.entity_id AND ei.store_id = 0
            JOIN eav_attribute a ON a.attribute_id = ei.attribute_id
            WHERE o.entity_id = :entityId AND a.attribute_code = :attributeCode AND ei.`value` = :attributeValue
        ";
        $params = [
            'entityId' => $orderId,
            'attributeCode' => 'is_retain24_product',
            'attributeValue' => 1
        ];
        return Mage::getSingleton('core/resource')->getConnection('core_read')->fetchAll($sqlQuery, $params);
    }

    /**
     * Get item data prepared for voucher purchase
     *
     * @param  Mage_Sales_Model_Order
     * @return array
     */
    public function getVoucherItemList($order)
    {
        $orderId = $order->getId();
        if (!array_key_exists($orderId, $this->voucherItemListCache)) {

            $itemList = [];
            $parentList = [];
            $storeId = $order->getStoreId();
            $orderItems = $order->getItemsCollection()->getItems();

            // Let's find out if order contains any voucher products.
            foreach ($orderItems as $item) {
                $itemId = intval($item->getId());

                // Search for vouchers
                if ($item->getProductType() == 'virtual') {
                    $productId = intval($item->getProductId());
                    $product = Mage::getModel('catalog/product')->setStoreId($storeId)->load($productId);
                    if ($product->getData(self::VOUCHER_PRODUCT)) {

                        // Found voucher product
                        $parentItemId = intval($item->getParentItemId());
                        if (!empty($parentItemId)) {
                            // Product is child, save parent and look up data for it later
                            $itemList[$itemId]['parent_item_id'] = $parentItemId;
                            $parentList[$parentItemId] = $itemId;
                        } else {
                            $itemList[$itemId]['parent_item_id'] = null;
                        }

                        // Fetch all retain24 specific attribute values
                        $itemList[$itemId]['product_config'] = $this->getProductConfig($product);

                        // Set data as if voucher have fixed value, change later if not so
                        $itemList[$itemId]['item_id'] = $itemId;
                        $itemList[$itemId]['value'] = $itemList[$itemId]['product_config'][self::VOUCHER_VALUE];
                        $itemList[$itemId]['qty'] = intval($item->getQtyOrdered());
                        $itemList[$itemId]['sku'] = $product->getSku();
                        $itemList[$itemId]['name'] = $product->getName();
                    }
                }
            }

            // Search for parent product
            if (!empty($parentList)) {
                foreach ($orderItems as $item) {
                    $itemId = intval($item->getId());
                    if (array_key_exists($itemId, $parentList)) {
                        $childId = $parentList[$itemId];
                        $productId = $item->getProductId();
                        $product = Mage::getModel('catalog/product')->setStoreId($storeId)->load($productId);

                        // Check if parent is part of voucher
                        if ($product->getData(self::VOUCHER_PRODUCT)) {
                            // Check if voucher uses dynamic value AND have no predefined actual value
                            if ($itemList[$childId]['product_config'][self::VOUCHER_DYNAMIC] && empty($itemList[$itemId]['value'])) {
                                // If so: set 'value' to be child qty and 'qty' to be parent qty
                                $itemList[$childId]['value'] = $itemList[$childId]['qty'];
                                $itemList[$childId]['qty'] = intval($item->getQtyOrdered());
                            }
                        }
                    }
                }
            }

            $this->voucherItemListCache[$orderId] = $itemList;
        }

        return $this->voucherItemListCache[$orderId];
    }

    /**
     * Extract Retain24 specific product data
     *
     * @param  Mage_Catalog_Model_Product
     * @return array
     */
    public function getProductConfig($product)
    {
        $steps = $product->getData(self::VOUCHER_STEPS);
        $bcc = $product->getData(self::VOUCHER_EMAIL_BCC);
        $imgParams = $product->getData(self::VOUCHER_IMAGE_PARAMS);

        return [
            'product_id'                 => intval($product->getId()),
            'sku'                        => $product->getSku(),
            'name'                       => $product->getName(),
            self::VOUCHER_TEMPLATE       => $product->getData(self::VOUCHER_TEMPLATE),
            self::VOUCHER_DYNAMIC        => $product->getData(self::VOUCHER_DYNAMIC) ? true : false,
            self::VOUCHER_REQUIRE_PIN    => $product->getData(self::VOUCHER_REQUIRE_PIN) ? true : false,
            self::VOUCHER_MIN            => intval($product->getData(self::VOUCHER_MIN)),
            self::VOUCHER_MAX            => intval($product->getData(self::VOUCHER_MAX)),
            self::VOUCHER_VALUE          => intval($product->getData(self::VOUCHER_VALUE)),
            self::VOUCHER_STEPS          => empty($steps) ? [] : explode(',', $steps),
            self::VOUCHER_MULTIPLIER     => floatval($product->getData(self::VOUCHER_MULTIPLIER)),
            self::VOUCHER_VALID_TO       => $this->getValidTo($product->getData(self::VOUCHER_VALID_TO)),
            self::VOUCHER_SALES_END      => $product->getData(self::VOUCHER_SALES_END),
            self::VOUCHER_USAGE_START    => $product->getData(self::VOUCHER_USAGE_START),
            self::VOUCHER_DISTRIBUTION   => intval($product->getData(self::VOUCHER_DISTRIBUTION)),
            self::VOUCHER_EMAIL_TEMPLATE => $product->getData(self::VOUCHER_EMAIL_TEMPLATE),
            self::VOUCHER_EMAIL_BCC      => empty($bcc) ? [] : array_filter(array_map('trim', explode(',', $bcc))),
            self::VOUCHER_EMAIL_TEXT     => $product->getData(self::VOUCHER_EMAIL_TEXT),
            self::VOUCHER_EMAIL_TERMS     => $product->getData(self::VOUCHER_EMAIL_TERMS),
            self::VOUCHER_IMAGE_TYPE     => $product->getData(self::VOUCHER_IMAGE_TYPE),
            self::VOUCHER_IMAGE_PARAMS   => empty($imgParams) ? [] : json_decode($imgParams, true)
        ];
    }

    /**
     * Get country name from iso code
     *
     * @param  string
     * @return string
     */
    public function getCountryName($countryId)
    {
        try {
            $sqlQuery2 = "SELECT name FROM country_codes WHERE country_code = ?";
            return Mage::getSingleton('core/resource')->getConnection("integration_read")->fetchOne($sqlQuery2, $countryId);
        } catch (Exception $exception) {
            $this->logException($exception, 'Exception while getting country name!');
        }
    }

    /**
     * Calculate valid_to.
     *
     * @param  string|int (yyyy-mm-dd, yy-mm-dd or number of days from now)
     * @return string
     */
    public function getValidTo($value)
    {
        if (preg_match('/^[0-9]{2,4}-[0-9]{2}-[0-9]{2}$/', $value)) {
            $time = strtotime($value);
            return date('Y-m-d', $time);
        } elseif(is_numeric($value)) {
            $days = intval($value);
            $time = strtotime("+{$days} days");
            return date('Y-m-d', $time);
        }
        return null;
    }

    /**
     * Save voucher
     *
     * @param  array
     * @param  array
     * @return void
     */
    public function saveVoucher($itemData, $receipt)
    {
        $voucher = Mage::getModel("Crossroads_Retain24/voucher");
        $newData = [
            'item_id' => $itemData['item_id'],
            'voucher_template_id' => $itemData['product_config'][self::VOUCHER_TEMPLATE],
            'voucher_distribution' => $itemData['product_config'][self::VOUCHER_DISTRIBUTION],
            'voucher_value' => $itemData['value'],
            'voucher_id' => $receipt['ID'],
            'voucher_code' => $receipt['MULTICODE'],
            'voucher_pin' => $itemData['product_config'][self::VOUCHER_REQUIRE_PIN] ? (!empty($receipt['PIN']) ? $receipt['PIN'] : null) : null,
            'valid_to' => $itemData['product_config'][self::VOUCHER_VALID_TO]
        ];
        $voucher->addData($newData);
        $voucher->save();

        return $voucher->getId();

    }

    /**
     * Extract image information from product
     *
     * @param  int
     * @param  int
     * @return array
     */
    public function getImagesForEmail($productId, $storeId = 0)
    {
        $retval = array('top' => null, 'bottom' => null);
        $sqlQuery = "SELECT T1.value, T2.store_id, T2.label FROM catalog_product_entity_media_gallery T1 JOIN catalog_product_entity_media_gallery_value T2 ON T2.value_id = T1.value_id WHERE T1.entity_id = ? AND T2.label != ''";
        $images = Mage::getSingleton('core/resource')->getConnection('core_read')->fetchAll($sqlQuery, $productId);
        if ($images) {
            foreach ($images as $image) {
                $labelData = explode('_', $image['label']);
                if (!empty($labelData)) {
                    $label = $labelData[0];
                    if (array_key_exists($label, $retval)) {
                        if ($image['store_id'] == $storeId || ($image['store_id'] == 0 && empty($retval[$label]))) {
                            if ($label == 'top') {
                                $retval['background-color'] = empty($labelData[1]) ? 'ffffff' : $labelData[1];
                                $retval['border-color'] = empty($labelData[2]) ? $retval['background-color'] : $labelData[2];
                                $retval['font-color'] = empty($labelData[3]) ? '000000' : $labelData[3];
                            } elseif ($label == 'bottom') {
                                $retval['font-color'] = empty($labelData[1]) ? '000000' : $labelData[1];
                            }
                            $retval[$label] = Mage::getStoreConfig('web/unsecure/base_media_url', $storeId) . 'catalog/product' . $image['value'];
                        }
                    }
                }
            }
        }

        return $retval;
    }

    /**
     * Send email to customer, for all product in order containing voucher codes.
     *
     * @param  Mage_Sales_Model_Order
     * @return void
     */
    public function sendCustomerEmailForOrder($order)
    {
        $itemList = $this->getVoucherItemList($order);

        foreach($itemList as $itemData) {
            $this->sendCustomerEmailForItem($order, $itemData);
        }

    }

    /**
     * Send email to customer, for specified product containing voucher codes.
     *
     * https://stackoverflow.com/questions/22039156/cant-pass-variable-in-block-directive-from-transactional-email-template
     * {{block type="mymodule/sales_order_email_description" template="email/order/description.phtml" order=$order}}
     * http://excellencemagentoblog.com/blog/2011/11/25/magento-advanced-transactional-email-templates/
     * https://magento.stackexchange.com/questions/3769/pass-data-to-getchildhtml-or-call-method-on-child-block
     * https://magento.stackexchange.com/questions/141586/how-to-use-custom-variables-in-phtml-file-for-email
     *
     * @param  Mage_Sales_Model_Order
     * @param  array, see getVoucherItemList()
     * @return void
     */
    public function sendCustomerEmailForItem($order, $itemData, $codes = null)
    {
        $incrementId = $order->getIncrementId();
        $storeId = $order->getStoreId();

        $customerEmail = $order->getCustomerEmail();
        if (empty($customerEmail)) {
            $this->_sendAdminEmail('Retain24 order missing email address', "Order {$incrementId} is missing customer email address.");
            Mage::throwException("Order {$incrementId} is missing customer email address.");
        }

        $customerName = $order->getBillingAddress()->getName() ?: ($order->getShippingAddress()->getName() ?: 0);
        if (empty($customerName)) {
            $this->_sendAdminEmail('Retain24 order missing customer name', "Order {$incrementId} is missing customer name.");
            Mage::throwException("Order {$incrementId} is missing customer name.");
        }

        if (empty($codes)) {
            $codes = Mage::getModel("Crossroads_Retain24/voucher")->getCollection()->addItemFilter($itemData['item_id']);
        }

        $codeQty = count($codes);
        if (empty($codeQty)) {
            Mage::throwException("Order {$incrementId} is missing vouchers for product [{$itemData['sku']}].");
        }

        // Remove PIN from code(s) if template does not require it.
        if (!$itemData['product_config'][self::VOUCHER_REQUIRE_PIN]) {
            foreach ($codes as &$code) {
                $code['voucher_pin'] = null;
            }
        }

        $mailTemplate = $itemData['product_config'][self::VOUCHER_EMAIL_TEMPLATE];
        if (is_numeric($mailTemplate)) {
            $emailTemplate = Mage::getModel('core/email_template')->load($mailTemplate);
        } else {
            $emailTemplate = Mage::getModel('core/email_template')->loadDefault($mailTemplate);
        }
        $emailTemplate->setSenderName(Mage::getStoreConfig('trans_email/ident_sales/name', $storeId));
        $emailTemplate->setSenderEmail(Mage::getStoreConfig('trans_email/ident_sales/email', $storeId));

        if (!empty($itemData['product_config'][self::VOUCHER_EMAIL_BCC])) {
            $emailTemplate->addBcc($itemData['product_config'][self::VOUCHER_EMAIL_BCC]);
        }

        $emailTemplateVariables = [
            'order' => $order,
            'product' => Mage::getModel('catalog/product')->setStoreId($storeId)->load($itemData['product_config']['product_id']),
            'codes' => $codes,
            'store_name' => Mage::getStoreConfig('general/store_information/name', $storeId),
            self::VOUCHER_EMAIL_TEXT => $itemData['product_config'][self::VOUCHER_EMAIL_TEXT],
            self::VOUCHER_EMAIL_TERMS => $itemData['product_config'][self::VOUCHER_EMAIL_TERMS]
        ];

        $imageData = $this->getImagesForEmail($itemData['product_config']['product_id'], $storeId);
        if (!empty($imageData)) {
            $emailTemplateVariables['image_top'] = $imageData['top'] ?? '';
            $emailTemplateVariables['image_bottom'] = $imageData['bottom'] ?? '';
            $emailTemplateVariables['background-color'] = $imageData['background-color'] ?? '';
            $emailTemplateVariables['border-color'] = $imageData['border-color'] ?? '';
            $emailTemplateVariables['font-color'] = $imageData['font-color'] ?? '';
        }

        if (self::USE_EMAIL_QUEUE) {
            /** @var $emailQueue Mage_Core_Model_Email_Queue */
            $emailQueue = Mage::getModel('core/email_queue');
            $emailQueue->setEventType(self::EMAIL_EVENT_TYPE)
                ->setEntityType(Mage_Sales_Model_Order::ENTITY)
                ->setEntityId($order->getId())
                ->setIsForceCheck(false);
            $emailTemplate->setQueue($emailQueue);
        }

        $emailTemplate->send($customerEmail, $customerName, $emailTemplateVariables);

        // Mark each code as sent
        foreach($codes as $code) {
            $code->setData('sent_at', date('Y-m-d H:i:s'))->save();
        }

        // Prepare comment for order about this sending of code(s)
        $codeWord = $codeQty > 1 ? "codes" : "code";
        $sentBy = Mage::registry("retain24_sent_by") ?: "system";
        $previousComment = Mage::registry("retain24_order_comment");
        $newComment = "{$previousComment}User '{$sentBy}' successfully sent {$codeQty} {$codeWord} for [{$itemData["product_config"]["sku"]}] {$itemData["product_config"]["name"]}.<br>\n";
        try {
            Mage::register("retain24_order_comment", $newComment);
        } catch (Exception $ex) {
            Mage::unregister("retain24_order_comment");
            Mage::register("retain24_order_comment", $newComment);
        }
    }

    public function createCodeImage($voucherId, $itemData, $receipt)
    {
        $renderCode = $receipt['MULTICODE'];
        if (!empty($itemData['product_config'][self::VOUCHER_IMAGE_PARAMS]['code_prefix'])) {
            $renderCode = $itemData['product_config'][self::VOUCHER_IMAGE_PARAMS]['code_prefix'] . $renderCode;
        }
        if (!empty($itemData['product_config'][self::VOUCHER_IMAGE_PARAMS]['code_suffix'])) {
            $renderCode = $renderCode . $itemData['product_config'][self::VOUCHER_IMAGE_PARAMS]['code_suffix'];
        }

        switch ($itemData['product_config'][self::VOUCHER_IMAGE_TYPE]) {
            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_BARCODE:
                $this->createBARcode($voucherId, $renderCode);
                break;

            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_BARCODE128C:
                $this->createBARcode128c($voucherId, $renderCode);
                break;

            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_QRCODE:
                $this->createQRcode($voucherId, $renderCode);
                break;

            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_AZTECCODE:
                $this->createAZTECcode($voucherId, $renderCode);
                break;

            default:
                break;

        }

    }

    public function createBARcode($voucherId, $renderCode)
    {
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGFontFile.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGColor.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGDrawing.php');

        // Originally in the serialcodes module, SERIALCODES_IMAGE_BARCODE means BCGcode39
        // Not sure why we use BCGcode128 here?
        // require_once(Mage::getBaseDir("lib") . "/barcodegen.1d-php5.v5.1.0/class/BCGcode39.barcode.php");
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGcode128.barcode.php');

        try {

            // Loading Font
            $font = new BCGFontFile(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/font/Arial.ttf', 18);

            // The arguments are R, G, B for color.
            $color_black = new BCGColor(0, 0, 0);
            $color_white = new BCGColor(255, 255, 255);

            $drawException = null;
            try {
                $codeImage = new BCGcode128();
                $codeImage->setScale(2); // Resolution
                $codeImage->setThickness(30); // Thickness
                $codeImage->setForegroundColor($color_black); // Color of bars
                $codeImage->setBackgroundColor($color_white); // Color of spaces
                $codeImage->setFont($font); // Font (or 0)
                $codeImage->parse($renderCode); // Code
                $codeImage->clearLabels(); // Remove plain text code below barcode
            } catch (Exception $exception) {
                $drawException = $exception;
            }

            $newData = array(
                'voucher_id' => $voucherId,
                'image_hash' => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                'image_data' => null
            );

            $path = Mage::getBaseDir("media") . "/barcodes";
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = "{$path}/{$newData["image_hash"]}.png";

            $drawing = new BCGDrawing($fileName, $color_white);

            if ($drawException) {
                $drawing->drawException($drawException);
            } else {
                $drawing->setBarcode($codeImage);
                $drawing->draw();
            }

            $drawing->finish(BCGDrawing::IMG_FORMAT_PNG);
            $newData['image_data'] = file_get_contents($fileName);
            unlink($fileName);

            $voucherImage = Mage::getModel("Crossroads_Retain24/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();

        } catch (Exception $exception) {
            // Just silently ignore any error...
            Mage::logException($exception);
        }

        return null;
    }

    public function createBARcode128c($voucherId, $renderCode)
    {
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGFontFile.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGColor.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGDrawing.php');
        require_once(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/class/BCGcode128.barcode.php');

        try {

            // Loading Font
            $font = new BCGFontFile(Mage::getBaseDir('lib') . '/barcodegen.1d-php5.v5.1.0/font/Arial.ttf', 18);

            // The arguments are R, G, B for color.
            $color_black = new BCGColor(0, 0, 0);
            $color_white = new BCGColor(255, 255, 255);

            $drawException = null;
            try {
                $codeImage = new BCGcode128("C");
                $codeImage->setScale(2); // Resolution
                $codeImage->setThickness(30); // Thickness
                $codeImage->setForegroundColor($color_black); // Color of bars
                $codeImage->setBackgroundColor($color_white); // Color of spaces
                $codeImage->setFont($font); // Font (or 0)
                $codeImage->parse($renderCode); // Code
                $codeImage->clearLabels(); // Remove plain text code below barcode
            } catch (Exception $exception) {
                $drawException = $exception;
            }

            $newData = array(
                'voucher_id' => $voucherId,
                'image_hash' => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                'image_data' => null
            );

            $path = Mage::getBaseDir("media") . "/barcodes";
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = "{$path}/{$newData["image_hash"]}.png";

            $drawing = new BCGDrawing($fileName, $color_white);

            if ($drawException) {
                $drawing->drawException($drawException);
            } else {
                $drawing->setBarcode($codeImage);
                $drawing->draw();
            }

            $drawing->finish(BCGDrawing::IMG_FORMAT_PNG);
            $newData['image_data'] = file_get_contents($fileName);
            unlink($fileName);

            $voucherImage = Mage::getModel("Crossroads_Retain24/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();

        } catch (Exception $exception) {
            // Just silently ignore any error...
            Mage::logException($exception);
        }

        return null;
    }

    public function createQRcode($voucherId, $renderCode)
    {
        require_once(Mage::getBaseDir('lib') . '/phpqrcode/qrlib.php');

        try {
            $newData = array(
                'voucher_id' => $voucherId,
                'image_hash' => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                'image_data' => null
            );

            $path = Mage::getBaseDir("media") . "/barcodes";
            if (!is_dir($path)) {
                mkdir($path, 0777, true);
            }
            $fileName = "{$path}/{$newData["image_hash"]}.png";

            QRcode::png($renderCode, $fileName, QR_ECLEVEL_L, 8);
            $newData['image_data'] = file_get_contents($fileName);
            unlink($fileName);

            $voucherImage = Mage::getModel("Crossroads_Retain24/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();

        } catch (Exception $exception) {
            // Just silently ignore any error...
            Mage::logException($exception);
        }

        return null;
    }

    public function createAZTECcode($voucherId, $renderCode)
    {
        require_once Mage::getBaseDir('lib') . DS . 'Metzli' . DS . 'Metzli.php';
        $renderFactor = 8; // 8 = 152px x 152px when aztec code is compact

        try {
            $aztecCode = MetzliEncoder::encode($renderCode);
            $renderer = new MetzliPngRenderer($renderFactor);

            $newData = array(
                'voucher_id' => $voucherId,
                'image_hash' => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                'image_data' => $renderer->render($aztecCode)
            );

            $voucherImage = Mage::getModel("Crossroads_Retain24/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();

        } catch (Exception $exception) {
            // Just silently ignore any error...
            Mage::logException($exception);
        }

        return null;
    }

    public function groupBy($subject, $grouping, $glue = " ")
    {
        return implode($glue, str_split($subject, $grouping));
    }

    public function getRecaptchaSecret(Mage_Core_Model_Store $store): string {
        return (string)$store->getConfig(self::CONFIG_RECAPTCHA_SECRET);
    }

    public function getRecaptchaScoreThreshold(Mage_Core_Model_Store $store): float {
        return (float)$store->getConfig(self::CONFIG_RECAPTCHA_SCORE_THRESHOLD);
    }

    public function getEnabled(Mage_Core_Model_Store $store): bool {
        return (bool)$store->getConfig(self::CONFIG_ENABLED);
    }
}