<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class Crossroads_Retain24_Model_Schema extends MageQL_Core_Model_Schema_Abstract {
    const SUCCESS = "success";

    const VALIDATION_ERROR_CODES = [
        self::SUCCESS => [
            "description" => "Retain24 validation successfull"
        ],
        "_777" => [
            "description" => "Retain24 validation failed"
        ],
        "_779" => [
            "description" => "Retail valuable closed for use"
        ],
        "_780" => [
            "description" => "Retail valuable closed or cancelled"
        ],
        "_781" => [
            "description" => "No use left on the retail valuable"
        ],
        "_782" => [
            "description" => "The retail valuable cannot be used yet"
        ],
        "_783" => [
            "description" => "The expiry date of the retail valuable has passed"
        ],
        "_784" => [
            "description" => "No activated retail valuable"
        ],
        "_785" => [
            "description" => "Could not decode XML indata"
        ],
        "_786" => [
            "description" => "Unknown error with validation"
        ],
        "_789" => [
            "description" => "Retain24 code is for other currency"
        ],
        "_790" => [
            "description" => "User have no quote"
        ],
        "_791" => [
            "description" => "Code requires pin for validation"
        ],
        "_792" => [
            "description" => "Invalid pin"
        ]
    ];

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "Retain24Voucher":
            return $this->object("An issued voucher")
                ->setResolveField("GraphQL\Executor\Executor::defaultFieldResolver");

        case "Retain24Validation":
            return $this->object("Retain24 payment validation")
                ->setResolveField("GraphQL\Executor\Executor::defaultFieldResolver");

        case "Retain24ValidationResult":
            return $this->object("Retain24 payment validation result")
                ->setResolveField("GraphQL\Executor\Executor::defaultFieldResolver");

        case "Retain24StatusCode":
            return $this->enum("Status code from setQuoteRetain24Validation", self::VALIDATION_ERROR_CODES);

        default:
            return null;
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "OrderItem":
            return [
                "retain24vouchers" => $this->field("[Retain24Voucher!]", "List of voucher codes, if any.")
                    ->setResolver("Crossroads_Retain24_Model_Schema::resolveRetain24Vouchers"),
            ];

        case "Retain24Voucher":
            return [
                "code" => $this->field("String!", "Voucher code"),
                "pin" => $this->field("String", "PIN code, if any"),
                "value" => $this->field("Float!", "Voucher value"),
                "image_url" => $this->field("String", "URL to voucher code as image (barcode, QRcode, etc), if any"),
                "expiryDate" => $this->field("String", "Expiry date, if any")->setResolver("Crossroads_Retain24_Model_Schema::resolveDate"),
            ];

        case "Quote":
            return [
                "retain24validation" => $this->field("Retain24Validation", "Retain24 payment validation")
                    ->setResolver("Crossroads_Retain24_Model_Schema::resolveRetain24Validation"),
            ];

        case "Retain24Validation":
            return [
                "isPhone" => $this->field("Boolean!", "Is code a phone number"),
                "code" => $this->field("String!", "Voucher code"),
                "pin" => $this->field("String", "PIN code, if any"),
                "value" => $this->field("Float!", "Voucher value"),
                "requiresPin" => $this->field("Boolean!", "Is pin required?"),
                "pinVerified" => $this->field("Boolean!", "Is pin verified?"),
            ];

        case "Query":
            return [
                "validateRetain24Code" => $this->field("Retain24Validation", "Retain24 payment validation")
                    ->addArgument("isPhone", $this->argument("Boolean!", "Is code a phone number"))
                    ->addArgument("code", $this->argument("String!", "Retain24 voucher code"))
                    ->addArgument("pin", $this->argument("String", "Retain24 voucher pin"))
                    ->setResolver("Crossroads_Retain24_Model_Schema::resolveValidateRetain24Code"),
            ];

        case "Mutation":
            return [
                "setQuoteRetain24Validation" => $this->field("Retain24ValidationResult!", "Retain24 payment validation result")
                    ->addArgument("isPhone", $this->argument("Boolean!", "Is code a phone number"))
                    ->addArgument("code", $this->argument("String!", "Retain24 voucher code"))
                    ->addArgument("pin", $this->argument("String", "Retain24 voucher pin"))
                    ->setResolver("Crossroads_Retain24_Model_Schema::mutateCreatePaymentValidation"),
                "removeQuoteRetain24Validation" => $this->field("Retain24ValidationResult!", "Retain24 payment validation result")
                    ->setResolver("Crossroads_Retain24_Model_Schema::mutateRemovePaymentValidation"),
            ];

        case "Retain24ValidationResult":
            return [
                "result" => $this->field("Retain24StatusCode!", "Retain24 status code"),
                "Retain24Validation" => $this->field("Retain24Validation", "Retain24 payment validation"),
            ];

        default:
            return [];
        }
    }

    public static function resolveDate($param): ?string {
        if (!empty($param['expiryDate'])) {
            $time = strtotime($param['expiryDate']);
            if (!empty($time)) {
                return date('Y-m-d', $time);
            }
        }
        return null;
    }

    /**
     * Load all vouchers for specified item
     *
     * @param Mage_Sales_Model_Order_Item
     * @return ?Array<array{ code: string, pin: ?string, value: float, image_url: ?string, expiryDate: ?string }>
     */
    public static function resolveRetain24Vouchers(Mage_Sales_Model_Order_Item $item): ?array {
        $vouchers = null;

        $collection = Mage::getModel('Crossroads_Retain24/voucher')->getCollection()->addItemFilter($item->getId());
        $url = Crossroads_Retain24_Helper_Data::RETAIN24_IMAGE_URL;

        if (!empty($collection) && count($collection) > 0) {
            $vouchers = [];
            foreach ($collection as $voucher) {

                // How do i map this to Retain24Voucher in getTypeFields() ?
                $vouchers[] = [
                    "code" => $voucher->getVoucherCode(),
                    "pin" => $voucher->getVoucherPin(),
                    "value" => $voucher->getVoucherValue(),
                    "image_url" => $voucher->getImageHash() ? sprintf("%s%s%s.png", $item->getOrder()->getStore()->getBaseUrl(), $url, $voucher->getImageHash()) : null,
                    "expiryDate" => $voucher->getValidTo()
                ];
            }
        }

        return $vouchers;
    }

    /**
     * Load validation for quote
     *
     * @param Mage_Sales_Model_Quote
     * @return ?Array<array{ isPhone: boolean, code: string, pin: ?string, value: float }>
     */
    public static function resolveRetain24Validation(Mage_Sales_Model_Quote $quote): ?array {
        $validation = Mage::getModel('Crossroads_Retain24/validation')->load($quote->getId());
        if (!$validation || !$validation->getId()) {
            return null;
        }
        return [
            "isPhone" => $validation->getIsPhone() ? true : false,
            "code" => $validation->getCode(),
            "pin" => $validation->getPin(),
            "value" => floatval($validation->getQty() * $validation->getValue()),
        ];
    }

    /**
     * Load validation for code
     *
     * @return ?Array<array{ isPhone: boolean, code: string, pin: ?string, value: float }>
     */
    public static function resolveValidateRetain24Code($src, array $args, $ctx) {
        $validation = Mage::helper("Retain24")->validate(
            $ctx->getStore(),
            $args["code"],
            $args["isPhone"],
            $args["pin"] ?? null
        );
        if (!empty($validation)) {
            return [
                "isPhone" => $args["isPhone"] ? true : false,
                "code" => $args["code"],
                "pin" => $args["pin"] ?? null,
                "value" => floatval($validation["QTY"] * $validation["VALUE"]),
            ];
        }

        return null;
    }

    /**
     * Add validation to quote
     *
     * @return ?Array<array{ isPhone: boolean, code: string, pin: ?string, value: float }>
     */
    public static function mutateCreatePaymentValidation($src, array $args, $ctx) {
        $quote = Mage::getSingleton("mageql_sales/quote")->getQuote();
        $quoteId = $quote->getEntityId();

        if (empty($quoteId)) {
            $quote->collectTotals()->save();
            $quoteId = $quote->getEntityId();

            if (empty($quoteId)) {
                return [
                    "result" => "_790",
                    "Retain24Validation" => null
                ];
            }

            Mage::getSingleton('checkout/session')->setQuoteId($quoteId);
        }

        try {
            $observer = New Varien_Event_Observer();
            $observer->setQuote($quote);
            $observer->setParams(new Varien_Object);
            $observer->getParams()->setRetain24([
                "isPhone" => $args["isPhone"] ? true : false,
                "code" => $args["code"],
                "pin" => $args["pin"] ?? null
            ]);

            $model = Mage::getModel("Crossroads_Retain24/Observer");
            $model->checkoutPostMerge($observer);

            $validation = Mage::getModel("Crossroads_Retain24/validation")->load($quoteId);
            if ($validation && $validation->getValidationId()) {
                $retain24ValidationResult = [
                    "result" => self::SUCCESS,
                    "Retain24Validation" => [
                        "isPhone" => $validation->getIsPhone() ? true : false,
                        "code" => $validation->getCode(),
                        "pin" => $validation->getPin(),
                        "value" => floatval($validation->getQty() * $validation->getValue()),
                        "requiresPin" => $validation->getRequiresPin() ? true : false,
                        "pinVerified" => $validation->getPinVerified() ? true : false,
                    ]
                ];

                Mage::dispatchEvent(
                    'crossroads_retain24_create_payment_validation_success',
                    ['quote' => $quote]
                );

                $quote->collectTotals()->save();

            } else {
                $retain24ValidationResult = [
                    "result" => "_786",
                    "Retain24Validation" => null
                ];
            }

        } catch(Crossroads_API_ResponseException $e) {
            Mage::logException($e);
            $code = "_{$e->getCode()}";
            if (!array_key_exists($code, self::VALIDATION_ERROR_CODES)) {
                throw $e;
            }
            $retain24ValidationResult = [
                "result" => $code,
                "Retain24Validation" => null
            ];
        }

        return $retain24ValidationResult;
    }

    /**
     * Remove validation from quote
     *
     * @return ?Array<array{ isPhone: boolean, code: string, pin: ?string, value: float }>
     */
    public static function mutateRemovePaymentValidation($src, array $args, $ctx) {
        // ToDo: make this work
        return [
            "result" => self::SUCCESS,
            "Retain24Validation" => null
        ];
    }

}
