<?php

use \Klarna\XMLRPC\Flags;

class Crossroads_Retain24_Model_Observer extends Mage_Core_Model_Abstract {

    /**
     * Observes `crossroads_api_quote_post_data_prepare` to add retain24 fields.
     *
     * @param  Varien_Event_Observer
     */
    public function quotePostDataPrepare(Varien_Event_Observer $observer)
    {
        $quote = $observer->getQuote();
        $totals = $observer->getTotals();
        $data = $observer->getPreparedData();

        $summary = $data->getData("summary");

        $summary["retain24"] = array_key_exists("retain24", $totals) ? round($totals["retain24"]->getValueInclTax()) : 0;

        $data->setData("summary", $summary);
    }

    public function checkoutPostDataPrepare(Varien_Event_Observer $observer)
    {
        $quote = $observer->getQuote();
        $data = $observer->getPreparedData();
        $model = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());

        if ($model->getCode()) {
            $data->setData("retain24", [
                "name" => $model->getName(),
                "value" => round(($model->getValue() * $model->getQty())),
                "code" => $model->getCode(),
                "codeType" => $model->codeType(),
                "requiresPin" => (bool) $model->getRequiresPin(),
                "pinVerified" => (bool) $model->getPinVerified(),
                "isPhone" => (bool) $model->getIsPhone(),
            ]);
        } else {
            $data->setData("retain24", null);
        }
    }

    public function checkoutPostMerge(Varien_Event_Observer $observer)
    {
        $quote = $observer->getQuote();
        $data = $observer->getParams();
        $store = $quote->getStore();
        $helper = Mage::helper("Retain24");

        if (!$data->getData("retain24")) {
            return;
        }

        if ($data->getData("retain24") === null) {
            return $this->removeCode($quote);
        }

        if (!is_array($data->getData("retain24"))) {
            throw Crossroads_API_ResponseException::create(400, "Key 'retain24' must be an object or null", null, 778);
        }

        $retain24 = $data->getData("retain24");

        if (array_key_exists("code", $retain24) && $retain24["code"]) {
            $model = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());
            $code = trim($retain24["code"]);
            $isPhone = array_key_exists("isPhone", $retain24) ? (bool) $retain24["isPhone"] : false;

            if ($code != $model->getCode() || !$model->getValidationId() ||
                    ($isPhone != $model->getIsPhone()) ||
                    ($isPhone || $model->getRequiresPin()) &&
                    (array_key_exists("pin", $retain24) && $retain24["pin"] !== $model->getPin())) {
                $quote->setRetain24Code($code);
                $quote->setRetain24Pin(array_key_exists("pin", $retain24) ? $retain24["pin"] : null);

                $code_data = $helper->validate($store, $code, $isPhone, array_key_exists("pin", $retain24) ? $retain24["pin"] : null);

                if (!$code_data) {
                    $model->delete();

                    throw Crossroads_API_ResponseException::create(400, "Retain24 validation failed", null, 777);
                }

                if ($code_data["ERRORCODE"] != 0) {
                    switch ($code_data["ERRORCODE"]) {
                        case 25001:
                            throw Crossroads_API_ResponseException::create(400, "Retail valuable closed for use", null, 779);
                            break;

                        case 25002:
                            throw Crossroads_API_ResponseException::create(400, "Retail valuable closed or cancelled", null, 780);
                            break;

                        case 25003:
                            throw Crossroads_API_ResponseException::create(400, "No use left on the retail valuable", null, 781);
                            break;

                        case 25004:
                            throw Crossroads_API_ResponseException::create(400, "The retail valuable cannot be used yet", null, 782);
                            break;

                        case 25005:
                            throw Crossroads_API_ResponseException::create(400, "The expiry date of the retail valuable has passed", null, 783);
                            break;

                        case 25070:
                            throw Crossroads_API_ResponseException::create(400, "No activated retail valuable", null, 784);
                            break;

                        case 30001:
                            throw Crossroads_API_ResponseException::create(400, "Could not decode XML indata", null, 785);
                            break;

                        default:
                            throw Crossroads_API_ResponseException::create(400, "Unknown error with validation", null, 786);
                    }
                }

                if ($code_data["CURRENCY"] !== $quote->getBaseCurrencyCode()) {
                    throw Crossroads_API_ResponseException::create(400, sprintf("Retain24 code is for the currency '%s', we have '%s'.", $code_data["CURRENCY"], $quote->getBaseCurrencyCode()), null, 789);
                }

                $model->addData([
                    "quote_id" => $quote->getEntityId(),
                    "validation_id" => $code_data["ID"],
                    "code" => $code,
                    "code_type" => $code_data["CPNTYPE"],
                    "code_status" => $code_data["STATUS"],
                    "name" => $code_data["NAME"],
                    "qty" => $code_data["QTY"],
                    "value" => $code_data["VALUE"],
                    "currency" => $code_data["CURRENCY"],
                    "requires_pin" => $code_data["REQUIRES_PIN"] === "YES",
                    "description" => $code_data["DESCRIPTION"],
                    "qty_max" => $code_data["QTYMAX"],
                    "pin_verified" => $code_data["PIN_VERIFICATION"] === "OK",
                    "pin" => array_key_exists("pin", $retain24) ? $retain24["pin"] : null,
                    "is_phone" => $isPhone,
                ])->save();
            }
        } elseif (array_key_exists("code", $retain24) && $retain24["code"] === null) {
            return $this->removeCode($quote);
        }
    }

    protected function removeCode($quote)
    {
        $quote->unsRetain24Code();
        $quote->unsRetain24Pin();

        Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId())->delete();
    }

    /**
     * Observer listening to `sales_order_invoice_save_after` to update deposit amount invoiced in order.
     *
     * @param  Varien_Event_Observer
     */
    public function invoiceSaveAfter(Varien_Event_Observer $observer)
    {
        $invoice = $observer->getInvoice();
        $order = $invoice->getOrder();

        if ($invoice->getBaseRetain24Amount()) {
            $order->setRetain24AmountInvoiced($order->getRetain24AmountInvoiced() + $invoice->getRetain24Amount());
            $order->setBaseRetain24AmountInvoiced($order->getBaseRetain24AmountInvoiced() + $invoice->getBaseRetain24Amount());
        }

        return $this;
    }

    /**
     * Observer listening to `sales_order_creditmemo_save_after` to update deposit amount refunded in order.
     *
     * @param  Varien_Event_Observer
     */
    public function creditmemoSaveAfter(Varien_Event_Observer $observer)
    {
        $creditmemo = $observer->getCreditmemo();
        $order = $creditmemo->getOrder();

        if ($creditmemo->getBaseRetain24Amount()) {
            $order->setRetain24AmountRefunded($order->getRetain24AmountRefunded() + $creditmemo->getRetain24Amount());
            $order->setBaseRetain24AmountRefunded($order->getBaseRetain24AmountRefunded() + $creditmemo->getBaseRetain24Amount());
        }

        return $this;
    }

    // Validate and reserve amount if validation succeeds.
    public function quoteSubmitBefore(Varien_Event_Observer $observer)
    {
        $quote = $observer->getQuote();
        $store = $quote->getStore();
        $helper = Mage::helper("Retain24");
        $validation = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());
        $reservation = Mage::getModel("Crossroads_Retain24/reservation")->load($quote->getEntityId());

        if ($validation->getValidationId()) {
            $amount = 0;

            foreach ($quote->getAddressesCollection() as $a) {
                $amount += $a->getBaseRetain24Amount();
            }

            $qty = round(abs($amount / $validation->getValue()));

            if ($reservation->getReferenceId() && $reservation->getStatus() == 1) {
                // We have a previous reservation, cancel it and replace it
                $this->cancelReservation($store, $reservation);
            }

            $res = $helper->reserve_instance($store, $validation->getValidationId(), $qty, $validation->getPin());

            if ($res["INFO"]["DATA"]["ERRORCODE"] != 0) {
                switch ($res["INFO"]["DATA"]["ERRORCODE"]) {
                    case 25200:
                        throw Crossroads_API_ResponseException::create(400, "Reservation failed", null, 787);
                        break;

                    default:
                        throw Crossroads_API_ResponseException::create(400, "Unknown error with reservation", null, 788);
                }
            }

            $reservation->addData(array(
                "quote_id" => $quote->getEntityId(),
                "reference_id" => $res["REFERENCE"],
                "expiry_date" => $res["EXPIRES"],
                "reserved_qty" => $qty,
                "validation_id" => $res["INFO"]["DATA"]["ID"],
                "status" => 1,
            ))->save();
        }
    }

    // Redeem or cancel reservation based upon status.
    public function orderStatusChange(Varien_Event_Observer $observer)
    {
        $order = $observer->getEvent()->getOrder();
        $store = $order->getStore();

        $validation = Mage::getModel("Crossroads_Retain24/validation")->load($order->getQuoteId());
        $reservation = Mage::getModel("Crossroads_Retain24/reservation")->load($order->getQuoteId());

        if ($order->getStatus() == "processing" && $reservation->getStatus() == 1) { // Reserved status only
            $this->redeemReservation($store, $validation, $reservation);
        } elseif ($order->getStatus() == "canceled" && $reservation->getStatus() == 1) { // Reserved status only
            $this->cancelReservation($store, $reservation);
        }
    }

    // Redeem reserved amount when orderstatus changes to processing.
    private function redeemReservation($store, $validation, $reservation)
    {
        $redeem = Mage::helper("Retain24")->redeem_reservation($store, $reservation->getReferenceId(), $validation->getValidationId(), $reservation->getReservedQty());

        if ($redeem["STATUS"] == "OK") {
            $reservation->setStatus(2)->save();
        } else {
            $reservation->setStatus(99)->setStatusMessage($redeem["MESSAGE"])->save();
        }
    }

    // Cancel reservation when orderstatus changes to cancelled.
    private function cancelReservation($store, $reservation)
    {
        $cancel = Mage::helper("Retain24")->cancel_reservation($store, $reservation->getReferenceId());

        if ($cancel["STATUS"] == "OK") {
            $reservation->setStatus(3)->save();
        } else {
            $reservation->setStatus(99)->setStatusMessage($cancel["MESSAGE"])->save();
        }
    }

    public function klarnaReserveAmount(Varien_Event_Observer $event)
    {
        $order = $event->getOrder();
        $klarna = $event->getKlarna();

        if (abs($order->getBaseRetain24Amount()) > 0) {
            $total = $order->getBaseGrandTotal() - $order->getBaseRetain24Amount();
            $totalNoTax = $total - $order->getTaxAmount();
            $vatPercent = $totalNoTax > 0 ? ($total / $totalNoTax - 1) * 100 : 0;

            $klarna->addArticle(
                    1, "Retain24", Mage::helper("Retain24")->__("Retain24"),
                    // BaseRetain24Amount is negative, so we remove that from the total
                    $order->getBaseRetain24Amount(), $vatPercent, 0, Flags::INC_VAT
            );
        }
    }

    public function issueVoucherEvent(Varien_Event_Observer $event)
    {
        $order = $event->getOrder();
        if(empty($order)) {
            return;
        }

        $orderId = $order->getId();
        if(empty($orderId)) {
            return;
        }

        $orderData = Mage::helper("Retain24")->getOrderData($orderId);
        // Only process orders that actually contain Retain24 products
        if (empty($orderData)) {
            return;
        }
        
        // Make sure we only issue vouchers for orders with correct status.
        if (!in_array($order->getStatus(), Crossroads_Retain24_Helper_Data::$issueVoucherOrderStatuses)) {
            return;
        }

        $storeId = $order->getStoreId();
        // If issueVoucherEvent() is triggered by script or other method, make sure we use same store context as order.
        $emulate = $storeId != Mage::app()->getStore()->getId();
        Mage::register('retain24_order_comment', "");

        // retain24_sent_by can be set by script or other method.
        // If not, try to figure out apropriate username
        $username = Mage::registry('retain24_sent_by');
        if (empty($username)) {
            // Normally issueVoucherEvent() is triggered by customer
            $username = 'customer';
            $adminSession = Mage::getSingleton('admin/session');
            if ($adminSession) {
                $adminUser = $adminSession->getUser();
                if ($adminUser) {
                    $username = $adminUser->getUsername();
                }
            }
            Mage::register('retain24_sent_by', $username);
        }
        
        // We need this to fetch correct data for emails
        if ($emulate) {
            $appEmulation = Mage::getSingleton('core/app_emulation');
            $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);
        }

        try {
            Mage::helper("Retain24")->purchase_voucher($order);
            Mage::helper("Retain24")->sendCustomerEmailForOrder($order);
        } catch (Exception $ex) {
            Mage::logException($ex);
            // ToDo: Perhaps send email about this?
        }

        $comment = Mage::registry('retain24_order_comment');
        if (!empty($comment)) {
            $history = $order->addStatusHistoryComment($comment);
            $history->setIsCustomerNotified(Mage_Sales_Model_Order_Status_History::CUSTOMER_NOTIFICATION_NOT_APPLICABLE);
            $order->save();
        }

        if ($emulate) {
            $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
        }
    }

}
