<?php

class Crossroads_Retain24_Helper_Data extends Mage_Core_Helper_Abstract
{
    const CONFIG_TEST           = "retain24/general/test";
    const CONFIG_COUNTRY_NUMBER = "retain24/general/country_number";
    const CONFIG_SSL_KEY_NAME   = "retain24/general/ssl_key_name";

    // Email setup
    const USE_EMAIL_QUEUE = true; // Magento >= 1.9 uses this
    const EMAIL_EVENT_TYPE = 'retain24';

    // URL's for the Retain24 endpoint.
    const BASE_URL_TEST        = "https://pilot.mvoucher.se/ta/servlet/TA_XML?TA_ACTION=";
    const BASE_URL_LIVE        = "https://live.mvoucher.se/ta/servlet/TA_XML?TA_ACTION=";
    const TEMPLATELIST         = "5-45103";
    const VALIDATE             = "5-43101";
    const RESERVE_INSTANCE     = "5-43109";
    const REDEEM_RESERVATION   = "5-43110";
    const VALIDATE_RESERVATION = "5-43112";
    const CANCEL_RESERVATION   = "5-43111";
    const SEND                 = "5-45102";
    const TRANSACTION_REPORT   = "5-49081";

    // Product attributes
    const VOUCHER_PRODUCT        = "is_retain24_product";
    const VOUCHER_TEMPLATE       = "retain24_template_id";
    const VOUCHER_DYNAMIC        = "retain24_dynamic";
    const VOUCHER_REQUIRE_PIN    = "retain24_require_pin";
    const VOUCHER_MIN            = "retain24_min_value";
    const VOUCHER_MAX            = "retain24_max_value";
    const VOUCHER_VALUE          = "retain24_value";
    const VOUCHER_STEPS          = "retain24_value_steps";
    const VOUCHER_MULTIPLIER     = "retain24_value_multiplier";
    const VOUCHER_VALID_TO       = "retain24_valid_to";
    const VOUCHER_SALES_END      = "retain24_sales_end";
    const VOUCHER_USAGE_START    = "retain24_usage_start";
    const VOUCHER_DISTRIBUTION   = "retain24_distribution";
    const VOUCHER_EMAIL_TEMPLATE = "retain24_email_template";
    const VOUCHER_EMAIL_BCC      = "retain24_email_bcc";

    // Retain24 ssl certificates
    const CRT_FILE = MAGENTO_ROOT.DS.'app'.DS.'etc'.DS.'retain24'.DS."test_crt.pem";
    const KEY_FILE = MAGENTO_ROOT.DS.'app'.DS.'etc'.DS.'retain24'.DS."test_key.pem";

    // Retain24 distribution methods (CHANNEL_CODE)
    const DISTRIBUTION_SMS      = 1;
    const DISTRIBUTION_EMAIL    = 2;
    const DISTRIBUTION_EXTERNAL = 4;
    const DISTRIBUTION_POST     = 5;
    const DISTRIBUTION_PRINTOUT = 6;

    protected $voucherItemListCache = [];
    protected $currentStoreId = null;

    public static $issueVoucherOrderStatuses = [
        Mage_Sales_Model_Order::STATE_PROCESSING,
        Mage_Sales_Model_Order::STATE_COMPLETE
    ];

    // Endpoint URL.
    protected function get_url($store, $url)
    {
        // Save store id for later use
        $storeId = $store->getId();
        if ($this->currentStoreId != $storeId) {
            $this->currentStoreId = $storeId;
        }

        if($store->getConfig(self::CONFIG_TEST)) {
            return self::BASE_URL_TEST.rawurlencode($url);
        }

        return self::BASE_URL_LIVE . rawurlencode($url);
    }

    // Initialize XML request with common data.
    protected function init_xml()
    {
        $xml = new SimpleXMLElement('<?xml version="1.0" encoding="ISO-8859-1"?><TICKETANYWHERE></TICKETANYWHERE>');
        $coupon = $xml->addChild('COUPON');
        $coupon->addAttribute('VER', '1.0');

        return $xml;
    }

    // Post data to Retain24
    protected function post_data($url, $xml)
    {
        // Find out which ssl key to use
        $crtFile = self::CRT_FILE;
        $keyFile = self::KEY_FILE;

        if ($this->currentStoreId !== null) {
            $keyName = Mage::getStoreConfig(self::CONFIG_SSL_KEY_NAME, $this->currentStoreId);
            if (!empty($keyName)) {
                $tmpFile1 = DS.'etc'.DS.'ssl'.DS.'retain24'.DS.$keyName."_crt.pem";
                $tmpFile2 = DS.'etc'.DS.'ssl'.DS.'retain24'.DS.$keyName."_key.pem";
                if (file_exists($tmpFile1) && file_exists($tmpFile2)) {
                    $crtFile = $tmpFile1;
                    $keyFile = $tmpFile2;
                }
            }
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_VERBOSE, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_SSLKEY, $keyFile);
        curl_setopt($ch, CURLOPT_SSLCERT, $crtFile);
        curl_setopt($ch, CURLOPT_SSLCERTTYPE, 'PEM');
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $xml->asXML());
        $response = curl_exec($ch);

        if(curl_errno($ch)) {
            echo 'Curl error: ' . curl_error($ch);
            die;
        }

        return $response;
    }

    public function formatPhone($store, $phone)
    {
        $phone = preg_replace("/[^0-9+]/", "", $phone);

        if(strlen($phone) > 0 && $phone[0] === '+') {
            return "00".substr($phone, 1);
        }
        else if(substr($phone, 0, 2) === "00") {
            return $phone;
        }
        else if(strlen($phone) > 0 && $phone[0] === '0') {
            $phone = substr($phone, 1);
        }

        return "00".$store->getConfig(self::CONFIG_COUNTRY_NUMBER).$phone;
    }

    // List the various templates we can use. We're not using this.
    // Merely here because it was mentioned in the documentation.
    public function templatelist($store)
    {
        $url = $this->get_url($store, self::TEMPLATELIST);
        $xml = $this->init_xml();

        $xml->COUPON->addChild('TEMPLATELIST');

        Mage::log("Retain24::TEMPLATELIST");

        $result = $this->post_data($url, $xml);

        return simplexml_load_string($result);
    }

    /**
     * Validate retain24 code.
     *
     * @param  Mage_Core_Store
     * @param  string
     * @param  boolean
     * @param  string|null
     * @return array
     */
    public function validate($store, $code, $isPhone, $pin = null)
    {
        $xml = $this->init_xml();

        $phone    = $this->formatPhone($store, $code);
        $validate = $xml->COUPON->addChild('VALIDATE');
        $validate->addAttribute('TYPE', 'STANDARD');
        if($isPhone) {
            $validate->addChild('MSISDN', $phone);
        }
        else {
            $validate->addChild('MULTICODE', $code);
        }

        if($pin) {
            $validate->addChild('PIN', $pin);
        }

        $result = $this->post_data($this->get_url($store, self::VALIDATE), $xml);

        $data = json_decode(json_encode(simplexml_load_string($result)->COUPON->RESPONSE), true); // Fult men fungerar

        Mage::log(sprintf("Retain24::VALIDATE(msisdn: %s, multicode: %s): %s", $phone, $code, json_encode($data, JSON_UNESCAPED_SLASHES)));

        if(empty($data)) {
            return;
        }

        if (empty($data["CPNINFO"]["STATUS"])) {
            foreach ($data["CPNINFO"] as $d) {
                if (in_array($d["STATUS"], array(4, 6, 7, 10))) {
                    $return_data = array();
                    $return_data = $d;
                    $return_data["ID"] = $d["@attributes"]["ID"];
                    unset($return_data["@attributes"]);

                    return $return_data;
                }
            }

            return;
        } else {
            $return_data = array();
            $return_data = $data["CPNINFO"];
            $return_data["ID"] = $data["CPNINFO"]["@attributes"]["ID"];
            unset($return_data["@attributes"]);

            return $return_data;
        }
    }

    // Reserve specific amount (qty) on a retain24 code.
    public function reserve_instance($store, $id, $qty, $pin_code)
    {
        $xml = $this->init_xml();

        $reserve_instance = $xml->COUPON->addChild('RESERVE_INSTANCE');
        $cpnselection = $reserve_instance->addChild('CPNSELECTION');
        $cpnselection->addChild('ID', $id);
        $cpnselection->addChild('QTY', $qty);
        $cpnselection->addChild('PINCODE', $pin_code);

        $result = $this->post_data($this->get_url($store, self::RESERVE_INSTANCE), $xml);

        $data = json_decode(json_encode(simplexml_load_string($result)->COUPON->RESPONSE), true); // Fult men fungerar

        Mage::log(sprintf("Retain24::RESERVE_INSTANCE(id: %s, qty: %s): %s", $id, $qty, json_encode($data, JSON_UNESCAPED_SLASHES)));

        return $data["RESERVATION"];
    }

    // Redeem reservation based on values from validation and reservation. Also takes amount (qty).
    public function redeem_reservation($store, $reference_id, $cpnselection_id, $qty)
    {
        $xml = $this->init_xml();

        $redeem_reservation = $xml->COUPON->addChild('REDEEM_RESERVATION');
        $reference = $redeem_reservation->addChild('REFERENCE');
        $reference->addAttribute("ID", $reference_id);
        $cpnselection = $reference->addChild('CPNSELECTION');
        $cpnselection->addAttribute("ID", $cpnselection_id);
        $cpnselection->addChild('QTY', $qty);

        $result = $this->post_data($this->get_url($store, self::REDEEM_RESERVATION), $xml);

        $data = json_decode(json_encode(simplexml_load_string($result)->COUPON->RESPONSE), true); // Fult men fungerar

        $return_data = $data["RESERVATION"];
        $return_data["REFERENCE"] = $data["RESERVATION"]["@attributes"]["REFERENCE"];
        unset($return_data["@attributes"]);

        return $return_data;
    }

    // Validate reservation. Based on value from reservation.
    public function validate_reservation($store, $reference_id)
    {
        $xml = $this->init_xml();

        $validate_reservation = $xml->COUPON->addChild('VALIDATE_RESERVATION');
        $reference = $validate_reservation->addChild('REFERENCE', $reference_id);

        $result = $this->post_data($this->get_url($store, self::VALIDATE_RESERVATION), $xml);

        $data = json_decode(json_encode(simplexml_load_string($result)->COUPON->RESPONSE), true); // Fult men fungerar

        $return_data = array();
        $return_data = $data["RESERVATION"];
        $return_data["REFERENCE"] = $data["RESERVATION"]["@attributes"]["REFERENCE"];
        unset($return_data["@attributes"]);

        Mage::log(sprintf("Retain24::VALIDATE_RESERVATION(reference: %s): %s", $reference_id, json_encode($return_data, JSON_UNESCAPED_SLASHES)));

        return $data;
    }

    // Cancel reservation. Based on value from reservation.
    public function cancel_reservation($store, $reference_id)
    {
        $xml = $this->init_xml();

        $cancel_reservation = $xml->COUPON->addChild('CANCEL_RESERVATION');
        $reference = $cancel_reservation->addChild('REFERENCE', $reference_id);

        $result = $this->post_data($this->get_url($store, self::CANCEL_RESERVATION), $xml);

        $data = json_decode(json_encode(simplexml_load_string($result)->COUPON->RESPONSE), true); // Fult men fungerar

        Mage::log(sprintf("Retain24::CANCEL_RESERVATION(reference: %s): %s", $reference_id, json_encode($data, JSON_UNESCAPED_SLASHES)));

        return $data["RESERVATION_REFERENCE"];
    }

    // Purchase code/voucher for specific item or all items in order.
    public function purchase_voucher($order, $specificItemId = null, $qty = 0)
    {
        // Let's find out if order contains any voucher products.
        $itemList = $this->getVoucherItemList($order);

        // Loop through all voucher products, if we have any.
        if (!empty($itemList)) {
            Mage::log(
                sprintf(
                    "Retain24::purchase_voucher(order: %s): %s",
                    $order->getIncrementId(),
                    json_encode($itemList, JSON_UNESCAPED_SLASHES)
                )
            );
            
            foreach ($itemList as $itemId => $itemData) {

                // Only process vouchers for specified item, if supplied
                if (!empty($specificItemId) && $itemId != $specificItemId) {
                    continue;
                }

                // Check to se if we are supposed to purchase specified amount instead of all
                if (!empty($specificItemId) && !empty($qty)) {
                    $itemData['qty'] = $qty;
                }

                // Make sure item has Retain24 template id
                if (!empty($itemData['product_config'][self::VOUCHER_TEMPLATE])) {
                    $this->send_voucher($order, $itemData);
                } else {
                    Mage::throwException("Missing template id");
                }

                // Prepare comment for order with info about purchase
                $codeWord = $itemData['qty'] > 1 ? "codes" : "code";
                $sentBy = Mage::registry('retain24_sent_by') ?: "system";
                Mage::register('retain24_order_comment', Mage::registry('retain24_order_comment') . "User '{$sentBy}' successfully issued {$itemData['qty']} {$codeWord} for [{$itemData['product_config']['sku']}] {$itemData['product_config']['name']}\n");
            }
        }
        
    }

    /**
     * send_voucher (purchase) retain24 code.
     *
     * @param  Mage_Sales_Model_Order
     * @param  Mage_Catalog_Model_Product
     * @param  array
     * @return void
     */
    public function send_voucher($order, $itemData)
    {
        $store = $order->getStore();

        $billingAddressId = $order->getBillingAddressId();
        $shippingAddressId = $order->getShippingAddressId();
        if (empty($billingAddressId) && empty($shippingAddressId)) {
            // Allow order without shipping AND billing address? Don't think so!
            Mage::throwException("Missing both billing address and shipping address");
        } elseif (empty($billingAddressId) && !empty($shippingAddressId)) {
            // Free order get same billing address as shipping address.
            $shippingAddress = Mage::getModel('sales/order_address')->load($shippingAddressId);
            $billingAddress = $shippingAddress;
        } elseif (!empty($billingAddressId) && empty($shippingAddressId)) {
            // Virtual order get same shipping address as billing address.
            $billingAddress = Mage::getModel('sales/order_address')->load($billingAddressId);
            $shippingAddress = $billingAddress;
        } else {
            // Normal order with both billing and shipping address.
            $billingAddress = Mage::getModel('sales/order_address')->load($billingAddressId);
            $shippingAddress = Mage::getModel('sales/order_address')->load($shippingAddressId);
        }

        $xml = $this->init_xml();
        $xml->COUPON->addChild("SEND");
        $xml->COUPON->SEND->addChild("TEMPLATE", $itemData['product_config'][self::VOUCHER_TEMPLATE]);

        // QTY is only used for dynamic vouchers
        if ($itemData['product_config'][self::VOUCHER_DYNAMIC]) {
            $actualQty = $itemData['value'];
            if (!empty($itemData['product_config'][self::VOUCHER_MULTIPLIER])) {
                $actualQty *= $itemData['product_config'][self::VOUCHER_MULTIPLIER];
            }
            $xml->COUPON->SEND->addChild("QTY", $actualQty);
        }

        switch ($itemData['product_config'][self::VOUCHER_DISTRIBUTION]) {
            case self::DISTRIBUTION_SMS:
                $phone = $shippingAddress->getTelephone() ?: ($billingAddress->getTelephone() ?: 0);
                if (empty($phone)) {
                    Mage::throwException("Missing phone number");
                }
                $xml->COUPON->SEND->addChild("MSISDN", $this->formatPhone($store, $phone));
                $xml->COUPON->SEND->addChild("SMS_TEXT", "ToDo: set message");
                $xml->COUPON->SEND->addChild("SEND_DATE", "ToDo: set date");
                break;

            case self::DISTRIBUTION_EMAIL:
                $customerEmail = $order->getCustomerEmail();
                if (empty($customerEmail)) {
                    Mage::throwException("Missing email address");
                }
                $xml->COUPON->SEND->addChild("EMAIL_ADDRESS", $customerEmail);
                $xml->COUPON->SEND->addChild("EMAIL_TEXT", "ToDo: set message");
                $xml->COUPON->SEND->addChild("SEND_DATE", "ToDo: set date");
                break;

            case self::DISTRIBUTION_EXTERNAL:
                // No additional information needed
                break;

            case self::DISTRIBUTION_POST:
                $address = $shippingAddress->getStreet() ?:  ($billingAddress->getStreet() ?: []);
                if (!is_array($address)) { // Sometimes the street seems to be a string, not an array (should have 2 rows)
                    $address = [$address];
                }
                if (empty($address[0])) {
                    Mage::throwException("Missing address (street)");
                }
                $zipcode = $shippingAddress->getPostcode() ?: ($billingAddress->getPostcode() ?: 0);
                if (empty($zipcode)) {
                    Mage::throwException("Missing address (zipcode)");
                }

                $city = $shippingAddress->getCity() ?: ($billingAddress->getCity() ?: 0);
                if (empty($city)) {
                    Mage::throwException("Missing address (city)");
                }

                $countryCode = $shippingAddress->getCountryId() ?: ($billingAddress->getCountryId() ?: "SE");
                $countryName = $this->getCountryName($countryCode);
                if (empty($countryName)) {
                    Mage::throwException("Missing country name for: {$countryCode}");
                }
                $xml->COUPON->SEND->addChild("POSTADDRESS");
                $xml->COUPON->SEND->POSTADDRESS->addChild("DATA");
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("RECEIVE_DATE", "ToDo: set date");
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("FIRSTNAME", $order->getCustomerFirstname());
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("LASTNAME", $order->getCustomerLastname());
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("ADDRESS", $address[0]);
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("ADDRESS2", $address[1] ?? "");
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("ZIPCODE", $zipcode);
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("CITY", $city);
                $xml->COUPON->SEND->POSTADDRESS->DATA->addChild("COUNTRY", $countryName);
                $xml->COUPON->SEND->addChild("SEND_DATE", "ToDo: set date");
                break;

            case self::DISTRIBUTION_PRINTOUT:
                // No additional information needed
                break;

        }

        $url = $this->get_url($store, self::SEND);
        for ($iterations = 0; $iterations < $itemData['qty']; $iterations++) {

            // Pause for a while if we are supposed to buy more than 1 voucher, so we don't trigger any hacking or DDOS filter
            if ($iterations > 0 && $itemData['qty'] > 1) {
                usleep(100000);
            }

            $result = $this->post_data($url, $xml);
            Mage::log(
                sprintf(
                    "Retain24::SEND_VOUCHER(order: %s, item: %s, qty: %s of %s): %s",
                    $order->getIncrementId(),
                    $itemData['item_id'],
                    $iterations + 1,
                    $itemData['qty'],
                    json_encode($result, JSON_UNESCAPED_SLASHES)
                )
            );

            $xmlData = simplexml_load_string($result);
            if (!empty($xmlData->COUPON->RESPONSE->RECEIPT)) {
                $receipt = json_decode(json_encode($xmlData->COUPON->RESPONSE->RECEIPT), true);
                if (!empty($receipt['STATUS'])) {
                    if ($receipt['STATUS'] == "OK") {
                        $this->saveVoucher($itemData, $receipt);
                    } else {
                        $message = "Status = {$receipt['STATUS']}";
                        $message .= !empty($receipt['MESSAGE']) ? ", message = {$receipt['MESSAGE']}" : "";
                        Mage::throwException($message);
                    }
                } else {
                    Mage::throwException('Empty status');
                }
            } else {
                Mage::throwException('Missing receipt');
            }
        }

    }

    public function transaction_report($store, $cpntype)
    {
        if (is_numeric($store)) {
            Mage::app()->setCurrentStore($store);
            $store = Mage::app()->getStore();
        }

        $xml = $this->init_xml();
        $xml->COUPON->addChild("TRANSACTION_REPORT");
        $xml->COUPON->TRANSACTION_REPORT->addChild("STARTDATE", date('Y-m-d 00:00', strtotime('-1 month')));
        $xml->COUPON->TRANSACTION_REPORT->addChild("ENDDATE", date('Y-m-d H:i'));
        $xml->COUPON->TRANSACTION_REPORT->addChild("CPNTYPE", $cpntype);
        $xml->COUPON->TRANSACTION_REPORT->addChild("USERONLY", "1");

        $url = $this->get_url($store, self::TRANSACTION_REPORT);
        //return $this->post_data($url, $xml);
        $result = $this->post_data($url, $xml);

        $xmlData = simplexml_load_string($result);
        if (!empty($xmlData->COUPON->RESPONSE->TRANSACTION_REPORT)) {
            $report = json_decode(json_encode($xmlData->COUPON->RESPONSE->TRANSACTION_REPORT), true);
            if (!empty($report['TRANSACTIONDATA'])) {
                return $report['TRANSACTIONDATA'];
            } else {
                return [];
            }
        } else {
            Mage::throwException('Missing result: TRANSACTION_REPORT');
        }
    }

    public function getOrderData($orderId)
    {
        $sqlQuery = "
            SELECT
                o.entity_id,
                o.increment_id,
                o.state,
                o.`status`,
                p.method AS payment_method,
                i.product_id,
                e.sku
            FROM sales_flat_order o
            JOIN sales_flat_order_payment p ON p.parent_id = o.entity_id
            JOIN sales_flat_order_item i ON i.order_id = o.entity_id
            JOIN catalog_product_entity e ON e.entity_id = i.product_id
            JOIN catalog_product_entity_int ei ON ei.entity_id = e.entity_id AND ei.store_id = 0
            JOIN eav_attribute a ON a.attribute_id = ei.attribute_id
            WHERE o.entity_id = :entityId AND a.attribute_code = :attributeCode AND ei.`value` = :attributeValue
        ";
        $params = [
            'entityId' => $orderId,
            'attributeCode' => 'is_retain24_product',
            'attributeValue' => 1
        ];
        return Mage::getSingleton('core/resource')->getConnection('core_read')->fetchAll($sqlQuery, $params);
    }

    public function getVoucherItemList($order)
    {
        $orderId = $order->getId();
        if (!array_key_exists($orderId, $this->voucherItemListCache)) {
                
            $itemList = [];
            $parentList = [];
            $storeId = $order->getStoreId();
            $orderItems = $order->getItemsCollection()->getItems();

            // Let's find out if order contains any voucher products.
            foreach ($orderItems as $item) {
                $itemId = intval($item->getId());

                // Search for vouchers
                if ($item->getProductType() == 'virtual') {
                    $productId = intval($item->getProductId());
                    $product = Mage::getModel('catalog/product')->setStoreId($storeId)->load($productId);
                    if ($product->getData(self::VOUCHER_PRODUCT)) {

                        // Found voucher product
                        $parentItemId = intval($item->getParentItemId());
                        if (!empty($parentItemId)) {
                            // Product is child, save parent and look up data for it later
                            $itemList[$itemId]['parent_item_id'] = $parentItemId;
                            $parentList[$parentItemId] = $itemId;
                        } else {
                            $itemList[$itemId]['parent_item_id'] = null;
                        }

                        // Fetch all retain24 specific attribute values
                        $itemList[$itemId]['product_config'] = $this->getProductConfig($product);

                        // Set data as if voucher have fixed value, change later if not so
                        $itemList[$itemId]['item_id'] = $itemId;
                        $itemList[$itemId]['value'] = $itemList[$itemId]['product_config'][self::VOUCHER_VALUE];
                        $itemList[$itemId]['qty'] = intval($item->getQtyOrdered());
                        $itemList[$itemId]['sku'] = $product->getSku();
                        $itemList[$itemId]['name'] = $product->getName();
                    }
                }
            }

            // Search for parent product
            if (!empty($parentList)) {
                foreach ($orderItems as $item) {
                    $itemId = intval($item->getId());
                    if (array_key_exists($itemId, $parentList)) {
                        $childId = $parentList[$itemId];
                        $productId = $item->getProductId();
                        $product = Mage::getModel('catalog/product')->setStoreId($storeId)->load($productId);

                        // Check if parent is part of voucher
                        if ($product->getData(self::VOUCHER_PRODUCT)) {
                            // Check if voucher uses dynamic value AND have no predefined actual value
                            if ($itemList[$childId]['product_config'][self::VOUCHER_DYNAMIC] && empty($itemList[$itemId]['value'])) {
                                // If so: set 'value' to be child qty and 'qty' to be parent qty
                                $itemList[$childId]['value'] = $itemList[$childId]['qty'];
                                $itemList[$childId]['qty'] = intval($item->getQtyOrdered());
                            }
                        }
                    }
                }
            }

            $this->voucherItemListCache[$orderId] = $itemList;
        }

        return $this->voucherItemListCache[$orderId];
    }

    public function getProductConfig($product)
    {
        $steps = $product->getData(self::VOUCHER_STEPS);
        $bcc = $product->getData(self::VOUCHER_EMAIL_BCC);

        return [
            'product_id'                 => intval($product->getId()),
            'sku'                        => $product->getSku(),
            'name'                       => $product->getName(),
            self::VOUCHER_TEMPLATE       => $product->getData(self::VOUCHER_TEMPLATE),
            self::VOUCHER_DYNAMIC        => $product->getData(self::VOUCHER_DYNAMIC) ? true : false,
            self::VOUCHER_REQUIRE_PIN    => $product->getData(self::VOUCHER_REQUIRE_PIN) ? true : false,
            self::VOUCHER_MIN            => intval($product->getData(self::VOUCHER_MIN)),
            self::VOUCHER_MAX            => intval($product->getData(self::VOUCHER_MAX)),
            self::VOUCHER_VALUE          => intval($product->getData(self::VOUCHER_VALUE)),
            self::VOUCHER_STEPS          => empty($steps) ? [] : explode(',', $steps),
            self::VOUCHER_MULTIPLIER     => floatval($product->getData(self::VOUCHER_MULTIPLIER)),
            self::VOUCHER_VALID_TO       => $this->getValidTo($product->getData(self::VOUCHER_VALID_TO)),
            self::VOUCHER_SALES_END      => $product->getData(self::VOUCHER_SALES_END),
            self::VOUCHER_USAGE_START    => $product->getData(self::VOUCHER_USAGE_START),
            self::VOUCHER_DISTRIBUTION   => intval($product->getData(self::VOUCHER_DISTRIBUTION)),
            self::VOUCHER_EMAIL_TEMPLATE => $product->getData(self::VOUCHER_EMAIL_TEMPLATE),
            self::VOUCHER_EMAIL_BCC      => empty($bcc) ? [] : array_filter(array_map('trim', explode(',', $bcc))),
            'retain24_text'              => $product->getData('gift_card_text'),
            'retain24_terms'             => $product->getData('gift_card_terms'),
        ];
    }

    public function getCountryName($countryId)
    {
        try {
            $sqlQuery = "SELECT name FROM crossroads_integration_country WHERE iso = ?";
            return Mage::getSingleton('core/resource')->getConnection("core_read")->fetchOne($sqlQuery, $countryId);
        } catch (Exception $exception) {
            $this->logException($exception, 'Exception while getting country name!');
        }
    }

    public function getValidTo($value)
    {
        if (preg_match('/^[0-9]{2,4}-[0-9]{2}-[0-9]{2}$/', $value)) {
            $time = strtotime($value);
            return date('Y-m-d', $time);
        } elseif(is_numeric($value)) {
            $days = intval($value);
            $time = strtotime("+{$days} days");
            return date('Y-m-d', $time);
        }
        return null;
    }

    public function saveVoucher($itemData, $receipt)
    {
        $voucher = Mage::getModel("Crossroads_Retain24/voucher");
        $newData = [
            'item_id' => $itemData['item_id'],
            'voucher_template_id' => $itemData['product_config'][self::VOUCHER_TEMPLATE],
            'voucher_distribution' => $itemData['product_config'][self::VOUCHER_DISTRIBUTION],
            'voucher_value' => $itemData['value'],
            'voucher_id' => $receipt['ID'],
            'voucher_code' => $receipt['MULTICODE'],
            'voucher_pin' => $itemData['product_config'][self::VOUCHER_REQUIRE_PIN] ? (!empty($receipt['PIN']) ? $receipt['PIN'] : null) : null,
            'valid_to' => $itemData['product_config'][self::VOUCHER_VALID_TO]
        ];
        $voucher->addData($newData);
        $voucher->save();
    }

    public function getImagesForEmail($productId, $storeId = 0)
    {
        $retval = array('top' => null, 'bottom' => null);
        $sqlQuery = "SELECT T1.value, T2.store_id, T2.label FROM catalog_product_entity_media_gallery T1 JOIN catalog_product_entity_media_gallery_value T2 ON T2.value_id = T1.value_id WHERE T1.entity_id = ? AND T2.label != ''";
        $images = Mage::getSingleton('core/resource')->getConnection('core_read')->fetchAll($sqlQuery, $productId);
        if ($images) {
            foreach ($images as $image) {
                $labelData = explode('_', $image['label']);
                if (!empty($labelData)) {
                    $label = $labelData[0];
                    if (array_key_exists($label, $retval)) {
                        if ($image['store_id'] == $storeId || ($image['store_id'] == 0 && empty($retval[$label]))) {
                            if ($label == 'top') {
                                $retval['background-color'] = empty($labelData[1]) ? 'ffffff' : $labelData[1];
                                $retval['border-color'] = empty($labelData[2]) ? $retval['background-color'] : $labelData[2];
                                $retval['font-color'] = empty($labelData[3]) ? '000000' : $labelData[3];
                            } elseif ($label == 'bottom') {
                                $retval['font-color'] = empty($labelData[1]) ? '000000' : $labelData[1];
                            }
                            $retval[$label] = Mage::getStoreConfig('web/unsecure/base_url', $storeId) . 'media/catalog/product' . $image['value'];
                        }
                    }
                }
            }
        }

        return $retval;
    }

    public function sendCustomerEmailForOrder($order)
    {
        $itemList = $this->getVoucherItemList($order);

        foreach($itemList as $itemData) {
            $this->sendCustomerEmailForItem($order, $itemData);
        }

    }
    
    // https://stackoverflow.com/questions/22039156/cant-pass-variable-in-block-directive-from-transactional-email-template
    // {{block type="mymodule/sales_order_email_description" template="email/order/description.phtml" order=$order}}
    // http://excellencemagentoblog.com/blog/2011/11/25/magento-advanced-transactional-email-templates/
    // https://magento.stackexchange.com/questions/3769/pass-data-to-getchildhtml-or-call-method-on-child-block
    // https://magento.stackexchange.com/questions/141586/how-to-use-custom-variables-in-phtml-file-for-email
    public function sendCustomerEmailForItem($order, $itemData)
    {
        $incrementId = $order->getIncrementId();
        $storeId = $order->getStoreId();

        $customerEmail = $order->getCustomerEmail();
        if (empty($customerEmail)) {
            $this->_sendAdminEmail('Retain24 order missing email address', "Order {$incrementId} is missing customer email address.");
            Mage::throwException("Order {$incrementId} is missing customer email address.");
        }

        $customerName = $order->getBillingAddress()->getName() ?: ($order->getShippingAddress()->getName() ?: 0);
        if (empty($customerName)) {
            $this->_sendAdminEmail('Retain24 order missing customer name', "Order {$incrementId} is missing customer name.");
            Mage::throwException("Order {$incrementId} is missing customer name.");
        }

        $codes = Mage::getModel("Crossroads_Retain24/voucher")->getCollection()->addItemFilter($itemData['item_id']);
        $codeQty = count($codes);
        if (empty($codeQty)) {
            Mage::throwException("Order {$incrementId} is missing vouchers for product [{$itemData['sku']}].");
        }

        // Remove PIN from code(s) if template does not require it.
        if (!$itemData['product_config'][self::VOUCHER_REQUIRE_PIN]) {
            foreach ($codes as &$code) {
                $code['voucher_pin'] = null;
            }
        }

        $mailTemplate = $itemData['product_config'][self::VOUCHER_EMAIL_TEMPLATE];
        if (is_numeric($mailTemplate)) {
            $emailTemplate = Mage::getModel('core/email_template')->load($mailTemplate);
        } else {
            $emailTemplate = Mage::getModel('core/email_template')->loadDefault($mailTemplate);
        }
        $emailTemplate->setSenderName(Mage::getStoreConfig('trans_email/ident_sales/name', $storeId));
        $emailTemplate->setSenderEmail(Mage::getStoreConfig('trans_email/ident_sales/email', $storeId));

        if (!empty($itemData['product_config'][self::VOUCHER_EMAIL_BCC])) {
            $emailTemplate->addBcc($itemData['product_config'][self::VOUCHER_EMAIL_BCC]);
        }

        $emailTemplateVariables = [
            'order' => $order,
            'product' => Mage::getModel('catalog/product')->setStoreId($storeId)->load($itemData['product_config']['product_id']),
            'codes' => $codes,
            'store_name' => Mage::getStoreConfig('general/store_information/name', $storeId),
            'retain24_text' => $itemData['product_config']['retain24_text'],
            'retain24_terms' => $itemData['product_config']['retain24_terms']
        ];

        $imageData = $this->getImagesForEmail($itemData['product_config']['product_id'], $storeId);
        if (!empty($imageData)) {
            $emailTemplateVariables['image_top'] = $imageData['top'] ?? '';
            $emailTemplateVariables['image_bottom'] = $imageData['bottom'] ?? '';
            $emailTemplateVariables['background-color'] = $imageData['background-color'] ?? '';
            $emailTemplateVariables['border-color'] = $imageData['border-color'] ?? '';
            $emailTemplateVariables['font-color'] = $imageData['font-color'] ?? '';
        }

        if (self::USE_EMAIL_QUEUE) {
            /** @var $emailQueue Mage_Core_Model_Email_Queue */
            $emailQueue = Mage::getModel('core/email_queue');
            $emailQueue->setEventType(self::EMAIL_EVENT_TYPE)
                ->setEntityType(Mage_Sales_Model_Order::ENTITY)
                ->setEntityId($order->getId())
                ->setIsForceCheck(false);
            $emailTemplate->setQueue($emailQueue);
        }

        $emailTemplate->send($customerEmail, $customerName, $emailTemplateVariables);

        // Mark each code as sent
        foreach($codes as $code) {
            $code->setData('sent_at', date('Y-m-d H:i:s'))->save();
        }

        // Prepare comment for order about this sending of code(s)
        $codeWord = $codeQty > 1 ? "codes" : "code";
        $sentBy = Mage::registry('retain24_sent_by') ?: "system";
        Mage::register('retain24_order_comment', Mage::registry('retain24_order_comment') . "User '{$sentBy}' successfully sent {$codeQty} {$codeWord} for [{$itemData['product_config']['sku']}] {$itemData['product_config']['name']}\n");
    }

}