<?php

use \Klarna\XMLRPC\Flags;

class Crossroads_Retain24_Model_Observer extends Mage_Core_Model_Abstract {
    public function checkoutPostMerge(Varien_Event_Observer $observer): ?Crossroads_Retain24_Model_Validation
    {
        $quote = $observer->getQuote();
        $data = $observer->getParams();
        $store = $quote->getStore();
        /** @var Crossroads_Retain24_Helper_Data */
        $helper = Mage::helper("Retain24");

        if (!$data->getData("retain24")) {
            return null;
        }

        if ($data->getData("retain24") === null) {
            $this->removeCode($quote);
            return null;
        }

        if (!is_array($data->getData("retain24"))) {
            throw new Crossroads_Retain24_ResponseException("Key 'retain24' must be an object or null", 778);
        }

        $retain24 = $data->getData("retain24");

        if (array_key_exists("code", $retain24) && $retain24["code"]) {
            $model = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());
            $code = trim($retain24["code"]);
            $isPhone = (bool) $retain24["isPhone"] ?? false;
            $pin = $retain24["pin"] ?? null;

            if (
                $code != $model->getCode()
                || !$model->getValidationId()
                || $isPhone != $model->getIsPhone()
                || (
                    ($isPhone || $model->getRequiresPin())
                    && $pin !== $model->getPin()
                )
            ) {
                $quote->setRetain24Code($code);
                $quote->setRetain24Pin($pin);

                $code_data = $helper->validate($store, $code, $isPhone, $pin);

                if (!$code_data) {
                    $model->delete();
                    throw new Crossroads_Retain24_ResponseException("Retain24 validation failed", 777);
                }

                if ($code_data["REQUIRES_PIN"] === "YES" && !$pin) {
                    throw new Crossroads_Retain24_ResponseException("Code requires pin for validation", 791);
                }

                if ($code_data["REQUIRES_PIN"] === "YES" && $pin && $code_data["PIN_VERIFICATION"] !== "OK") {
                    throw new Crossroads_Retain24_ResponseException("Invalid pin", 792);
                }

                if ($code_data["CURRENCY"] !== $quote->getBaseCurrencyCode()) {
                    throw new Crossroads_Retain24_ResponseException(sprintf("Retain24 code is for the currency '%s', we have '%s'.", $code_data["CURRENCY"], $quote->getBaseCurrencyCode()), 789);
                }

                $extra_code_data = $helper->validate_phone($store, $code, $isPhone, $pin);
                $userdefined = $extra_code_data["USERDEFINED"] ?? "";

                $model->addData([
                    "quote_id" => $quote->getEntityId(),
                    "validation_id" => $code_data["ID"],
                    "code" => $code,
                    "code_type" => $code_data["CPNTYPE"],
                    "code_status" => $code_data["STATUS"],
                    "name" => $code_data["NAME"],
                    "qty" => $code_data["QTY"],
                    "value" => $code_data["VALUE"],
                    "currency" => $code_data["CURRENCY"],
                    "requires_pin" => $code_data["REQUIRES_PIN"] === "YES",
                    "description" => $code_data["DESCRIPTION"],
                    "userdefined" => (is_array($userdefined) || empty($userdefined)) ? null : (string)$userdefined,
                    "qty_max" => $code_data["QTYMAX"],
                    "usage_end" => $helper->resolveTimestamp($code_data["USAGE_END"], 'Europe/Stockholm'),
                    "pin_verified" => $code_data["PIN_VERIFICATION"] === "OK",
                    "pin" => $pin,
                    "is_phone" => $isPhone,
                ]);

                switch ($code_data["ERRORCODE"]) {
                    case 0:
                        // Success
                        $observer->setResultCode('success');
                        $model->save();
                        try {
                            $model->validateQuote($quote);
                        }
                        finally {
                            $quote->save();
                        }
                        break;

                    case 25001:
                        $observer->setResultCode('_779');
                        break;

                    case 25002:
                        $observer->setResultCode('_780');
                        break;

                    case 25003:
                        $observer->setResultCode('_781');
                        break;

                    case 25004:
                        $observer->setResultCode('_782');
                        break;

                    case 25005:
                        $observer->setResultCode('_783');
                        break;

                    case 25070:
                        $observer->setResultCode('_784');
                        break;

                    case 30001:
                        $observer->setResultCode('_785');
                        break;

                    default:
                        $observer->setResultCode('_786');
                }

                return $model;
            } else {
                // Validation failed
                return null;
            }
        } elseif (array_key_exists("code", $retain24) && $retain24["code"] === null) {
            $this->removeCode($quote);
        }
        return null;
    }

    protected function removeCode($quote)
    {
        $quote->unsRetain24Code();
        $quote->unsRetain24Pin();

        Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId())->delete();
    }

    /**
     * Observer listening to `sales_order_invoice_save_after` to update deposit amount invoiced in order.
     *
     * @param  Varien_Event_Observer
     */
    public function invoiceSaveAfter(Varien_Event_Observer $observer)
    {
        $invoice = $observer->getInvoice();
        $order = $invoice->getOrder();

        if ($invoice->getBaseRetain24Amount()) {
            $order->setRetain24AmountInvoiced($order->getRetain24AmountInvoiced() + $invoice->getRetain24Amount());
            $order->setBaseRetain24AmountInvoiced($order->getBaseRetain24AmountInvoiced() + $invoice->getBaseRetain24Amount());
        }

        return $this;
    }

    /**
     * Observer listening to `sales_order_creditmemo_save_after` to update deposit amount refunded in order.
     *
     * @param  Varien_Event_Observer
     */
    public function creditmemoSaveAfter(Varien_Event_Observer $observer)
    {
        $creditmemo = $observer->getCreditmemo();
        $order = $creditmemo->getOrder();

        if ($creditmemo->getBaseRetain24Amount()) {
            $order->setRetain24AmountRefunded($order->getRetain24AmountRefunded() + $creditmemo->getRetain24Amount());
            $order->setBaseRetain24AmountRefunded($order->getBaseRetain24AmountRefunded() + $creditmemo->getBaseRetain24Amount());
        }

        return $this;
    }

    public function quoteSaveBefore(Varien_Event_Observer $observer): void {
        $quote = $observer->getQuote();
        $validation = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());

        if ($validation->getValidationId()) {
            $validation->validateQuote($quote);
        }
    }

    // Validate and reserve amount if validation succeeds.
    public function quoteSubmitBefore(Varien_Event_Observer $observer)
    {
        $quote = $observer->getQuote();
        $store = $quote->getStore();
        $helper = Mage::helper("Retain24");
        $validation = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());
        $reservation = Mage::getModel("Crossroads_Retain24/reservation")->load($quote->getEntityId());

        if ($validation->getValidationId()) {
            $validation->validateQuote($quote);

            $amount = 0;

            foreach ($quote->getAddressesCollection() as $a) {
                $amount += $a->getBaseRetain24Amount();
            }

            $qty = round(abs($amount / $validation->getValue()));

            if ($reservation->getReferenceId() && $reservation->getStatus() == 1) {
                // We have a previous reservation, cancel it and replace it
                $this->cancelReservation($store, $reservation);
            }

            $res = $helper->reserve_instance($store, $validation->getValidationId(), $qty, $validation->getPin());

            if ($res["INFO"]["DATA"]["ERRORCODE"] != 0) {
                switch ($res["INFO"]["DATA"]["ERRORCODE"]) {
                    case 25200:
                        throw new Crossroads_Retain24_ResponseException("Reservation failed", 787);

                    default:
                        throw new Crossroads_Retain24_ResponseException("Unknown error with reservation", 788);
                }
            }

            $reservation->addData(array(
                "quote_id" => $quote->getEntityId(),
                "reference_id" => $res["REFERENCE"],
                "expiry_date" => $res["EXPIRES"],
                "reserved_qty" => $qty,
                "validation_id" => $res["INFO"]["DATA"]["ID"],
                "status" => 1,
            ))->save();
        }
    }

    public function quoteSubmitAfter(Varien_Event_Observer $observer): void
    {
        $quote = $observer->getQuote();
        $order = $observer->getOrder();
        $validation = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());
        if ($validation->getValidationId()) {
            $validation->setOrderId($order->getEntityId())->save();
        }
    }

    // Redeem or cancel reservation based on status.
    public function orderStatusChange(Varien_Event_Observer $observer)
    {
        $order = $observer->getEvent()->getOrder();
        $store = $order->getStore();

        $reservation = Mage::getModel("Crossroads_Retain24/reservation")->load($order->getQuoteId());

        if ($reservation->getStatus() == 1) { // Only process orders with status 1 (reserved)
            if (in_array($order->getStatus(), [ "processing","complete" ])) {
                $validation = Mage::getModel("Crossroads_Retain24/validation")->load($order->getQuoteId());
                $this->redeemReservation($store, $validation, $reservation);
            } elseif ($order->getStatus() == "canceled") {
                $this->cancelReservation($store, $reservation);
            }
        }
    }

    // Redeem reserved amount when orderstatus changes to processing.
    private function redeemReservation($store, $validation, $reservation)
    {
        $redeem = Mage::helper("Retain24")->redeem_reservation($store, $reservation->getReferenceId(), $validation->getValidationId(), $reservation->getReservedQty());

        if ($redeem["STATUS"] == "OK") {
            $reservation->setStatus(2)->save();
        } else {
            $reservation->setStatus(99)->setStatusMessage($redeem["MESSAGE"])->save();
        }
    }

    // Cancel reservation when orderstatus changes to cancelled.
    private function cancelReservation($store, $reservation)
    {
        $cancel = Mage::helper("Retain24")->cancel_reservation($store, $reservation->getReferenceId());

        if ($cancel["STATUS"] == "OK") {
            $reservation->setStatus(3)->save();
        } else {
            $reservation->setStatus(99)->setStatusMessage($cancel["MESSAGE"])->save();
        }
    }

    public function klarnaReserveAmount(Varien_Event_Observer $event)
    {
        $order = $event->getOrder();
        $klarna = $event->getKlarna();

        if (abs($order->getBaseRetain24Amount()) > 0) {
            $total = $order->getBaseGrandTotal() - $order->getBaseRetain24Amount();
            $totalNoTax = $total - $order->getTaxAmount();
            $vatPercent = $totalNoTax > 0 ? ($total / $totalNoTax - 1) * 100 : 0;

            $klarna->addArticle(
                    1, "Retain24", Mage::helper("Retain24")->__("Retain24"),
                    // BaseRetain24Amount is negative, so we remove that from the total
                    $order->getBaseRetain24Amount(), $vatPercent, 0, Flags::INC_VAT
            );
        }
    }

    /**
     * Adds items to the list of items sent to collector checkout
     * with negative value equals to the value of the voucher attached to the quote.
     * One item per tax-rate is added to the collection.
     */
    public function collectorCartBodyPostAddItems(Varien_Event_Observer $event)
    {
        $quote = $event->getQuote();
        $itemsContainer = $event->getContainer();
        $items = $itemsContainer->getItems();
        if (empty($items)) {
            return; // Don't try to manipulate an empty cart
        }

        $validation = Mage::getModel("Crossroads_Retain24/validation")->load($quote->getEntityId());

        if (empty($validation)) {
            return;
        }

        $retain24value = floatval($validation->getQty() * $validation->getValue());

        if ($retain24value <= 0) {
            return;
        }

        $total = 0.0;
        $rates = [];

        // loop all items to get total per vat-rate
        // also collect total of all items
        foreach ($items as $item) {
            $total += $item["unitPrice"] * $item["quantity"];

            if (isset($rates["{$item['vat']}"])) {
                $rates["{$item['vat']}"] += $item["unitPrice"] * $item["quantity"];
            }
            else {
                $rates["{$item['vat']}"] =  $item["unitPrice"] * $item["quantity"];
            }
        }

        $unitPriceTotal = 0;
        $idx = count($items);
        $maxIdx = $idx;

        // create one item per vat-rate and add to items
        foreach ($rates as $vat => $rateTotal) {
            $unitPrice = round(($rateTotal / $total) * $retain24value, 2);
            $unitPriceTotal += $unitPrice;

            $items[$idx] = [
                'id'          => "retain24voucher-vat_$vat",
                'description' => "Attached Retain24 Voucher (Vat $vat%)",
                'unitPrice'   => -($unitPrice),
                'quantity'    => 1,
                'vat'         => floatval($vat),
            ];

            if ($items[$idx]["unitPrice"] > $items[$maxIdx]["unitPrice"]) {
                $maxIdx = $idx;
            }

            $idx++;
        }

        $diff = $retain24value - $unitPriceTotal;

        if ($diff < 0.00 || $diff > 0.00) {
            $items[$maxIdx]["unitPrice"] = round($items[$maxIdx]["unitPrice"] - $diff, 2);
        }

        $itemsContainer->setItems($items);
    }

    public function issueVoucherEvent(Varien_Event_Observer $event)
    {
        $order = $event->getOrder();
        if(empty($order)) {
            return;
        }

        $orderId = $order->getId();
        if(empty($orderId)) {
            return;
        }

        $orderData = Mage::helper("Retain24")->getOrderData($orderId);
        // Only process orders that actually contain Retain24 products
        if (empty($orderData)) {
            return;
        }

        // Make sure we only issue vouchers for orders with correct status.
        if (!in_array($order->getStatus(), Crossroads_Retain24_Helper_Data::$issueVoucherOrderStatuses)) {
            return;
        }

        $storeId = $order->getStoreId();
        // If issueVoucherEvent() is triggered by script or other method, make sure we use same store context as order.
        $emulate = $storeId != Mage::app()->getStore()->getId();
        Mage::register('retain24_order_comment', "");

        // retain24_sent_by can be set by script or other method.
        // If not, try to figure out apropriate username
        $username = Mage::registry('retain24_sent_by');
        if (empty($username)) {
            // Normally issueVoucherEvent() is triggered by customer
            $username = 'customer';
            $adminSession = Mage::getSingleton('admin/session');
            if ($adminSession) {
                $adminUser = $adminSession->getUser();
                if ($adminUser) {
                    $username = $adminUser->getUsername();
                }
            }
            Mage::register('retain24_sent_by', $username);
        }

        // We need this to fetch correct data for emails
        if ($emulate) {
            $appEmulation = Mage::getSingleton('core/app_emulation');
            $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);
        }

        try {
            Mage::helper("Retain24")->purchase_voucher($order);
            Mage::helper("Retain24")->sendCustomerEmailForOrder($order);
        } catch (Exception $ex) {
            Mage::logException($ex);
            // ToDo: Perhaps send email about this?
        }

        $comment = Mage::registry('retain24_order_comment');
        if (!empty($comment)) {
            $history = $order->addStatusHistoryComment($comment);
            $history->setIsCustomerNotified(Mage_Sales_Model_Order_Status_History::CUSTOMER_NOTIFICATION_NOT_APPLICABLE);
            $order->save();
        }

        if ($emulate) {
            $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
        }
    }

    public function appendAdminHtml(Varien_Event_Observer $event)
    {
        $block = $event->getBlock();
        if ($block->getNameInLayout() != 'order_payment') {
            return;
        }
        $child = $block->getChild('retain24.redemptions');
        $transport = $event->getTransport();
        if ($child && $transport) {
            $transport->setHtml("{$transport->getHtml()}{$child->toHtml()}");
        }
    }
}
