<?php

class Crossroads_Retain24_Model_Validation extends Mage_Core_Model_Abstract {

    const TYPE_UNSPECIFIED = "UNSPECIFIED";
    const TYPE_GIFT_CARD = "GIFT_CARD";
    const TYPE_COUPON = "COUPON";
    const TYPE_BONUS_VOUCHER = "BONUS_VOUCHER";
    const TYPE_CREDIT_NOTE = "CREDIT_NOTE";
    const TYPE_BONUS_CARD = "BONUS_CARD";
    const TYPE_PREPAID_CARD = "PREPAID_CARD";

    protected function _construct()
    {
        $this->_init("Crossroads_Retain24/validation");
    }

    public function codeType()
    {
        switch ($this->getCodeType()) {
            case 0:
                return self::TYPE_UNSPECIFIED;
            case 1:
                return self::TYPE_GIFT_CARD;
            case 2:
                return self::TYPE_COUPON;
            case 3:
                return self::TYPE_BONUS_VOUCHER;
            case 4:
                return self::TYPE_CREDIT_NOTE;
            case 5:
                return self::TYPE_BONUS_CARD;
            case 6:
                return self::TYPE_PREPAID_CARD;
            default:
                return null;
        }
    }

    /**
     * @throws Crossroads_Retain24_ProductCategoryDisallowedException
     */
    public function validateQuote(Mage_Sales_Model_Quote $quote): void {
        $categoryId = trim($this->getUserdefined() ?: "");

        if(empty($categoryId)) {
            Mage::log(sprintf(
                "%s: No limits specified on validation %s",
                __METHOD__,
                $this->getValidationId()
            ));

            return;
        }

        $category = Mage::getModel("catalog/category");
        $category->load($categoryId);

        if( ! $category->getId()) {
            Mage::log(sprintf(
                "%s: Category id '%s' not found on validation %s",
                __METHOD__,
                $this->getUserdefined(),
                $this->getValidationId()
            ));

            throw new Crossroads_Retain24_ProductCategoryDisallowedException();
        }

        $invalidCategories = false;
        $categoryIds = [(int)$category->getId()];
        $categoryChildren = $category->getAllChildren(true);

        foreach($categoryChildren as $c) {
            $categoryIds[] = (int)$c;
        }

        foreach($quote->getAllVisibleItems() as $i) {
            if($i->isDeleted()) {
                // Skip already deleted items
                continue;
            }

            if($i->getParentItem()) {
                // Verify the parent item, children do not always exist in the categories
                $i = $i->getParentItem();
            }

            $ids = array_map("intval", $i->getProduct()->getCategoryIds());

            if(count(array_intersect($ids, $categoryIds)) === 0) {
                Mage::log(sprintf(
                    "%s: Product '%s' (%d) on quote %d with categories [%s] is not in allowed categories [%s] on validation %s",
                    __METHOD__,
                    $i->getSku(),
                    $i->getProductId(),
                    $quote->getId(),
                    implode(", ", $ids),
                    implode(", ", $categoryIds),
                    $this->getUserdefined(),
                    $this->getValidationId()
                ));

                $i->delete();

                $invalidCategories = true;
            }
        }

        if($invalidCategories) {
            // Recalculate and save since we deleted items, then throw with error
            $quote->setTotalsCollectedFlag(false);
            $quote->collectTotals();

            throw new Crossroads_Retain24_ProductCategoryDisallowedException();
        }

        Mage::log(sprintf(
            "%s: Quote %d passed categories [%s] on validation %s",
            __METHOD__,
            $quote->getId(),
            implode(", ", $categoryIds),
            $this->getValidationId()
        ));
    }
}
