<?php

class Crossroads_PurchaseLimiter_Model_Order extends Mage_Core_Model_Abstract {
    /**
     * Lists all items exceeding the limit present in the cart.
     *
     * @param  Mage_Core_Model_Store
     * @param  integer
     * @param  Mage_Sales_Model_Quote
     * @return Array<[sku => string, mode => "QTY"|"ROW_TOTAL"], limit => double, req => double>
     */
    public function getLimitExceedingItems($store, $customerId, $quote) {
        $limits   = $this->getRelevantLimits(Mage::helper("crossroads_purchaselimiter")->getLimitedSkus($store), $quote);

        $skus     = call_user_func_array("array_merge", array_merge([[], []], array_column($limits, "skus")));
        $after    = date("Y-m-d", strtotime("-".($store->getConfig(Crossroads_PurchaseLimiter_Helper_Data::CONFIG_DURATION) ?: 0)));
        $cart     = $this->getLimitedItemsInCart($skus, $quote);
        $consumed = $customerId ? $this->fetchProductQuantities($skus, $after, $customerId) : [];
        $exceeded = [];

        // Summarize per limit-row since SKUs can be shared between rows
        foreach($limits as $limit) {
            $key = null;
            $current = [
                "skus"  => [],
                "qty"   => 0,
                "total" => 0,
            ];

            foreach($limit["skus"] as $s) {
                if(array_key_exists($s, $cart)) {
                    if( ! in_array($s, $current["skus"])) {
                        $current["skus"][] = $s;
                    }

                    $current["qty"]   += $cart[$s]["qty"];
                    $current["total"] += $cart[$s]["total"];
                }

                if(array_key_exists($s, $consumed)) {
                    if( ! in_array($s, $current["skus"])) {
                        $current["skus"][] = $s;
                    }

                    $current["qty"]   += $consumed[$s]["qty"];
                    $current["total"] += $consumed[$s]["total"];
                }
            }

            switch($limit["mode"]) {
            case "QTY":
                $key = "qty";
                break;
            case "ROW_TOTAL":
                $key = "total";
                break;
            default:
                Mage::log(sprintf("%s: Unknown limit mode '%s'.", __METHOD__, $limit["mode"]));
            }

            if($key && ($current[$key] > $limit["limit"])) {
                $exceeded[] = [
                    "skus"  => $limit["skus"],
                    "mode"  => $limit["mode"],
                    "limit" => (double)$limit["limit"],
                    "req"   => (double)$current[$key],
                ];
            }
        }

        return $exceeded;
    }

    /**
     *
     * @param  Array<[skus => Array<string>]>
     * @return Array<[skus => Array<string>]>
     */
    public function getRelevantLimits($limits, $quote) {
        $items = $quote->getAllItems();

        return array_filter($limits, function($l) use($items) {
            foreach($items as $i) {
                if(in_array($i->getSku(), $l["skus"])) {
                    return true;
                }
            }

            return false;
        });
    }

    /**
     * Lists the limited items in the cart, their quantities and totals.
     *
     * @param  Array<[sku => Array<string>]>
     * @param  Mage_Sales_Quote
     * @return Array<[sku => string, qty => double, total => double]>
     */
    public function getLimitedItemsInCart($skus, $quote) {
        $cart   = [];

        foreach($quote->getAllItems() as $i) {
            if( ! in_array($i->getSku(), $skus)) {
                continue;
            }

            if(empty($cart[$i->getSku()])) {
                $cart[$i->getSku()] = [
                    "sku"   => $i->getSku(),
                    "qty"   => 0,
                    "total" => 0,
                ];
            }

            $cart[$i->getSku()]["qty"]   += $i->getQty();
            $cart[$i->getSku()]["total"] += $i->getBaseRowTotal();
        }

        return $cart;
    }

    /**
     * Lists all restricted items the customer has bought the last year.
     *
     * @param  Array<string>
     * @param  string
     * @param  integer
     * @return Array<[sku => string, qty => double, total => double]>
     */
    public function fetchProductQuantities($skus, $orderedAfter, $customerId) {
        if(empty($skus)) {
            return [];
        }

        $res  = Mage::getResourceModel("core/resource");
        $conn = $res->getReadConnection();

        $query = $conn->select()
            ->from([ "o" => $res->getTable("sales/order")], [])
            ->join([ "i" => $res->getTable("sales/order_item")], "o.entity_id = i.order_id", ["sku"])
            ->where("o.customer_id = ?", $customerId)
            ->where("o.created_at >= ?", $orderedAfter)
            ->where("i.sku IN (?)", $skus)
            ->columns([
                "qty"   => new Zend_Db_Expr("SUM(i.qty_ordered)"),
                "total" => new Zend_Db_Expr("SUM(i.base_row_total_incl_tax)"),
            ])
            ->group(["i.sku"]);

        $values = array_values($conn->query($query)->fetchAll());
        $keys   = array_map(function($v) { return $v["sku"]; }, $values);

        return array_combine($keys, $values);
    }
}
