<?php

declare(strict_types=1);

namespace Crossroads\PsalmPluginMagento;

use PhpParser\Node;
use PhpParser\NodeVisitor;
use PhpParser\NodeVisitorAbstract;
use PhpParser\Node\Arg;
use PhpParser\Node\Expr;
use PhpParser\Node\Expr\StaticCall;
use PhpParser\Node\Identifier;
use PhpParser\Node\Name;
use PhpParser\Node\Scalar\String_;
use PhpParser\Node\Stmt;
use PhpParser\Node\VariadicPlaceholder;
use Psalm\DocComment;
use Psalm\Exception\DocblockParseException;
use Psalm\Internal\Codebase\Scanner;

class Visitor extends NodeVisitorAbstract implements NodeVisitor {
    /**
     * @var Scanner
     */
    protected $scanner;

    /**
     * @var string
     */
    protected $filePath;

    public function __construct(Scanner $scanner, string $filePath) {
        $this->scanner = $scanner;
        $this->filePath = $filePath;
    }

    /**
     * @psalm-suppress InternalProperty
     */
    public function leaveNode(Node $node) {
        $docComment = $node->getDocComment();

        if($docComment) {
            try {
                $comment = DocComment::parsePreservingLength($docComment);

                if(isset($comment->tags["psalm-scope-this"])) {
                    $candidate = trim(\reset($comment->tags["psalm-scope-this"]));

                    $this->scanner->queueClassLikeForScanning($candidate);
                }
            }
            catch(DocblockParseException $e) {
                // Intentionally empty
            }
        }

        if($node instanceof StaticCall &&
            $node->class instanceof Name &&
            (string)$node->class === "Mage") {
            $candidate = null;
            $firstArg = $this->getFirstArgValue($node->args);

            if($firstArg instanceof String_) {
                $name = $node->name instanceof Identifier ? $node->name->toString() : null;

                switch($name) {
                case "getModel":
                case "getSingleton":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "model");
                    break;

                case "getResourceHelper":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "resource_helper");
                    break;

                case "getResourceModel":
                    // Append _resource to the model name
                    $items = explode("/", $firstArg->value);
                    $items[0] .= "_resource";

                    $candidate = MagentoClassFinder::findCandidate(implode("/", $items), "model");
                    break;

                case "helper":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "helper");
                    break;
                }

                if($candidate) {
                    $this->scanner->queueClassLikeForScanning($candidate);
                }
            }
        }
    }

    /**
     * @param Array<Arg|VariadicPlaceholder> $args
     */
    private function getFirstArgValue(array $args): ?Expr {
        if(count($args) === 0) {
            return null;
        }

        $first = $args[0];

        return $first instanceof Arg ? $first->value : null;
    }
}
