<?php

declare(strict_types=1);

namespace Crossroads\PsalmPluginMagento;

use PhpParser\Node\Expr;
use PhpParser\Node\Expr\StaticCall;
use PhpParser\Node\Identifier;
use PhpParser\Node\Name;
use PhpParser\Node\Scalar\String_;
use Psalm\Codebase;
use Psalm\Context;
use Psalm\FileManipulation;
use Psalm\Plugin\EventHandler\MethodReturnTypeProviderInterface;
use Psalm\Plugin\EventHandler\Event\MethodReturnTypeProviderEvent;
use Psalm\Plugin\PluginEntryPointInterface;
use Psalm\Plugin\RegistrationInterface;
use Psalm\StatementsSource;
use Psalm\Type\Atomic\TNamedObject;
use Psalm\Type\Union;
use SimpleXMLElement;

/**
 * Finds instances of Mage::getModel, Mage::getSingleton, Mage::getHelper
 * and updates their return types to a more specific type.
 */
class Plugin
    implements
    MethodReturnTypeProviderInterface,
    PluginEntryPointInterface
{
    public function __invoke(RegistrationInterface $psalm, ?SimpleXMLElement $config = null): void {
        $psalm->addStubFile(MagentoClassFinder::findMageFile());
        $psalm->registerHooksFromClass(self::class);
        $psalm->addFileTypeScanner("php", Scanner::class);
    }

    // List of classes we override return types for
    public static function getClassLikeNames(): array {
        return [
            "Mage"
        ];
    }

    public static function getMethodReturnType(MethodReturnTypeProviderEvent $event): ?Union {
        if ($event->getFqClasslikeName() !== "Mage") {
            return null;
        }

        $type = null;
        $args = $event->getCallArgs();
        $firstArg = count($args) > 0 ? $args[0]->value : null;

        if ( ! $firstArg instanceof String_) {
            return null;
        }

        $value = $firstArg->value;

        // TODO: Add more static methods
        switch($event->getMethodNameLowercase()) {
        case "getsingleton":
        case "getmodel":
            $type = "model";
            break;

        case "getresourcehelper":
            $type = "resource_helper";
            break;

        case "getresourcemodel":
            // Append _resource to the model name
            $items = explode("/", $value);
            $items[0] .= "_resource";
            $value = implode("/", $items);
            $type = "model";
            break;

        case "helper":
            $type = "helper";
            break;
        }

        if($type) {
            $candidate = MagentoClassFinder::findCandidate($value, $type);

            if($candidate) {
                return new Union([
                    new TNamedObject($candidate)
                ]);
            }
        }

        return null;
    }
}
