<?php

declare(strict_types=1);

namespace Crossroads\PsalmPluginMagento;

use PhpParser\Node\Expr;
use PhpParser\Node\Expr\StaticCall;
use PhpParser\Node\Identifier;
use PhpParser\Node\Name;
use PhpParser\Node\Scalar\String_;
use Psalm\Codebase;
use Psalm\Context;
use Psalm\FileManipulation;
use Psalm\Plugin\Hook\AfterMethodCallAnalysisInterface;
use Psalm\Plugin\PluginEntryPointInterface;
use Psalm\Plugin\RegistrationInterface;
use Psalm\StatementsSource;
use Psalm\Type\Atomic\TNamedObject;
use Psalm\Type\Union;
use SimpleXMLElement;

/**
 * Finds instances of Mage::getModel, Mage::getSingleton, Mage::getHelper
 * and updates their return types to a more specific type.
 */
class Plugin
    implements
    AfterMethodCallAnalysisInterface,
    PluginEntryPointInterface
{
    public function __invoke(RegistrationInterface $psalm, ?SimpleXMLElement $config = null): void {
        $psalm->addStubFile(MagentoClassFinder::findMageFile());
        $psalm->registerHooksFromClass(self::class);
    }

    /**
     * @param  FileManipulation[] $file_replacements
     *
     * @return void
     */
    public static function afterMethodCallAnalysis(
        Expr $expr,
        string $method_id,
        string $appearing_method_id,
        string $declaring_method_id,
        Context $context,
        StatementsSource $statements_source,
        Codebase $codebase,
        array &$file_replacements = [],
        Union &$return_type_candidate = null
    ): void {
        if ($expr instanceof StaticCall
            && $expr->class instanceof Name
            && (string)$expr->class === "Mage"
        ) {
            $candidate = null;
            $firstArg = count($expr->args) > 0 ? $expr->args[0]->value : null;

            if ($firstArg instanceof String_) {
                $name = $expr->name instanceof Identifier ? $expr->name->toString() : null;

                // TODO: Add more static methods
                switch($name) {
                case "getSingleton":
                case "getModel":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "model");
                    break;

                case "getResourceHelper":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "resource_helper");
                    break;

                case "getResourceModel":
                    // Append _resource to the model name
                    $items = explode("/", $firstArg->value);
                    $items[0] .= "_resource";

                    $candidate = MagentoClassFinder::findCandidate(implode("/", $items), "model");
                    break;

                case "helper":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "helper");
                    break;
                }

                if($candidate) {
                    $return_type_candidate = new Union([new TNamedObject($candidate)]);
                }
            }
        }
    }
}
