<?php

declare(strict_types=1);

namespace Crossroads\PsalmPluginMagento;

class MagentoClassFinder {
    protected static $namespaces = null;

    /**
     * Returns the path to the Mage.php file.
     */
    public static function findMageFile() {
        $root = self::findMagentoRoot();

        return implode(DIRECTORY_SEPARATOR, [$root, "app", "Mage.php"]);
    }

    /**
     * Returns the path to magento root.
     */
    protected static function findMagentoRoot(): ?string {
        $dir = getcwd();

        while( ! file_exists($dir . DIRECTORY_SEPARATOR . "composer.json") && substr_count($dir, DIRECTORY_SEPARATOR) > 1) {
            $dir = dirname($dir);
        }

        $composerJson = $dir . DIRECTORY_SEPARATOR . "composer.json";

        if( ! file_exists($composerJson)) {
            return ".";
        }

        $data = json_decode(file_get_contents($composerJson), true);

        if( ! is_array($data)) {
            return ".";
        }

        if( ! array_key_exists("extra", $data)) {
            return ".";
        }

        if( ! is_array($data["extra"])) {
            return ".";
        }

        return $data["extra"]["magento-root-dir"];
    }

    /**
     * Returns a list of all module-paths registered in the Magento installation $root.
     */
    protected static function findModules(string $root): array {
        $configs = glob(implode(DIRECTORY_SEPARATOR, [$root, "app", "etc", "modules", "*.xml"]));

        return call_user_func_array("array_merge", array_map(function($file) use($root) {
            $xml = simplexml_load_file($file);
            $paths = [];

            foreach($xml->modules->children() as $name => $def) {
                if((string)$def->active === "true") {
                    $paths[] = implode(DIRECTORY_SEPARATOR, [
                        $root,
                        "app",
                        "code",
                        (string)$def->codePool,
                        str_replace("_", DIRECTORY_SEPARATOR, $name),
                    ]);
                }
            }

            return $paths;
        }, $configs));
    }

    /**
     * Returns maps for blocks, helpers and models of identifier -> class prefix for the supplied module config.
     *
     * @return array{blocks:Array<string, string>, helpers:Array<string, string>, models:Array<string, string>}
     */
    protected static function loadModuleConfig(string $module): array {
        $xml = simplexml_load_file(implode(DIRECTORY_SEPARATOR, [$module, "etc", "config.xml"]));
        $cfg = [
            "blocks" => [],
            "helpers" => [],
            "models" => [],
        ];

        foreach(array_keys($cfg) as $area) {
            if(isset($xml->global->{$area})) {
                foreach($xml->global->{$area}->children() as $name => $def) {
                    $class = ((string)$def->class) ?: ((string)$def->model);

                    if( ! empty($class)) {
                        $cfg[$area][$name] = $class;
                    }
                }
            }
        }

        return $cfg;
    }

    /**
     * Retrieves the namespace configuration for all magento module blocks, helpers and models.
     *
     * @return array{blocks:Array<string, string>, helpers:Array<string, string>, models:Array<string, string>}
     */
    public static function getNamespaces() {
        if( ! self::$namespaces) {
            $magentoRoot = self::findMagentoRoot();
            $modules = self::findModules($magentoRoot);
            $configs = array_map(__CLASS__."::loadModuleConfig", $modules);
            self::$namespaces = array_reduce($configs, function($cfg, $module) {
                return array_merge_recursive($cfg, $module);
            }, [
                "blocks" => [],
                "helpers" => [],
                "models" => [],
            ]);
        }

        return self::$namespaces;
    }

    /**
     * Returns a candidate class name for a specified id and type.
     */
    public static function findCandidate(string $id, string $type): string {
        if($type === "helper" && strpos($id, "/") === false) {
            $id .= "/data";
        }

        if($type === "resource_helper") {
            $id = sprintf("%s/resource_helper_%s", $id, "Mysql4");
            $type = "model";
        }

        $parts = explode("/", $id);
        $group = $type."s";

        if(count($parts) === 1) {
            return $parts[0];
        }

        [$ns, $class] = $parts;

        $namespaces = self::getNamespaces();

        if(isset($namespaces[$group][$ns])) {
            return ucwords(self::$namespaces[$group][$ns]."_".$class, "_");
        }

        return ucwords(implode("_", ["mage", $ns, $type, $class]), "_");
    }
}
