<?php

declare(strict_types=1);

namespace Crossroads\PsalmPluginMagento;

use PhpParser\Node;
use PhpParser\NodeVisitor;
use PhpParser\NodeVisitorAbstract;
use PhpParser\Node\Expr\StaticCall;
use PhpParser\Node\Name;
use PhpParser\Node\Scalar\String_;
use PhpParser\Node\Stmt;
use Psalm\DocComment;
use Psalm\Internal\Codebase\Scanner;

class Visitor extends NodeVisitorAbstract implements NodeVisitor {
    /**
     * @var Scanner
     */
    protected $scanner;

    /**
     * @var string
     */
    protected $filePath;

    public function __construct(Scanner $scanner, string $filePath) {
        $this->scanner = $scanner;
        $this->filePath = $filePath;
    }

    public function leaveNode(Node $node) {
        $docComment = $node->getDocComment();

        if($docComment) {
            try {
                $comment = DocComment::parsePreservingLength($docComment);

                if(isset($comment->tags["psalm-scope-this"])) {
                    $candidate = trim(\reset($comment->tags["psalm-scope-this"]));

                    $this->scanner->queueClassLikeForScanning($candidate, $this->filePath);
                }
            }
            catch(DocblockParseException $e) {
                // Intentionally empty
            }
        }

        if($node instanceof StaticCall &&
            $node->class instanceof Name &&
            (string)$node->class === "Mage") {
            $candidate = null;
            $firstArg = count($node->args) > 0 ? $node->args[0]->value : null;

            if($firstArg instanceof String_) {
                switch((string)$node->name) {
                case "getModel":
                case "getSingleton":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "model");
                    break;

                case "getResourceHelper":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "resource_helper");
                    break;

                case "helper":
                    $candidate = MagentoClassFinder::findCandidate($firstArg->value, "helper");
                    break;
                }

                if($candidate) {
                    $this->scanner->queueClassLikeForScanning($candidate, $this->filePath);
                }
            }
        }
    }
}
