<?php

declare(strict_types=1);

namespace Crossroads\PsalmPluginMagento;

use Exception;
use SimpleXMLElement;

/**
 * @psalm-type NamespaceData array{ blocks: Array<string, string>, helpers: Array<string, string>, models: Array<string, string> }
 */
class MagentoClassFinder {
    /**
     * @var ?NamespaceData
     */
    protected static $namespaces = null;

    /**
     * Returns the path to the Mage.php file.
     */
    public static function findMageFile(): string {
        $root = self::findMagentoRoot();

        return implode(DIRECTORY_SEPARATOR, [$root, "app", "Mage.php"]);
    }

    /**
     * Returns the path to magento root.
     */
    protected static function findMagentoRoot(): string {
        $dir = getcwd();

        while( ! file_exists($dir . DIRECTORY_SEPARATOR . "composer.json") && substr_count($dir, DIRECTORY_SEPARATOR) > 1) {
            $dir = dirname($dir);
        }

        $composerJson = $dir . DIRECTORY_SEPARATOR . "composer.json";

        if( ! file_exists($composerJson)) {
            return ".";
        }

        $data = json_decode(file_get_contents($composerJson), true);

        if( ! is_array($data)) {
            return ".";
        }

        if( ! array_key_exists("extra", $data)) {
            return ".";
        }

        if( ! is_array($data["extra"])) {
            return ".";
        }

        return (string)($data["extra"]["magento-root-dir"] ?? ".");
    }

    /**
     * Returns a list of all module-paths registered in the Magento installation $root.
     *
     * @return Array<string>
     */
    protected static function findModules(string $root): array {
        $configs = glob(implode(DIRECTORY_SEPARATOR, [$root, "app", "etc", "modules", "*.xml"]));

        return array_merge(...array_map(function(string $file) use($root): array {
            $xml = self::loadXml($file);
            $paths = [];

            if(!$xml->modules instanceof SimpleXMLElement) {
                return [];
            }

            /**
             * @var SimpleXmlElement $def
             */
            foreach($xml->modules->children() ?? [] as $name => $def) {
                if($def && (string)$def->active === "true") {
                    $paths[] = implode(DIRECTORY_SEPARATOR, [
                        $root,
                        "app",
                        "code",
                        (string)$def->codePool,
                        str_replace("_", DIRECTORY_SEPARATOR, (string)$name),
                    ]);
                }
            }

            return $paths;
        }, $configs));
    }

    /**
     * Returns maps for blocks, helpers and models of identifier -> class prefix for the supplied module config.
     *
     * @return NamespaceData
     */
    protected static function loadModuleConfig(string $module): array {
        $xml = self::loadXml(implode(DIRECTORY_SEPARATOR, [$module, "etc", "config.xml"]));
        /**
         * @var NamespaceData $cfg
         */
        $cfg = [
            "blocks" => [],
            "helpers" => [],
            "models" => [],
        ];

        if(!$xml->global instanceof SimpleXMLElement) {
            return $cfg;
        }

        foreach(array_keys($cfg) as $area) {
            /**
             * @var mixed $group
             */
            $group = $xml->global->{$area} ?? null;

            if($group instanceof SimpleXMLElement) {
                /**
                 * @var SimpleXMLElement $def
                 */
                foreach($group->children() ?? [] as $name => $def) {
                    $class = ((string)$def->class) ?: ((string)$def->model);

                    if( ! empty($class)) {
                        $cfg[$area][(string)$name] = $class;
                    }
                }
            }
        }

        return $cfg;
    }

    /**
     * Retrieves the namespace configuration for all magento module blocks, helpers and models.
     *
     * @return NamespaceData
     */
    public static function getNamespaces() {
        if( ! self::$namespaces) {
            $magentoRoot = self::findMagentoRoot();
            $modules = self::findModules($magentoRoot);
            $configs = array_map(__CLASS__."::loadModuleConfig", $modules);
            /**
             * @var NamespaceData $data
             */
            $data = array_reduce($configs, function(array $cfg, array $module): array {
                return array_merge_recursive($cfg, $module);
            }, [
                "blocks" => [],
                "helpers" => [],
                "models" => [],
            ]);

            self::$namespaces = $data;
        }

        return self::$namespaces;
    }

    /**
     * Returns a candidate class name for a specified id and type.
     */
    public static function findCandidate(string $id, string $type): string {
        if($type === "helper" && strpos($id, "/") === false) {
            $id .= "/data";
        }

        if($type === "resource_helper") {
            $id = sprintf("%s/resource_helper_%s", $id, "Mysql4");
            $type = "model";
        }

        $parts = explode("/", $id);
        $group = $type."s";

        if(count($parts) === 1) {
            return $parts[0];
        }

        [$ns, $class] = $parts;

        $namespaces = self::getNamespaces();

        if(isset($namespaces[$group][$ns])) {
            return ucwords($namespaces[$group][$ns]."_".$class, "_");
        }

        // Special case for core resource models, Mage_Core does not have them
        // mapped in XML
        if($type === "model" && $ns === "core_resource") {
            $ns = "core";
            $class = "resource_".$class;
        }

        return ucwords(implode("_", ["mage", $ns, $type, $class]), "_");
    }

    private static function loadXml(string $file): SimpleXMLElement {
        $xml = simplexml_load_file($file);

        if($xml === false) {
            $error = libxml_get_last_error();

            libxml_clear_errors();

            throw new Exception(sprintf(
                "XMLError(%s:%d): %s",
                $error->file ?: $file,
                $error->line,
                $error->message
            ));
        }

        return $xml;
    }
}
