#!/usr/bin/env bash

# TODO:
# * Deduplicate container boot/run logic

trap exit INT
# Error on command errors, missing variables, and make pipes fail if any of the involved commands fail
set -euo pipefail
# Required to make *.sql glob return empty array instead of literal string
shopt -s nullglob

SCRIPT_NAME=$(basename "$0")

if [ "${BASH_VERSINFO[0]}" -lt 4 ] || ([ "${BASH_VERSINFO[0]}" -eq 4 ] && [ "${BASH_VERSINFO[1]}" -lt 4 ]); then
	# We need associative arrays, as well as the ${ARRAY[@]} syntax
	echo "$SCRIPT_NAME: Bash version 4.4 or greater is required"

	exit -1
fi

if [ -f .env ]; then
	source .env
fi

if ! command -v docker &> /dev/null; then
	echo "$SCRIPT_NAME: Docker CLI is required"

	exit -1
fi

function version { echo "$@" | awk -F. '{ printf("%d%03d%03d%03d\n", $1,$2,$3,$4); }'; }

DOCKER_VERSION=$(docker version --format '{{.Client.Version}}')

if [ $(version $DOCKER_VERSION) -lt $(version "17.05.0") ]; then
	echo "$SCRIPT_NAME: Docker client version 17.05.0 required, installed: $DOCKER_VERSION"

	exit -1
fi

DEBUG=
# Missing in Bash
GID=$(id -g)

# Registry for our custom images, set to empty to load them from a local copy
DOCKER_REGISTRY=${DOCKER_REGISTRY-groot-registry.crossroads.se/crossroads/modules/}
COMPOSER_IMAGE=${COMPOSER_IMAGE:-composer:2}
COMPOSER_HOME=${COMPOSER_HOME:-$HOME/.config/composer}

PROJECT_NAME=${PROJECT_NAME:-project-$(basename "$PWD" | tr '[:upper:]' '[:lower:]')}
NETWORK=$PROJECT_NAME-network

DB_CONTAINER=$PROJECT_NAME-mariadb
DATABASE_NAME=${DATABASE_NAME:-testing}
DB_DSN="mysql://root@$DB_CONTAINER:3306/$DATABASE_NAME"

FPM_CONTAINER=$PROJECT_NAME-fpm
MAGE_RUN_CODE=${MAGE_RUN_CODE:-default}
MAGENTO_CRYPT_KEY=${MAGENTO_CRYPT_KEY:-testing}
MAGENTO_VAR_FOLDER=${MAGENTO_VAR_FOLDER:-$PWD/var}

NGINX_HOST=${NGINX_HOST:-localhost}
NGINX_CONTAINER=$PROJECT_NAME-nginx
NGINX_PORT=${NGINX_PORT:-8080}

declare -A IMAGES=(
	[php-cli]="${DOCKER_REGISTRY}magento-lts/php:7.4-dev-cli"
	[php-cov]="${DOCKER_REGISTRY}magento-lts/php:7.4-dev-cov"
	[php-fpm]="${DOCKER_REGISTRY}magento-lts/php:7.4-dev-fpm"
	[php-n98]="${DOCKER_REGISTRY}magento-lts/php:7.4-dev-n98"
)

# Parameters to run Docker-images with the application as a volume
DOCKER_PARAMS=(
# We need to set UID and GID to allow is_writable to work
	-u "$UID:$GID"
	--rm
	-it
	-v "$PWD:/app"
	-w /app
)

PHP_PARAMS=(
	"${DOCKER_PARAMS[@]}"
# --init for an init wrapper to manage INT
	--init
	--tmpfs /tmp:rw
	--read-only
)

PHPUNIT_PARAMS=(
	"${PHP_PARAMS[@]}"
	--network="$NETWORK"
	-e RESOURCE_DEFAULT_SETUP="$DB_DSN"
	-e MAGENTO_CRYPT_KEY="$MAGENTO_CRYPT_KEY"
)

N98_MAGERUN_PARAMS=(
	"${PHP_PARAMS[@]}"
	--network="$NETWORK"
	-e RESOURCE_DEFAULT_SETUP="$DB_DSN"
	-e MAGENTO_CRYPT_KEY="$MAGENTO_CRYPT_KEY"
	--read-only
	"${IMAGES[php-n98]}"
)

DATABASE_VOLUMES=()

for f in "$PWD"/test/*.sql; do
	DATABASE_VOLUMES+=(-v "$f:/docker-entrypoint-initdb.d/$(basename "$f"):ro")
done

# Quicker health check since it is a test
# health check must use IP to avoid socket since socket is ready before TCP/IP
DATABASE_PARAMS=(
	-e MYSQL_ALLOW_EMPTY_PASSWORD=yes
	-e MYSQL_DATABASE="$DATABASE_NAME"
	--tmpfs /var/lib/mysql:rw
	--tmpfs /tmp:rw
	--network="$NETWORK"
	--health-cmd='mysql -h 127.0.0.1 -e "show databases;"'
	--health-interval=2s
	"${DATABASE_VOLUMES[@]}" mariadb
)

# Use fcgi to check if it is alive, should return status once running if
# pm.status_path = /status is set properly
FPM_PARAMS=(
	"${DOCKER_PARAMS[@]}"
	-e RESOURCE_DEFAULT_SETUP="$DB_DSN"
	-e MAGENTO_CRYPT_KEY="$MAGENTO_CRYPT_KEY"
	--init
	--network="$NETWORK"
	--name "$FPM_CONTAINER"
	--health-interval=2s
	"${IMAGES[php-fpm]}"
)

NGINX_VOLUMES=()

for f in "$PWD"/dev/nginx/*; do
	NGINX_VOLUMES+=(-v "$f:/etc/nginx/$(basename "$f"):ro")
done

NGINX_PARAMS=(
	--network="$NETWORK"
	-e PHP_FPM_HOST="$FPM_CONTAINER"
	-e MAGE_RUN_CODE="$MAGE_RUN_CODE"
	-v "$PWD:/app:ro"
	"${NGINX_VOLUMES[@]}"
	-p "$NGINX_PORT:80"
	--health-cmd="curl -Iq http://127.0.0.1:80"
	--health-interval=5s
	nginx:alpine
)

# TODO: Probably not needed anymore
create_image() {
	local image="${IMAGES[$1]}"
	local output=()

	if [ -n "$DEBUG" ]; then
		output=(--progress=plain)
	fi

	if [ -z "$image" ]; then
		echo "Unknown image $1" 1>&2; exit 1;
	fi

	if ! docker image inspect "$image" >/dev/null 2>&1; then
		# Pipe Dockerfile to avoid context
		docker build "${output[@]}" --target "$1" -t "$image" - < ./Dockerfile
	fi
}

is_container_running() {
	[ "$( docker container inspect -f '{{.State.Running}}' "$1" 2>/dev/null )" == "true" ]
}

start_container() {
	docker run -d --rm --name "$@"

	while [ "$(docker inspect --format "{{.State.Health.Status }}" "$1")" != "healthy" ]; do
		printf "."
		sleep 1
	done

	echo ""
}

help() {
	echo "Usage: $SCRIPT_NAME [-v] <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    composer   Run composer"
	echo "    test       Run tests"
	echo "    psalm      Run psalm"
	echo "    phpunit    Run phpunit"
	echo "    coverage   Run phpunit and record code coverage"
	echo "    syntax     Run PHP syntax check"
	echo "    php        Run php"
	echo "    network    Manage the test network"
	echo "    database   Manage the test database container"
	echo "    fpm        Manage the test PHP-FPM container"
	echo "    nginx      Manage the test Nginx container"
	echo "    start      Starts all containers"
	echo "    stop       Stops all containers, removes all networks"
	echo "               and clears the cache"
	echo "    redeploy   Redeploys all magento modules (copies/symlinks)"
	echo "    clearcache Clears the magento cache"
	echo "    clean      Removes containers, images, networks, unused volumes"
	echo "    distclean  Removes containers, images, networks, unused volumes,"
	echo "               coverage, vendor, and magento folders, composer.lock"
	echo ""
	echo "Options:"
	echo "    -v --verbose  Activates verbose print of containers"
	echo ""
	echo "Environment Variables:"
	echo "    PROJECT_NAME    The basename used for containers and images"
	echo "        default: project-\$DIR"
	echo "    MAGE_RUN_CODE   The run code forwarded to Nginx/FPM"
	echo "        default: magento"
	echo "    COMPOSER_IMAGE  Docker image for Composer"
	echo "        default: composer:2"
	echo "    NGINX_PORT      Local port mapped to Nginx container port 80"
	echo "        default: 8080"
	echo "    DATABASE_NAME   MariaDB database name to use"
	echo "        default: testing"
	echo "    DOCKER_REGISTRY Registry prefix for custom docker images,"
	echo "        set to empty string to use locally built images"
	echo "        default: groot-registry.crossroads.se/crossroads/modules/"
	echo ""
	echo "If a .env file exists in the current working directory this will be"
	echo "loaded and populate environment variables for the command."
	echo ""
	echo "For help with each subcommand run:"
	echo "$SCRIPT_NAME <subcommand> -h|--help"
	echo ""
}

syntax() {
	case "${1:-}" in
		"-h" | "--help")
			echo "Usage: $SCRIPT_NAME [-a]"
			echo ""
			echo "Options:"
			echo "    -a --all  Runs PHP syntax check on all .php files in the"
			echo "              current folder including files outside source control"
			echo ""

			;;
		"-a" | "--all")
			# --line-buffered is used to produce immediate output
			# -i only since we pipe input but do not need a TTY
			# && exit 1 || [ "$?" -eq "1" ] is used to produce ok exit code on no
			# output, and error on output
			find . -type f -name '*.php' | docker run --rm --init -i \
				-v "$PWD:/app:ro" -w /app \
				"${IMAGES[php-cli]}" sh -c "xargs -i sh -c \" php -l '{}' || true\"" | \
				grep -vE --line-buffered '^No syntax errors|^Errors parsing|^[[:space:]]*$' && \
				exit 1 || [ "$?" -eq "1" ]
			;;
		*)
			git ls-files '**.php' | docker run --rm --init -i \
				-v "$PWD:/app:ro" -w /app \
				"${IMAGES[php-cli]}" sh -c "xargs -i sh -c \"php -l {} || true\"" | \
				grep -vE --line-buffered '^No syntax errors|^Errors parsing|^[[:space:]]*$' && \
				exit 1 || [ "$?" -eq "1" ]
			;;
	esac
}

psalm() {
	docker run "${PHP_PARAMS[@]}" "${IMAGES[php-cli]}" php vendor/bin/psalm --show-info=true "$@"
}

phpunit() {
	database_start
	clearcache

	docker run "${PHPUNIT_PARAMS[@]}" "${IMAGES[php-cli]}" php vendor/bin/phpunit "$@"
}

n98() {
	database_start
	clearcache

	docker run "${N98_MAGERUN_PARAMS[@]}" "$@"
}

init-magento() {
	clearcache

	n98 sys:setup:run
	n98 admin:user:create test test@example.com test Test Testsson || true

	clearcache

	n98 config:set web/secure/base_url http://$NGINX_HOST:$NGINX_PORT/
	n98 config:set web/unsecure/base_url http://$NGINX_HOST:$NGINX_PORT/
	n98 config:set web/seo/use_rewrites 1
	n98 config:set web/cookie/cookie_httponly 0
	n98 config:set web/cookie/frontend_namespace dev_frontend

	clearcache
}

coverage() {
	database_start
	clearcache

	docker run "${PHPUNIT_PARAMS[@]}" "${IMAGES[php-coverage]}" php -d pcov.enabled=1 -d memory_limit=2048M vendor/bin/phpunit --coverage-html coverage
}

php() {
	docker run "${PHP_PARAMS[@]}" "${IMAGES[php-cli]}" php "$@"
}

test() {
	psalm
	phpunit
	syntax
}

start() {
	clearcache
	nginx_start
}

stop() {
	nginx_stop
	fpm_stop
	database_stop
	network_stop
	clearcache
}

clean() {
	stop

	rm -rf .psalm .phpunit.result.cache

	docker image rm -f "${IMAGES[@]}" 2> /dev/null || true

	for v in $(docker volume ls -qf dangling=true); do
		docker volume rm "$v"
	done
}

clearcache() {
	rm -rf "$MAGENTO_VAR_FOLDER/cache/*"
}

database() {
	database_help
}

database_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    info  Shows information about database host and port"
	echo "    query Opens up a mysql query prompt"
	echo "    start Starts the database container for testing"
	echo "    stop  Stops the database container for testing"
	echo "    run   Runs the database container interactively, useful"
	echo "          for debugging purposes"
	echo "    log   Shows the logs for database container for testing"
	echo ""
}

database_info() {
	echo "$DB_DSN"
}

database_query() {
	network_start
	database_start

	docker exec -it "$DB_CONTAINER" mysql -D "$DATABASE_NAME"
}

database_run() {
	network_start

	docker run -it --rm --name "$DB_CONTAINER" "${DATABASE_PARAMS[@]}"
}

database_start() {
	if is_container_running "$DB_CONTAINER"; then
		return
	fi

	network_start

	start_container "$DB_CONTAINER" "${DATABASE_PARAMS[@]}"
}

database_log() {
	docker logs "$DB_CONTAINER" "$@"
}

database_stop() {
	docker rm -f "$DB_CONTAINER" 2>/dev/null || true
}

network() {
	network_help
}

network_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    start Starts the network interface"
	echo "    stop  Stops the network interface"
	echo ""
}

network_start() {
	if docker network inspect -f '{{.Name}}' "$NETWORK" >/dev/null 2>&1; then
		return
	fi

	docker network create -d bridge "$NETWORK" >/dev/null
}

network_stop() {
	docker network rm "$NETWORK" 2>/dev/null || true
}

fpm() {
	fpm_help
}

fpm_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    start  Starts the PHP FPM container"
	echo "    stop   Stops the PHP FPM container"
	echo "    clean  Stops and removes the PHP FPM container and image"
	echo "    run    Runs the PHP FPM container interactively, useful"
	echo "           for debugging purposes"
	echo "    log    Shows the logs for the PHP FPM container"
	echo "    attach Opens a shell session in the running PHP FPM container"
	echo ""
}

fpm_run() {
	database_start
	network_start
	clearcache

	docker run "${FPM_PARAMS[@]}"
}

fpm_start() {
	if is_container_running "$FPM_CONTAINER"; then
		return
	fi

	database_start
	network_start
	clearcache

	start_container "$FPM_CONTAINER" "${FPM_PARAMS[@]}"
}

fpm_stop() {
	docker rm -f "$FPM_CONTAINER" 2>/dev/null || true
}

fpm_clean() {
	fpm_stop

	docker image rm -f "${IMAGES[php-fpm]}" 2> /dev/null || true
}

fpm_log() {
	docker logs "$FPM_CONTAINER" "$@"
}

fpm_attach() {
	docker exec -it "$FPM_CONTAINER" /bin/sh
}

nginx() {
	nginx_help
}

nginx_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    start  Starts the Nginx container"
	echo "    stop   Stops the Nginx container"
	echo "    log    Shows the logs for the Nginx container"
	echo "    attach Opens a shell session in the running nginx container"
	echo ""
}

nginx_run() {
	network_start
	fpm_start

	docker run --rm -it --name "$NGINX_CONTAINER" "${NGINX_PARAMS[@]}"
}

nginx_start() {
	if is_container_running "$NGINX_CONTAINER"; then
		return
	fi

	network_start
	fpm_start

	start_container "$NGINX_CONTAINER" "${NGINX_PARAMS[@]}"
}

nginx_stop() {
	docker rm -f "$NGINX_CONTAINER" 2>/dev/null || true
}

nginx_log() {
	docker logs "$NGINX_CONTAINER" "$@"
}

nginx_attach() {
	docker exec -it "$NGINX_CONTAINER" /bin/sh
}

composer() {
	mkdir -p $COMPOSER_HOME

	# We need to concatenate the composer command first, so we later can quote
	# everything for nested execution
	ARGS="composer ${@@Q}"
	GROUP_NAME=$(id -g -n)
	# Special here since we need to start as root to be able to create the
	# necessary uid/gid for SSH.
	COMPOSER_PARAMS=(
		-v "$PWD:/app"
		-w /app
		# Composer home mapped to tmp
		-v "$COMPOSER_HOME:/tmp"
		# .ssh volume to be able to use git in composer
		-v "$HOME/.ssh:/home/$USER/.ssh:ro"
		# No entrypoint since composer is trying to start composer anyway
		--entrypoint=""
		"$COMPOSER_IMAGE"
		# We have to run commands to create the necessary user and group before
		# we invoke composer, this is required for SSH to work on machines where
		# the host /etc/passwd and/or /etc/group files are not suitable to mount
		# as volumes (eg. MacOS or Windows).
		# This also necessitates multiple command wrappings.
		# First we also have to make sure to delete any existing group with the
		# same name if any exists, MacOS likes to create user groups with low
		# group ids.
		sh -c "awk -F: '{if(\$3 == $GID)print \$1}' /etc/group | xargs -r -n1 delgroup &&
addgroup -g $GID $GROUP_NAME && \
adduser --disabled-password --gecos '' --home '/home/$USER' --ingroup '$GROUP_NAME' --no-create-home --uid $UID '$USER' && \
su $USER -c ${ARGS@Q}"
	)

	docker run --rm -it "${COMPOSER_PARAMS[@]}"
}

redeploy() {
	composer run-script post-install-cmd -vvv -- --redeploy
}

distclean() {
	clean

	rm -rf coverage vendor composer.lock
}

main() {
	local command=""

	while [ $# -ge 1 ]; do
		case "$1" in
			"" | "-h" | "--help")
				command="${command}help_"
				break;
				;;
			"-v" | "--verbose")
				DEBUG="true"
				;;
			--)
				shift
				break
				;;
			*)
				if declare -f "${command}$1" > /dev/null; then
					SCRIPT_NAME="$SCRIPT_NAME $1"
					command="${command}$1_"
				else
					break
				fi
				;;
		esac

		shift
	done

	if [ -z "$command" ]; then
		if [ $# -ge 1 ]; then
			echo "Error: '$SCRIPT_NAME $1' is not a known subcommand." >&2
			echo "" >&2
		fi

		"${command}help" >&2

		exit -1
	fi

	if [ -n "$DEBUG" ]; then
		set -x
	fi

	"${command%%_}" "$@"
}

main "$@"