<?php

declare(strict_types=1);

namespace Points\Core\Issues;

use function Fixtures\loadPoints;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\TaxRates;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Group;
use Points\Core\Base;
use Points\Core\Extension\Quote;
use Points\Core\ProviderInterface;
use Throwable;
use Varien_Object;

require_once __DIR__."/../Base.php";
require_once __DIR__."/../../Fixtures/Points.php";
require_once __DIR__."/../../../vendor/crossroads/mageql-catalog/test/Fixtures/TaxRates.php";

/**
 * Not a completely 100% coverage of this issue, but it seems to arise when the following happens:
 *
 * * Configurable product
 * * The configurable product does not have tax
 * * Minimum amount of points for the configurable product is equal to its cost
 * * Another product with points payment is present
 */
class MinGreaterPrice extends Base {
    public function setUp(): void {
        loadPoints();

        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearQuotes();

        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $simple = Mage::getModel("catalog/product");
        $simplePrice = Mage::getModel("points_core/product_price");

        $simple->setStoreId($store->getId());
        $simple->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $simplePrice->loadByStoreProductTypeCustomerGroupId($store, $simple, "TEST", Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        $simplePrice->addData([
            "store_id" => $store->getId(),
            "product_id" => $simple->getId(),
            "customer_group_id" => Mage_Customer_Model_Group::NOT_LOGGED_IN_ID,
            "type" => "TEST",
            "price" => 1000,
            "min_price" => 1500,
            "max_price" => null,
        ]);
        $simplePrice->save();

        Mage::getSingleton("points_core/product_index")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        loadPoints();

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    public function testQuote(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        /**
         * @var Quote $quote
         */
        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(1000);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    requiresPointPayment
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                requiresPointPayment
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }
}
