<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Exception;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Customer;
use Mage_Customer_Model_Group;
use Mage_Sales_Model_Quote;
use Mage_Sales_Model_Quote_Item;
use Mage_Tax_Model_Config;
use Points_Core_Helper_Data;
use Throwable;
use Varien_Object;
use Varien_Profiler;

use Points\Core\Extension\Quote;
use Points\Core\Extension\QuoteAddressItem;
use Points\Core\Currency;
use Points\Core\Points;
use Points\Core\ProviderInterface;
use Points\Core\Total\Calculator;
use Points\Core\Total\Item as ItemTotal;
use Points\Core\Total\Shipping as ShippingTotal;
use Points\Core\Total\SpreadPoints;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\Customer;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/../Fixtures/Points.php";

// FIXME: Reimplement with Total\Calculator
class TotalsTest extends TestCase {
    use MatchesSnapshots;

    const SIMPLE_INCL_TAX = 12.34;
    const VIRTUAL_INCL_TAX = 9.99;
    const SHIPPING_INCL_TAX = 13.37;

    public static function setUpBeforeClass(): void {
        loadPoints();
    }

    public function setUp(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");
        MagentoManager::initAdmin();

        // Got to reset these
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->deleteConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, "websites");
        $setupModel->deleteConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, "websites");

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    /**
     * @return array{quote:Quote, item:QuoteAddressItem}
     */
    public function simpleShippingQuote(): array {
        /**
         * @var Quote
         */
        $quote = Mage::getModel("sales/quote");

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        /**
         * @var QuoteAddressItem|string We are cheating a bit here
         */
        $item = $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($item)) {
            throw new Exception($item);
        }

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return [
            "quote" => $quote,
            "item" => $item,
        ];
    }

    public function testSimple(): void {
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->simpleShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        // We have it including tax in the total
        $this->assertEquals([
            new ItemTotal($item, new SpreadPoints(
                new Amount(1332, true, 267),
                new Amount(0, true, 0),
                new Amount(1332, true, 267),
                new Amount(0, true, 0)
            )),
            new ShippingTotal($shippingAddress, null),
        ], $shippingTotals);

        $totalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $discountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(12.34, true, 2.47),
            new Amount(13.37, true, 2.67),
        ], $totalAmounts);
        $this->assertEquals([
            new Amount(0.0, true, 0.0),
            new Amount(0.0, true, 0.0),
        ], $discountAmounts);

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $billingTotal->getTotals());
        $this->assertEquals(25.71, $quote->getGrandTotal());
    }

    public function testSimplePriceNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->simpleShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($item, new SpreadPoints(
                new Amount(1332, false, 333),
                new Amount(0, false, 0),
                new Amount(1332, false, 333),
                new Amount(0, false, 0)
            )),
            new ShippingTotal($shippingAddress, null),
        ], $shippingTotals);

        $totalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $discountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(12.34, false, 3.09),
            new Amount(13.37, false, 3.34),
        ], $totalAmounts);
        $this->assertEquals([
            new Amount(0.0, false, 0.0),
            new Amount(0.0, false, 0.0),
        ], $discountAmounts);

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $billingTotal->getTotals());
        $this->assertEquals(32.14, $quote->getGrandTotal());
    }

    /**
     * @return array{quote:Quote, item:QuoteAddressItem}
     */
    public function virtualQuote(): array {
        /**
         * @var Quote
         */
        $quote = Mage::getModel("sales/quote");

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        /**
         * @var QuoteAddressItem|string
         */
        $item = $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($item)) {
            throw new Exception($item);
        }

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return [
            "quote" => $quote,
            "item" => $item,
        ];
    }

    public function testVirtual(): void {
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->virtualQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($item, new SpreadPoints(
                new Amount(1999, true, 400),
                new Amount(499, true, 100),
                new Amount(1499, true, 300),
                new Amount(0, true, 0)
            )),
        ], $billingTotals);

        $totalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $discountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([
            new Amount(9.99, true, 2.0),
        ], $totalAmounts);
        $this->assertEquals([
            new Amount(0.0, true, 0.0),
        ], $discountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $shippingTotal->getTotals());
        $this->assertEquals(9.99, $quote->getGrandTotal());
    }

    public function testVirtualNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->virtualQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($item, new SpreadPoints(
                new Amount(1999, false, 500),
                new Amount(499, false, 125),
                new Amount(1499, false, 375),
                new Amount(0, false, 0)
            )),
        ], $billingTotals);

        $totalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $discountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([
            new Amount(9.99, false, 2.5),
        ], $totalAmounts);
        $this->assertEquals([
            new Amount(0.0, false, 0.0),
        ], $discountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $shippingTotal->getTotals());
        $this->assertEquals(12.49, $quote->getGrandTotal());
    }

    /**
     * @return array{
     *   quote:Quote,
     *   simple:QuoteAddressItem,
     *   virtual:QuoteAddressItem
     * }
     */
    public function simpleVirtualShippingQuote(): array {
        /**
         * @var Quote
         */
        $quote = Mage::getModel("sales/quote");

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        /**
         * @var QuoteAddressItem|string We are cheating a bit here
         */
        $simpleItem = $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($simpleItem)) {
            throw new Exception($simpleItem);
        }

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        /**
         * @var QuoteAddressItem|string We are cheating a bit here
         */
        $virtualItem = $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($virtualItem)) {
            throw new Exception($virtualItem);
        }

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return [
            "quote" => $quote,
            "simple" => $simpleItem,
            "virtual" => $virtualItem,
        ];
    }

    public function testSimpleVirtual(): void {
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([], $billingTotals);

        $billingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $billingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([], $billingTotalAmounts);
        $this->assertEquals([], $billingDiscountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($simple, new SpreadPoints(
                new Amount(1332, true, 267),
                new Amount(0, true, 0),
                new Amount(1332, true, 267),
                new Amount(0, true, 0)
            )),
            new ItemTotal($virtual, new SpreadPoints(
                new Amount(1999, true, 400),
                new Amount(499, true, 100),
                new Amount(1499, true, 300),
                new Amount(0, true, 0)
            )),
            new ShippingTotal($shippingAddress, null),
        ], $shippingTotals);

        $shippingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $shippingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(12.34, true, 2.47),
            new Amount(9.99, true, 2.0),
            new Amount(13.37, true, 2.67),
        ], $shippingTotalAmounts);
        $this->assertEquals([
            new Amount(0.0, true, 0.0),
            new Amount(0.0, true, 0.0),
            new Amount(0.0, true, 0.0),
        ], $shippingDiscountAmounts);

        $this->assertEquals(35.7, $quote->getGrandTotal());
    }

    public function testSimpleVirtualNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([], $billingTotals);

        $billingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $billingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([], $billingTotalAmounts);
        $this->assertEquals([], $billingDiscountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($simple, new SpreadPoints(
                new Amount(1332, false, 333),
                new Amount(0, false, 0),
                new Amount(1332, false, 333),
                new Amount(0, false, 0)
            )),
            new ItemTotal($virtual, new SpreadPoints(
                new Amount(1999, false, 500),
                new Amount(499, false, 125),
                new Amount(1499, false, 375),
                new Amount(0, false, 0)
            )),
            new ShippingTotal($shippingAddress, null),
        ], $shippingTotals);

        $shippingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $shippingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(12.34, false, 3.09),
            new Amount(9.99, false, 2.49),
            new Amount(13.37, false, 3.35),
        ], $shippingTotalAmounts);
        $this->assertEquals([
            new Amount(0.0, false, 0.0),
            new Amount(0.0, false, 0.0),
            new Amount(0.0, false, 0.0),
        ], $shippingDiscountAmounts);

        $this->assertEquals(44.63, $quote->getGrandTotal());
    }

    public function testSimpleVirtualShipping(): void {
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([], $billingTotals);

        $billingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $billingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([], $billingTotalAmounts);
        $this->assertEquals([], $billingDiscountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($simple, new SpreadPoints(
                new Amount(1332, true, 267),
                new Amount(0, true, 0),
                new Amount(1332, true, 267),
                new Amount(0, true, 0)
            )),
            new ItemTotal($virtual, new SpreadPoints(
                new Amount(1999, true, 400),
                new Amount(499, true, 100),
                new Amount(1499, true, 300),
                new Amount(0, true, 0)
            )),
            new ShippingTotal($shippingAddress, new SpreadPoints(
                new Amount(99, true, 19),
                new Amount(0, true, 0),
                new Amount(99, true, 19),
                new Amount(0, true, 0),
            )),
        ], $shippingTotals);

        $shippingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $shippingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(12.34, true, 2.47),
            new Amount(9.99, true, 2.0),
            new Amount(13.37, true, 2.67),
        ], $shippingTotalAmounts);
        $this->assertEquals([
            new Amount(0.0, true, 0.0),
            new Amount(0.0, true, 0.0),
            new Amount(0.0, true, 0.0),
        ], $shippingDiscountAmounts);

        $this->assertEquals(35.7, $quote->getGrandTotal());
    }

    public function testSimpleVirtualShippingNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([], $billingTotals);

        $billingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $billingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([], $billingTotalAmounts);
        $this->assertEquals([], $billingDiscountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($simple, new SpreadPoints(
                new Amount(1332, false, 333),
                new Amount(0, false, 0),
                new Amount(1332, false, 333),
                new Amount(0, false, 0)
            )),
            new ItemTotal($virtual, new SpreadPoints(
                new Amount(1999, false, 500),
                new Amount(499, false, 125),
                new Amount(1499, false, 375),
                new Amount(0, false, 0)
            )),
            new ShippingTotal($shippingAddress, new SpreadPoints(
                new Amount(99, false, 25),
                new Amount(0, false, 0),
                new Amount(99, false, 25),
                new Amount(0, false, 0),
            )),
        ], $shippingTotals);

        $shippingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $shippingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(12.34, false, 3.09),
            new Amount(9.99, false, 2.49),
            new Amount(13.37, false, 3.35),
        ], $shippingTotalAmounts);
        $this->assertEquals([
            new Amount(0.0, false, 0.0),
            new Amount(0.0, false, 0.0),
            new Amount(0.0, false, 0.0),
        ], $shippingDiscountAmounts);

        $this->assertEquals(44.63, $quote->getGrandTotal());
    }

    public function testSimpleVirtualShippingDiscount(): void {
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $calculator = new Calculator();

        $billingTotal = $calculator->fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $billingTotals = $billingTotal->getTotals();

        $this->assertEquals([], $billingTotals);

        $billingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $billingTotals);
        $billingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $billingTotals);

        $this->assertEquals([], $billingTotalAmounts);
        $this->assertEquals([], $billingDiscountAmounts);

        $shippingTotal = $calculator->fromQuoteAddress($shippingAddress, "TEST", $mockProvider);
        $shippingTotals = $shippingTotal->getTotals();

        $this->assertEquals([
            new ItemTotal($simple, new SpreadPoints(
                new Amount(666, true, 133),
                new Amount(0, true, 0),
                new Amount(666, true, 133),
                new Amount(666, true, 134)
            )),
            new ItemTotal($virtual, new SpreadPoints(
                new Amount(999, true, 200),
                new Amount(250, true, 50),
                new Amount(749, true, 150),
                new Amount(1001, true, 200)
            )),
            new ShippingTotal($shippingAddress, new SpreadPoints(
                new Amount(99, true, 20),
                new Amount(0, true, 0),
                new Amount(99, true, 20),
                new Amount(0, true, 0),
            )),
        ], $shippingTotals);

        $shippingTotalAmounts = array_map(function($t) { return $t->getPrice(); }, $shippingTotals);
        $shippingDiscountAmounts = array_map(function($t) { return $t->getDiscount(); }, $shippingTotals);

        $this->assertEquals([
            new Amount(6.17, true, 1.23),
            new Amount(4.99, true, 1.0),
            new Amount(13.379999999999999, true, 2.68),
        ], $shippingTotalAmounts);
        $this->assertEquals([
            new Amount(6.17, true, 1.24),
            new Amount(5.0, true, 1.0),
            new Amount(0.0, true, 0.0),
        ], $shippingDiscountAmounts);

        $this->assertEquals(24.54, $quote->getGrandTotal());
    }
}
