<?php

declare(strict_types=1);

namespace Points\Core\Total;

use Mage_Tax_Model_Config;
use Mage_Sales_Model_Quote_Address;
use Points_Core_Model_Product_Price;
use Points\Core\Points;
use Points\Core\Amount;
use Points\Core\Extension\QuoteAddressItem;

use function Points\Core\quotePriceIncludesTax;

/**
 * @implements PointsInterface<float>
 */
class ShippingPoints implements PointsInterface {
    /**
     * @var Mage_Sales_Model_Quote_Address
     */
    private $address;
    /**
     * @var int
     */
    private $amount;

    /**
     * @param Mage_Sales_Model_Quote_Address $address
     * @param int $amount
     */
    public function __construct($address, $amount) {
        $this->address = $address;
        $this->amount = $amount;
    }

    /**
     * @return float
     */
    private function getDiscountRate() {
        $baseAmount = $this->address->getBaseShippingAmount();

        return $baseAmount > 0 ? $this->address->getBaseShippingDiscountAmount() / $baseAmount : 0.0;
    }

    /**
     * @param float $val
     * @return Amount<float>
     */
    private function getTaxedPoints($val) {
        $quote = $this->address->getQuote();
        $store = $quote->getStore();

        $shippingIncludesTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX);
        $shippingNonTax = $this->address->getBaseShippingAmount() + $this->address->getBaseShippingDiscountAmount();


        $taxRate = $shippingNonTax > 0 ?
            ($this->address->getBaseShippingTaxAmount() + $this->address->getBaseShippingHiddenTaxAmount()) / $shippingNonTax :
            0.0;

        $taxConversionRate = $shippingIncludesTax ? $taxRate / (1 + $taxRate) : $taxRate / 1;

        return new Amount($val, $shippingIncludesTax, $val * $taxConversionRate);
    }

    /**
     * @param float $val
     * @return float
     */
    private function getScaledPoints($val) {
        $nonDiscountRate = 1 - $this->getDiscountRate();
        $val = $nonDiscountRate * (float)$val;

        return $val;
    }

    public function getValue(): Amount {
        $points = $this->getScaledPoints($this->amount);

        return $this->getTaxedPoints($points);
    }

    public function getMin(): Amount {
        $points = $this->getScaledPoints(0);

        return $this->getTaxedPoints($points);
    }

    public function getMax(): Amount {
        $points = $this->getScaledPoints($this->amount);

        return $this->getTaxedPoints($points);
    }

    public function getDiscount(): Amount {
        $discountValue = $this->getDiscountRate() * $this->amount;

        return $this->getTaxedPoints($discountValue);
    }
}
