<?php

declare(strict_types=1);

namespace Points\Core\Total;

use Mage_Tax_Model_Config;
use Points_Core_Model_Product_Price;
use Points\Core\Points;
use Points\Core\Amount;
use Points\Core\Extension\QuoteAddress;

use function Points\Core\quotePriceIncludesTax;

/**
 * @template-covariant T of float|int
 * @implements TotalInterface<T>
 */
final class Shipping implements TotalInterface {
    /**
     * @var QuoteAddress
     */
    private $address;
    /**
     * @var ?PointsInterface<T>
     */
    private $points;
    /**
     * @var ?Amount<float>
     */
    private $pointTotalAmount = null;

    /**
     * @param QuoteAddress $address
     * @param ?PointsInterface<T> $points
     */
    public function __construct($address, $points) {
        $this->address = $address;
        $this->points = $points;
    }

    /**
     * @return QuoteAddress
     */
    public function getAddress() {
        return $this->address;
    }

    /**
     * @return ?PointsInterface<T>
     */
    public function getPoints() {
        return $this->points;
    }

    /**
     * @template U of float|int
     * @param ?PointsInterface<U> $points
     * @return self
     */
    public function withPoints($points) {
        $that = clone $this;

        /** @psalm-suppress InvalidPropertyAssignmentValue */
        $that->points = $points;

        return $that;
    }

    /**
     * @return ?Amount<float>
     */
    public function getPointTotalAmount() {
        return $this->pointTotalAmount;
    }

    /**
     * @param ?Amount<float> $pointTotalAmount
     * @return self
     */
    public function withPointTotalAmount($pointTotalAmount) {
        $that = clone $this;

        $that->pointTotalAmount = $pointTotalAmount;

        return $that;
    }

    /**
     * @return Amount<float>
     */
    public function getPrice() {
        $address = $this->address;

        if(quotePriceIncludesTax($address->getQuote())) {
            return new Amount(
                $address->getBaseShippingAmount() + $address->getBaseShippingTaxAmount() - $address->getBaseShippingDiscountAmount(),
                true,
                (float)$address->getBaseShippingTaxAmount()
            );
        }

        return new Amount(
            $address->getBaseShippingAmount() + $address->getBaseShippingHiddenTaxAmount(),
            false,
            (float)$address->getBaseShippingTaxAmount()
        );
    }

    /**
     * @return Amount<float>
     */
    public function getDiscount() {
        $address = $this->address;

        if(quotePriceIncludesTax($address->getQuote())) {
            return new Amount(
                (float)$address->getBaseShippingDiscountAmount(),
                true,
                (float)$address->getBaseShippingHiddenTaxAmount(),
            );
        }

        return new Amount(
            (float)$address->getBaseShippingDiscountAmount() - $address->getBaseShippingHiddenTaxAmount(),
            false,
            (float)$address->getBaseShippingHiddenTaxAmount(),
        );
    }
}
