<?php

declare(strict_types=1);

namespace Points\Core\Total;

use Points\Core\Extension\QuoteAddress as QuoteAddressExt;
use Points\Core\Amount;
use Mage_Sales_Model_Quote_Address;
use Mage_Sales_Model_Quote_Address_Item;
use Mage_Sales_Model_Quote_Item;

use function Points\Core\quotePriceIncludesTax;

class QuoteAddress {
    /**
     * @var QuoteAddressExt
     */
    private $address;
    /**
     * @var list<TotalInterface<int>>
     */
    private $totals;

    /**
     * @param QuoteAddressExt $address
     * @param list<TotalInterface<int>> $totals
     */
    public function __construct(
        Mage_Sales_Model_Quote_Address $address,
        array $totals
    ) {
        $this->address = $address;
        $this->totals = $totals;
    }

    /**
     * @return QuoteAddressExt
     */
    public function getAddress(): Mage_Sales_Model_Quote_Address {
        return $this->address;
    }

    /**
     * @return list<TotalInterface<int>>
     */
    public function getTotals(): array {
        return $this->totals;
    }

    public function getPoints(): QuoteAddressPoints {
        return new QuoteAddressPoints($this->totals, $this->getIncVat());
    }

    /**
     * Total available discount on the order.
     *
     * @return Amount<float>
     */
    public function getDiscountTotal(): Amount {
        /** @var Amount<float> */
        $discount = new Amount(0.0, $this->getIncVat(), 0.0);

        foreach($this->totals as $total) {
            $discount = $discount->add($total->getDiscount());
        }

        return $discount;
    }

    /**
     * @return Amount<float>
     */
    public function getTotal(): Amount {
        /** @var Amount<float> */
        $value = new Amount(0.0, $this->getIncVat(), 0.0);

        foreach($this->totals as $total) {
            $value = $value->add($total->getPrice());
        }

        return $value;
    }

    /**
     * Returns a filtered QuoteAddress instance only containing totals without
     * any point payment.
     */
    public function getExcluded(): QuoteAddress {
        $excluded = [];

        foreach($this->totals as $total) {
            if( ! $total->getPoints()) {
                $excluded[] = $total;
            }
        }

        return new QuoteAddress($this->address, $excluded);
    }

    /**
     * Returns a filtered QuoteAddress instance only containing totals with
     * point payment.
     */
    public function getIncluded(): QuoteAddress {
        $included = [];

        foreach($this->totals as $total) {
            if($total->getPoints()) {
                $included[] = $total;
            }
        }

        return new QuoteAddress($this->address, $included);
    }

    private function getIncVat(): bool {
        $quote = $this->address->getQuote();

        $quotePriceIncludesTax = quotePriceIncludesTax($quote);

        return $quote->getPointsWantedIncludesTax() ?? $quotePriceIncludesTax;
    }

    /**
     * Returns the associated Item total object if the provided item has a point total.
     * @param Mage_Sales_Model_Quote_Item|Mage_Sales_Model_Quote_Address_Item $item
     * @return ?Item<int>
     */
    public function getItemTotal($item) {
        /**
         * @var Item<int>|Shipping<int> $total
         */
        foreach($this->totals as $total) {
            if( ! $total instanceof Item || $total->getPoints() === null) {
                continue;
            }

            $addedItem = $total->getItem();

            if($addedItem instanceof Mage_Sales_Model_Quote_Item) {
                // Sometimes Magento uses quote items directly instead of
                // address items, so we have to match them here instead:
                if($addedItem === $item || ($addedItem->getId() && $addedItem->getId() == $item->getId())) {
                    return $total;
                }
            }
            else if($addedItem->getQuoteItem() === $item || ($addedItem->getQuoteItemId() && $addedItem->getQuoteItemId() == $item->getId())) {
                return $total;
            }
        }

        return null;
    }
}
