<?php

declare(strict_types=1);

namespace Points\Core\Total;

use Mage_Sales_Model_Quote_Address_Item;
use Mage_Tax_Model_Config;
use Points_Core_Model_Product_Price;
use Points\Core\Amount;
use Points\Core\Extension\QuoteAddressItem;
use Points\Core\Points;

use function Points\Core\quoteItemPriceIncludesTax;

/**
 * @implements PointsInterface<float>
 */
class ItemPoints implements PointsInterface {
    /**
     * @var Mage_Sales_Model_Quote_Address_Item
     */
    private $item;
    /**
     * @var Points_Core_Model_Product_Price
     */
    private $pointsPrice;

    /**
     * @param Mage_Sales_Model_Quote_Address_Item $item
     * @param Points_Core_Model_Product_Price $pointsPrice
     */
    public function __construct($item, $pointsPrice) {
        $this->item = $item;
        $this->pointsPrice = $pointsPrice;
    }

    /**
     * @return float
     */
    private function getDiscountRate() {
        $item = $this->item;

        $rowBeforeDiscount = $item->getBaseRowTotal() + $item->getBaseTaxAmount() + $item->getBaseHiddenTaxAmount();

        return $rowBeforeDiscount > 0 ? $item->getBaseDiscountAmount() / $rowBeforeDiscount : 0.0;
    }

    /**
     * @param float $val
     * @return Amount<float>
     */
    private function getTaxedPoints($val) {
        $item = $this->item;
        $pricesIncludeTax = quoteItemPriceIncludesTax($item);
        $taxRate = $item->getTaxPercent();
        $taxConversionRate = $pricesIncludeTax ? $taxRate / (100 + $taxRate) : $taxRate / 100;

        return new Amount($val, $pricesIncludeTax, $val * $taxConversionRate);
    }

    /**
     * @param float $val
     * @return float
     */
    private function getScaledPoints($val) {
        $nonDiscountRate = 1 - $this->getDiscountRate();
        $val = $nonDiscountRate * $this->item->getQty() * (float)$val;

        return $val;
    }

    public function getValue(): Amount {
        $points = $this->getScaledPoints($this->pointsPrice->getPrice());

        return $this->getTaxedPoints($points);
    }

    public function getMin(): Amount {
        $points = $this->getScaledPoints($this->pointsPrice->getMinPriceInRange());

        return $this->getTaxedPoints($points);
    }

    public function getMax(): Amount {
        $points = $this->getScaledPoints($this->pointsPrice->getMaxPrice() ?: $this->pointsPrice->getPrice());

        return $this->getTaxedPoints($points);
    }

    public function getDiscount(): Amount {
        $discountValue = $this->getDiscountRate() * $this->item->getQty() * (float)$this->pointsPrice->getPrice();

        return $this->getTaxedPoints($discountValue);
    }
}
