<?php

declare(strict_types=1);

namespace Points\Core\Total;

use Mage_Tax_Model_Config;
use Points_Core_Model_Product_Price;
use Points\Core\Points;
use Points\Core\Amount;
use Points\Core\Extension\QuoteAddressItem;
use Points\Core\Extension\QuoteItem;

use function Points\Core\quoteItemPriceIncludesTax;

/**
 * @template-covariant T of float|int
 * @implements TotalInterface<T>
 */
final class Item implements TotalInterface {
    /**
     * @var QuoteItem|QuoteAddressItem
     */
    private $item;
    /**
     * @var ?PointsInterface<T>
     */
    private $points;
    /**
     * @var ?Amount<float>
     */
    private $pointTotalAmount = null;

    /**
     * @param QuoteItem|QuoteAddressItem $item
     * @param ?PointsInterface<T> $points
     */
    public function __construct($item, $points) {
        $this->item = $item;
        $this->points = $points;
    }

    /**
     * @return QuoteItem|QuoteAddressItem
     */
    public function getItem() {
        return $this->item;
    }

    /**
     * @return ?PointsInterface<T>
     */
    public function getPoints() {
        return $this->points;
    }

    /**
     * @template U of float|int
     * @param ?PointsInterface<U> $points
     * @return self
     */
    public function withPoints($points) {
        $that = clone $this;

        /** @psalm-suppress InvalidPropertyAssignmentValue */
        $that->points = $points;

        return $that;
    }

    /**
     * @return ?Amount<float>
     */
    public function getPointTotalAmount() {
        return $this->pointTotalAmount;
    }

    /**
     * @param ?Amount<float> $pointTotalAmount
     * @return self
     */
    public function withPointTotalAmount($pointTotalAmount) {
        $that = clone $this;

        $that->pointTotalAmount = $pointTotalAmount;

        return $that;
    }

    /**
     * @return Amount<float>
     */
    public function getPrice() {
        if(quoteItemPriceIncludesTax($this->item)) {
            return new Amount(
                $this->item->getBaseRowTotalInclTax() - $this->item->getBaseDiscountAmount(),
                true,
                (float)$this->item->getBaseTaxAmount(),
            );
        }

        return new Amount(
            $this->item->getBaseRowTotal() - $this->item->getBaseDiscountAmount() + $this->item->getBaseHiddenTaxAmount(),
            false,
            (float)$this->item->getBaseTaxAmount(),
        );
    }

    /**
     * @return Amount<float>
     */
    public function getDiscount() {
        if(quoteItemPriceIncludesTax($this->item)) {
            return new Amount(
                (float)$this->item->getBaseDiscountAmount(),
                true,
                (float)$this->item->getBaseHiddenTaxAmount()
            );
        }

        return new Amount(
            (float)$this->item->getBaseDiscountAmount() - $this->item->getBaseHiddenTaxAmount(),
            false,
            (float)$this->item->getBaseHiddenTaxAmount(),
        );
    }
}
