<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Points\Core\QuoteAddressTotal;
use Points\Core\Points;
use Points\Core\Currency;

class QuoteShipping {
    /**
     * @var QuoteAddressTotal
     */
    protected $total;

    /**
     * @var Currency
     */
    protected $currency;

    /**
     * @var ?Currency
     */
    protected $discount;

    /**
     * @param array{currency:Currency, discount:?Currency} $amounts
     */
    public function __construct(
        QuoteAddressTotal $total,
        array $amounts
    ) {
        $this->total = $total;
        $this->currency = $amounts["currency"];
        $this->discount = $amounts["discount"];
    }

    public function getPoints(): ?QuoteShippingPoints {
        $parts = $this->getPointsParts();

        return $parts ? new QuoteShippingPoints($parts) : null;
    }

    public function getCurrency(): QuoteShippingCurrency {
        return new QuoteShippingCurrency([
            "currency" => $this->currency,
            "discount" => $this->discount,
        ]);
    }

    /**
     * @return ?array{min:Points, max:Points, value:Points, discount:?Points}
     */
    protected function getPointsParts(): ?array {
        $min = $this->total->getShippingPointsMin();
        $max = $this->total->getShippingPointsMax();
        $value = $this->total->getShippingPointsValue();
        $discount = $this->total->getPointsDiscount()[QuoteAddressTotal::ROW_SHIPPING] ?? null;

        // If one is set, the rest should be too, excluding discount
        if(!$min || !$max || !$value) {
            return null;
        }

        return [
            "min" => $min,
            "max" => $max,
            "value" => $value,
            "discount" => $discount,
        ];
    }
}
