<?php

declare(strict_types=1);

namespace Points\Core;

use Exception;
use Mage;
use Mage_Sales_Model_Order;
use Mage_Sales_Model_Order_Address;
use Mage_Sales_Model_Order_Item;
use Mage_Sales_Model_Quote;
use Mage_Sales_Model_Quote_Address;
use Mage_Sales_Model_Quote_Item;
use Mage_Customer_Model_Customer;
use Mage_Core_Model_Resource_Setup;

use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class Base extends TestCase {
    use MatchesSnapshots;

    const QUOTE_DIFF_KEYS = [
        "applied_rule_ids" => true,
        "converted_at" => true,
        "created_at" => true,
        "customer_group_id" => true,
        "customer_id" => true,
        "customer_tax_class_id" => true,
        "customer_taxvat" => true,
        "entity_id" => true,
        "gift_message_id" => true,
        "orig_order_id" => true,
        "password_hash" => true,
        "remote_ip" => true,
        "reserved_order_id" => true,
        "store_id" => true,
        "updated_at" => true,
    ];

    const QUOTE_ADDRESS_DIFF_KEYS = [
        "address_id" => true,
        "quote_id" => true,
        "created_at" => true,
        "updated_at" => true,
        "customer_id" => true,
        "customer_address_id" => true,
        "region_id" => true,
        "vat_id" => true,
        "vat_request_id" => true,
        "gift_message_id" => true,
        "selected_points_total_instance" => true,
        "cached_items_nonnominal" => true,
        "cached_items_nominal" => true,
        "cached_items_all" => true,
    ];

    const QUOTE_ITEM_DIFF_KEYS = [
        "additional_data" => true,
        "applied_rule_ids" => true,
        "created_at" => true,
        "gift_message_id" => true,
        "item_id" => true,
        "parent_item_id" => true,
        "product" => true,
        "product_id" => true,
        "qty_options" => true,
        "quote_id" => true,
        "store_id" => true,
        "tax_class_id" => true,
        "updated_at" => true,
    ];

    const ORDER_DIFF_KEYS = [
        "applied_rule_ids" => true,
        "billing_address_id" => true,
        "created_at" => true,
        "customer" => true,
        "customer_group_id" => true,
        "customer_id" => true,
        "customer_tax_class_id" => true,
        "customer_taxvat" => true,
        "entity_id" => true,
        "increment_id" => true,
        "protect_code" => true,
        "quote" => true,
        "quote_id" => true,
        "shipping_address_id" => true,
        "store_id" => true,
        "updated_at" => true,
    ];

    const ORDER_ADDRESS_DIFF_KEYS = [
        "store_id" => true,
        "customer_id" => true,
        "customer_address_id" => true,
        "vat_id" => true,
        "vat_request_id" => true,
        "region_id" => true,
        "parent_id" => true,
        "created_at" => true,
        "updated_at" => true,
        "entity_id" => true,
    ];

    const ORDER_ITEM_DIFF_KEYS = [
        "created_at" => true,
        "item_id" => true,
        "order_id" => true,
        "product" => true,
        "product_id" => true,
        "product_options" => true,
        "quote_item_id" => true,
        "quote_parent_item_id" => true,
        "store_id" => true,
        "updated_at" => true,
    ];

    protected function clearQuotes(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        try {
            $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

            $setupModel->deleteConfigData("carriers/freeshipping/active", "websites");
            $setupModel->deleteConfigData("carriers/freeshipping/free_shipping_subtotal", "websites");

            // Prevent any existing quotes from being loaded
            $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

            if($conn) {
                $conn->query("UPDATE sales_flat_quote SET is_active = 0");
            }
        }
        catch(Exception $e) {
            error_log((string)$e);
        }

        MagentoManager::reset();
    }

    protected function clearLimits(): void {
        try {
            MagentoManager::reset();
            MagentoManager::initAdmin();

            $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

            if($conn) {
                $conn->query("DELETE FROM points_limit_total")->execute();
                $conn->query("DELETE FROM points_limit_order")->execute();
                $conn->query("UPDATE sales_flat_order SET points_type = 'TEST_RENAMED' WHERE points_type = 'TEST'")->execute();
            }
        }
        finally {
            MagentoManager::reset();
        }
    }

    protected function assertQuoteMatchesSnapshot(Mage_Sales_Model_Quote $quote): void {
        $items = $quote->getAllItems();
        $quoteData = array_diff_key($quote->getData(), self::QUOTE_DIFF_KEYS);
        $addressData = array_map(function(Mage_Sales_Model_Quote_Address $a): array {
            return array_diff_key($a->getData(), self::QUOTE_ADDRESS_DIFF_KEYS);
        }, $quote->getAllAddresses());
        $itemData = array_map(function(Mage_Sales_Model_Quote_Item $i): array {
            return array_diff_key($i->getData(), self::QUOTE_ITEM_DIFF_KEYS);
        }, $items);

        $this->assertMatchesYamlSnapshot([
            "quote" => $quoteData,
            "addresses" => $addressData,
            "items" => $itemData,
        ]);
    }

    protected function assertOrderMatchesSnapshot(Mage_Sales_Model_Order $order): void {
        $items = $order->getAllItems();

        /**
         * @var Array<string, mixed>
         */
        $data = $order->getData();

        $orderData = array_diff_key($data, self::ORDER_DIFF_KEYS);
        $addressData = array_map(function(Mage_Sales_Model_Order_Address $a): array {
            return array_diff_key($a->getData(), self::ORDER_ADDRESS_DIFF_KEYS);
        }, $order->getAddressesCollection()->getItems());
        $itemData = array_map(function(Mage_Sales_Model_Order_Item $i): array {
            return array_diff_key($i->getData(), self::ORDER_ITEM_DIFF_KEYS);
        }, $items);

        $this->assertMatchesYamlSnapshot([
            "order" => $orderData,
            "addresses" => $addressData,
            "items" => $itemData,
        ]);
    }

    public function loginCustomer(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer");
        $session = Mage::getSingleton("customer/session");

        $customer->setWebsiteId($store->getWebsiteId());
        $customer->loadByEmail("test-customer@example.com");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
    }
}
