<?php

declare(strict_types=1);

namespace Points\Core;

use Exception;
use Mage;
use Mage_Sales_Model_Quote_Address;
use Mage_Sales_Model_Quote_Item;
use Mage_Sales_Model_Quote_Item_Abstract;
use Mage_Tax_Model_Config;

/**
 * TODO: Refactor this instance
 */
class QuoteAddressTotal extends Total {
    const ROW_SHIPPING = "shipping";
    const ITEM_ID_PREFIX = "item_id_";
    const ITEM_DYNAMIC_PREFIX = "dyn";

    public static function itemKey(
        Mage_Sales_Model_Quote_Item_Abstract $item
    ): string {
        $id = $item->getId();

        if( ! $id) {
            /**
             * We have a quote item here, even if it is the address version.
             *
             * @var Mage_Sales_Model_Quote_Item $item
             */
            $options = $item->getOptions();
            $options = array_filter($options, function($option) {
                return $option->getCode() !== "info_buyRequest";
            });
            $options = array_map(function($option) {
                return $option->getCode().":".$option->getValue();
            }, $options);
            $options = array_values($options);

            return implode("_", array_merge([
                self::ITEM_DYNAMIC_PREFIX,
                $item->getProductId(),
            ], $options));
        }

        return self::ITEM_ID_PREFIX.$id;
    }

    /**
     * @var Array<Currency>
     */
    protected $discount;

    /**
     * @var Array<string, Points>
     */
    protected $pointsDiscount;

    /**
     * @param array{min:Array<string, Points>, max:Array<string, Points>, value:Array<string, Points>, discount:Array<string, Points>} $points
     * @param Array<Currency> $currency
     * @param Array<Currency> $discount
     */
    protected function __construct(
        array $points,
        array $currency,
        array $discount
    ) {
        $this->discount = $discount;
        $this->pointsDiscount = $points["discount"];

        parent::__construct($currency, $points["min"], $points["max"], $points["value"]);
    }

    public function getRowPointsValue(Mage_Sales_Model_Quote_Item_Abstract $item): ?Points {
        return $this->points[self::itemKey($item)] ?? null;
    }

    public function getRowPointsMin(Mage_Sales_Model_Quote_Item_Abstract $item): ?Points {
        return $this->minPoints[self::itemKey($item)] ?? null;
    }

    public function getRowPointsMax(Mage_Sales_Model_Quote_Item_Abstract $item): ?Points {
        return $this->maxPoints[self::itemKey($item)] ?? null;
    }

    public function getRowCurrency(Mage_Sales_Model_Quote_Item_Abstract $item): ?Currency {
        return $this->currency[self::itemKey($item)] ?? null;
    }

    public function getShippingPointsValue(): ?Points {
        return $this->points[self::ROW_SHIPPING] ?? null;
    }

    public function getShippingPointsMin(): ?Points {
        return $this->minPoints[self::ROW_SHIPPING] ?? null;
    }

    public function getShippingPointsMax(): ?Points {
        return $this->maxPoints[self::ROW_SHIPPING] ?? null;
    }

    public function getShippingCurrency(): ?Currency {
        return $this->currency[self::ROW_SHIPPING] ?? null;
    }

    /**
     * @return Array<string, Currency>
     */
    public function getDiscount(): array {
        return $this->discount;
    }

    /**
     * @return Array<string, Points>
     */
    public function getPointsDiscount(): array {
        return $this->pointsDiscount;
    }

    public static function fromQuoteAddress(
        Mage_Sales_Model_Quote_Address $address,
        string $type,
        ProviderInterface $provider
    ): self {
        $items = $address->getAllItems();
        $quote = $address->getQuote();
        $customerGroupId = $quote->getCustomerGroupId();
        $store = $quote->getStore();
        $productIds = [];
        $indices = [];

        foreach($items as $i) {
            if($i->isDeleted()) {
                continue;
            }

            // Always fetch all product point-prices, they might be part of child-items
            $productIds[] = $i->getProductId();

            // Only consider parents, since children do not have any amounts
            if($i->getParentItem()) {
                continue;
            }

            // Kinda ugly way of keeping track of item indices
            $key = self::itemKey($i);

            $indices[$key] = $i;

            // FIXME: Handle Bundle
        }

        $resource = Mage::getResourceModel("points_core/product_price");
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $shippingIncludesTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX);

        $prices = $resource->getMatchingPrices($store, $customerGroupId, $type, $productIds);
        /**
         * @var Array<string, Currency>
         */
        $currency = [];
        /**
         * @var Array<string, Currency>
         */
        $discount = [];
        /**
         * @var array{min: Array<string, Amount<float>>, max: Array<string, Amount<float>>, value: Array<string, Amount<float>>} $points
         */
        $points = [
            "min" => [],
            "max" => [],
            "value" => [],
            "discount" => [],
        ];

        foreach($indices as $id => $item) {
            $included = false;
            $productId = $item->getProductId();

            if($item->getProductType() === "configurable") {
                // We have a configurable item, those store their price on the
                // child product but report the price on the parent row, we need
                // to use the child item product id to find the correct price:

                foreach($item->getChildren() as $i) {
                    $productId = $i->getProductId();

                    break;
                }
            }

            $rowBeforeDiscount = $item->getBaseRowTotal() + $item->getBaseTaxAmount() + $item->getBaseHiddenTaxAmount();
            $discountRate = $rowBeforeDiscount > 0 ? $item->getBaseDiscountAmount() / $rowBeforeDiscount : 0;
            $nonDiscountRate = 1 - $discountRate;

            if( ! empty($prices[$productId])) {
                $included = true;
                $p = $prices[$productId];

                $val = $nonDiscountRate * $item->getQty() * (float)$p->getPrice();
                $max = $nonDiscountRate * $item->getQty() * (float)($p->getMaxPrice() ? $p->getMaxPrice() : $p->getPrice());
                $min = $nonDiscountRate * $item->getQty() * (float)$p->getMinPrice();
                $discountValue = $discountRate * $item->getQty() * (float)$p->getPrice();

                $taxRate = $item->getTaxPercent() / 100;
                $taxConversionRate = $pricesIncludeTax ? $taxRate / (1 + $taxRate) : $taxRate;

                $points["min"][$id] = new Amount($min, $pricesIncludeTax, $min * $taxConversionRate);
                $points["max"][$id] = new Amount($max, $pricesIncludeTax, $max * $taxConversionRate);
                $points["value"][$id] = new Amount($val, $pricesIncludeTax, $val * $taxConversionRate);
                $points["discount"][$id] = new Amount($discountValue, $pricesIncludeTax, $discountValue * $taxConversionRate);
            }

            $currency[$id] = new Currency(
                $pricesIncludeTax ?
                    $item->getBaseRowTotalInclTax() - $item->getBaseDiscountAmount() :
                    $item->getBaseRowTotal() - $item->getBaseDiscountAmount() + $item->getBaseHiddenTaxAmount(),
                $pricesIncludeTax,
                (float)$item->getBaseTaxAmount(),
                $included
            );
            $discount[$id] = new Currency(
                $pricesIncludeTax ?
                    (float)$item->getBaseDiscountAmount() :
                    (float)$item->getBaseDiscountAmount() - $item->getBaseHiddenTaxAmount(),
                $pricesIncludeTax,
                (float)$item->getBaseHiddenTaxAmount(),
                $included
            );
        }

        if($address->getBaseShippingAmount() > 0) {
            $included = false;
            // TODO: Include an Amount instance to make it easier to calculate value
            $amount = $provider->getQuoteShippingPrice($address);

            if($amount !== null) {
                $included = true;
                $taxRate = ($address->getBaseShippingTaxAmount() + $address->getBaseShippingHiddenTaxAmount())
                    / ($address->getBaseShippingAmount() + $address->getBaseShippingDiscountAmount());
                $discountRate = $address->getBaseShippingDiscountAmount() / $address->getBaseShippingAmount();
                $nonDiscountRate = 1 - $discountRate;
                $taxConversionRate = $shippingIncludesTax ? $taxRate / (1 + $taxRate) : $taxRate;

                $value = $nonDiscountRate * $amount;
                $discountValue = $discountRate * $amount;

                $points["value"][self::ROW_SHIPPING] = new Amount($value, $shippingIncludesTax, $value * $taxConversionRate);
                // TODO: Configurable minimum shipping amount?
                $points["min"][self::ROW_SHIPPING] = new Amount(0, $shippingIncludesTax, 0);
                $points["max"][self::ROW_SHIPPING] = new Amount($value, $shippingIncludesTax, $value * $taxConversionRate);
                $points["discount"][self::ROW_SHIPPING] = new Amount($discountValue, $shippingIncludesTax, $discountValue * $taxConversionRate);
            }

            $currency[self::ROW_SHIPPING] = new Currency(
                $shippingIncludesTax ?
                    $address->getBaseShippingAmount() + $address->getBaseShippingTaxAmount() - $address->getBaseShippingDiscountAmount() :
                    $address->getBaseShippingAmount() + $address->getBaseShippingHiddenTaxAmount(),
                $shippingIncludesTax,
                (float)$address->getBaseShippingTaxAmount(),
                $included
            );
            $discount[self::ROW_SHIPPING] = new Currency(
                $pricesIncludeTax ?
                    (float)$address->getBaseShippingDiscountAmount() :
                    (float)$address->getBaseShippingDiscountAmount() - $address->getBaseShippingHiddenTaxAmount(),
                $pricesIncludeTax,
                (float)$address->getBaseShippingHiddenTaxAmount(),
                $included
            );
        }

        $spreadPoints = array_map("Points\\Core\\spreadPoints", $points);

        return new self($spreadPoints, $currency, $discount);
    }
}
