<?php

declare(strict_types=1);

class Points_Core_Model_Resource_Product_Index extends Mage_Core_Model_Resource_Db_Abstract {
    public function __construct() {
        $this->_init("points_core/product_index", "entity_id");
    }

    protected function _construct(): void {

    }

    public function reindexAll(): void {
        $write = $this->_getWriteAdapter();

        try {
            $this->beginTransaction();

            // Delete since we reindex everything anyway
            $write->query("DELETE FROM {$this->getTable("points_core/product_index")}");

            $this->reindexSimple($write);
            $this->reindexConfigurable($write);

            $this->commit();
        }
        catch(Exception $e) {
            $this->rollBack();

            Mage::logException($e);

            throw $e;
        }
    }

    protected function reindexSimple(
        Zend_Db_Adapter_Abstract $adapter,
        string $cond = null
    ): void {
        $pointsTable = $this->getTable("points_core/product_price");
        $indexTable = $this->getTable("points_core/product_index");
        $websiteTable = $this->getTable("core/website");
        $storeGroupTable = $this->getTable("core/store_group");
        $customerGroupTable = $this->getTable("customer/customer_group");
        $productTable = $this->getTable("catalog/product");

        $cond = $cond ? "AND $cond" : "";
        $query = <<<EOF
INSERT INTO {$indexTable} (
    website_id,
    customer_group_id,
    type,
    product_id,
    price,
    min_price,
    max_price
)
SELECT w.website_id,
    cg.customer_group_id,
    t.type,
    p.entity_id product_id,
    COALESCE(psg.price, ps.price, pbg.price, pb.price) price,
    COALESCE(psg.min_price, ps.min_price, pbg.min_price, pb.min_price) min_price,
    COALESCE(
        psg.max_price, psg.price,
        ps.max_price, ps.price,
        pbg.max_price, pbg.price,
        pb.max_price, pb.price
    ) max_price
FROM {$websiteTable} w
JOIN {$storeGroupTable} sg ON w.default_group_id = sg.group_id
JOIN {$customerGroupTable} cg
JOIN {$productTable} p
JOIN (SELECT DISTINCT t.type FROM {$pointsTable} t) t
LEFT JOIN {$pointsTable} pb
    ON  pb.store_id = 0
    AND pb.customer_group_id = 0
    AND pb.type = t.type
    AND pb.product_id = p.entity_id
LEFT JOIN {$pointsTable} pbg
    ON  pbg.store_id = 0
    AND pbg.customer_group_id = cg.customer_group_id
    AND pbg.type = t.type
    AND pbg.product_id = p.entity_id
LEFT JOIN {$pointsTable} ps
    ON  ps.store_id = sg.default_store_id
    AND ps.type = t.type
    AND ps.customer_group_id = 0
    AND ps.product_id = p.entity_id
LEFT JOIN {$pointsTable} psg
    ON  psg.store_id = sg.default_store_id
    AND psg.type = t.type
    AND psg.customer_group_id = cg.customer_group_id
    AND psg.product_id = p.entity_id
WHERE w.website_id <> 0 AND (
     pb.entity_id IS NOT NULL
  OR pbg.entity_id IS NOT NULL
  OR ps.entity_id IS NOT NULL
  OR psg.entity_id IS NOT NULL
)
{$cond}
ON DUPLICATE KEY UPDATE
    price = VALUES(price),
    min_price = VALUES(min_price),
    max_price = VALUES(max_price),
    updated_at = CURRENT_TIMESTAMP
EOF;

        $adapter->query($query);
    }

    /**
     * Indexes configurable products by looking at the indexed values of their child-products.
     */
    protected function reindexConfigurable(
        Zend_Db_Adapter_Abstract $adapter,
        string $cond = null
    ): void {
        $indexTable = $this->getTable("points_core/product_index");
        $websiteTable = $this->getTable("core/website");
        $storeGroupTable = $this->getTable("core/store_group");
        $productTable = $this->getTable("catalog/product");
        $productSuperLinkTable = $this->getTable("catalog/product_super_link");
        $hideOutOfStockJoin = "";
        $hideOutOfStockCond = "";

        if( ! Mage::helper("cataloginventory")->isShowOutOfStock()) {
            $stockIndexTable = $this->getTable("cataloginventory/stock_status");

            $hideOutOfStockJoin = "JOIN {$stockIndexTable} ciss ON ciss.product_id = p.entity_id AND ciss.website_id = w.website_id\n";
            $hideOutOfStockCond = sprintf("AND ciss.stock_status = %s", $adapter->quote(Mage_CatalogInventory_Model_Stock_Status::STATUS_IN_STOCK));
        }

        $cond = $cond ? "AND ($cond)" : "";

        $query = <<<EOF
INSERT INTO {$indexTable} (
    website_id,
    customer_group_id,
    type,
    product_id,
    price,
    min_price,
    max_price
)
SELECT w.website_id,
    i.customer_group_id,
    i.type,
    p.entity_id product_id,
    MIN(i.price) price,
    MIN(i.min_price) min_price,
    MAX(i.max_price) max_price
FROM {$websiteTable} w
JOIN {$storeGroupTable} sg ON w.default_group_id = sg.group_id
JOIN {$productTable} p
JOIN {$productSuperLinkTable} l ON l.parent_id = p.entity_id
{$hideOutOfStockJoin}
JOIN {$indexTable} i
  ON  i.website_id = w.website_id
  AND i.product_id = l.product_id
WHERE w.website_id <> 0 AND p.type_id = 'configurable'
{$hideOutOfStockCond}
{$cond}
GROUP BY w.website_id, i.customer_group_id, i.type, p.entity_id
ON DUPLICATE KEY UPDATE
    price = VALUES(price),
    min_price = VALUES(min_price),
    max_price = VALUES(max_price),
    updated_at = CURRENT_TIMESTAMP
EOF;

        $adapter->query($query);
    }

    public function reindexProductIds(array $productIds): void {
        $write = $this->_getWriteAdapter();

        try {
            $this->beginTransaction();

            // Escape
            $cond = implode(",", array_map(function($i) use($write): string {
                return $write->quote($i);
            }, $productIds));

            $write->query("DELETE FROM {$this->getTable("points_core/product_index")} WHERE product_id IN ({$cond})");

            $this->reindexSimple($write, "p.entity_id IN ($cond)");
            $this->reindexConfigurable($write, "p.entity_id IN ($cond)");

            $this->commit();
        }
        catch(Exception $e) {
            $this->rollBack();

            Mage::logException($e);

            throw $e;
        }
    }

    /**
     * @param int $childId
     */
    public function getProductParentsByChild($childId): array {
        $write = $this->_getWriteAdapter();
        $select = $write->select()
            ->from(["l" => $this->getTable("catalog/product_relation")], ["parent_id"])
            ->join(
                ["e" => $this->getTable("catalog/product")],
                "l.parent_id = e.entity_id",
                ["e.type_id"]
            )
            ->where("l.child_id = ?", $childId);

        return $write->fetchPairs($select);
    }
}
