<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Exception;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Customer;
use Mage_Customer_Model_Group;
use Mage_Sales_Model_Quote;
use Mage_Sales_Model_Quote_Item;
use Mage_Tax_Model_Config;
use Points_Core_Helper_Data;
use Throwable;
use Varien_Object;
use Varien_Profiler;

use Points\Core\Currency;
use Points\Core\Points;
use Points\Core\ProviderInterface;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\Customer;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/../Fixtures/Points.php";

class TotalsTest extends TestCase {
    use MatchesSnapshots;

    const SIMPLE_INCL_TAX = 12.34;
    const VIRTUAL_INCL_TAX = 9.99;
    const SHIPPING_INCL_TAX = 13.37;

    public static function setUpBeforeClass(): void {
        loadPoints();
    }

    public function setUp(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");
        MagentoManager::initAdmin();

        // Got to reset these
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->deleteConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, "websites");
        $setupModel->deleteConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, "websites");

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    /**
     * @return array{quote:Mage_Sales_Model_Quote, item:Mage_Sales_Model_Quote_Item}
     */
    public function simpleShippingQuote(): array {
        $quote = Mage::getModel("sales/quote");

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $item = $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($item)) {
            throw new Exception($item);
        }

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return [
            "quote" => $quote,
            "item" => $item,
        ];
    }

    public function testSimple(): void {
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->simpleShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it including tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Currency(12.34, true, 2.47, true),
            "shipping" => new Currency(13.37, true, 2.67, false),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1332, true, 267),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(0, true, 0),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1332, true, 267),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        $this->assertEquals(25.71, $quote->getGrandTotal());
    }

    public function testSimplePriceNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->simpleShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it excluding tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Currency(12.34, false, 3.09, true),
            "shipping" => new Currency(13.37, false, 3.34, false),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1332, false, 333),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(0, false, 0),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1332, false, 333),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        $this->assertEquals(32.14, $quote->getGrandTotal());
    }

    /**
     * @return array{quote:Mage_Sales_Model_Quote, item:Mage_Sales_Model_Quote_Item}
     */
    public function virtualQuote(): array {
        $quote = Mage::getModel("sales/quote");

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $item = $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($item)) {
            throw new Exception($item);
        }

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return [
            "quote" => $quote,
            "item" => $item,
        ];
    }

    public function testVirtual(): void {
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->virtualQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $shippingTotal->getCurrency());
        $this->assertEquals([], $shippingTotal->getPointsValue());
        $this->assertEquals([], $shippingTotal->getPointsMin());
        $this->assertEquals([], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Currency(9.99, true, 2.0, true),
        ], $billingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1999, true, 400),
        ], $billingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(499, true, 100),
        ], $billingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1499, true, 300),
        ], $billingTotal->getPointsMax());
        $this->assertEquals(9.99, $quote->getGrandTotal());
    }

    public function testVirtualNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "item" => $item] = $this->virtualQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        $this->assertEquals([], $shippingTotal->getCurrency());
        $this->assertEquals([], $shippingTotal->getPointsValue());
        $this->assertEquals([], $shippingTotal->getPointsMin());
        $this->assertEquals([], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Currency(9.99, false, 2.5, true),
        ], $billingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1999, false, 500),
        ], $billingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(499, false, 125),
        ], $billingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($item) => new Points(1499, false, 375),
        ], $billingTotal->getPointsMax());
        $this->assertEquals(12.49, $quote->getGrandTotal());
    }

    /**
     * @return array{
     *   quote:Mage_Sales_Model_Quote,
     *   simple:Mage_Sales_Model_Quote_Item,
     *   virtual:Mage_Sales_Model_Quote_Item
     * }
     */
    public function simpleVirtualShippingQuote(): array {
        $quote = Mage::getModel("sales/quote");

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $simpleItem = $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($simpleItem)) {
            throw new Exception($simpleItem);
        }

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $virtualItem = $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        if(is_string($virtualItem)) {
            throw new Exception($virtualItem);
        }

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return [
            "quote" => $quote,
            "simple" => $simpleItem,
            "virtual" => $virtualItem,
        ];
    }

    public function testSimpleVirtual(): void {
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it including tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Currency(12.34, true, 2.47, true),
            QuoteAddressTotal::itemKey($virtual) => new Currency(9.99, true, 2.0, true),
            "shipping" => new Currency(13.37, true, 2.67, false),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, true, 267),
            QuoteAddressTotal::itemKey($virtual) => new Points(1999, true, 400),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(0, true, 0),
            QuoteAddressTotal::itemKey($virtual) => new Points(499, true, 100),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, true, 267),
            QuoteAddressTotal::itemKey($virtual) => new Points(1499, true, 300),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        $this->assertEquals(35.7, $quote->getGrandTotal());
    }

    public function testSimpleVirtualNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(null);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it including tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Currency(12.34, false, 3.09, true),
            QuoteAddressTotal::itemKey($virtual) => new Currency(9.99, false, 2.49, true),
            "shipping" => new Currency(13.37, false, 3.35, false),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, false, 333),
            QuoteAddressTotal::itemKey($virtual) => new Points(1999, false, 500),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(0, false, 0),
            QuoteAddressTotal::itemKey($virtual) => new Points(499, false, 125),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, false, 333),
            QuoteAddressTotal::itemKey($virtual) => new Points(1499, false, 375),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        $this->assertEquals(44.63, $quote->getGrandTotal());
    }

    public function testSimpleVirtualShipping(): void {
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it including tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Currency(12.34, true, 2.47, true),
            QuoteAddressTotal::itemKey($virtual) => new Currency(9.99, true, 2.0, true),
            "shipping" => new Currency(13.37, true, 2.67, true),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, true, 267),
            QuoteAddressTotal::itemKey($virtual) => new Points(1999, true, 400),
            "shipping" => new Points(99, true, 19),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(0, true, 0),
            QuoteAddressTotal::itemKey($virtual) => new Points(499, true, 100),
            "shipping" => new Points(0, true, 0),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, true, 267),
            QuoteAddressTotal::itemKey($virtual) => new Points(1499, true, 300),
            "shipping" => new Points(99, true, 19),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        $this->assertEquals(35.7, $quote->getGrandTotal());
    }

    public function testSimpleVirtualShippingNotIncludeTax(): void {
        MagentoManager::init();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it including tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Currency(12.34, false, 3.09, true),
            QuoteAddressTotal::itemKey($virtual) => new Currency(9.99, false, 2.49, true),
            "shipping" => new Currency(13.37, false, 3.35, true),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, false, 333),
            QuoteAddressTotal::itemKey($virtual) => new Points(1999, false, 500),
            "shipping" => new Points(99, false, 25),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(0, false, 0),
            QuoteAddressTotal::itemKey($virtual) => new Points(499, false, 125),
            "shipping" => new Points(0, false, 0),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(1332, false, 333),
            QuoteAddressTotal::itemKey($virtual) => new Points(1499, false, 375),
            "shipping" => new Points(99, false, 25),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        $this->assertEquals(44.63, $quote->getGrandTotal());
    }

    public function testSimpleVirtualShippingDiscount(): void {
        MagentoManager::init();

        ["quote" => $quote, "simple" => $simple, "virtual" => $virtual] = $this->simpleVirtualShippingQuote();

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $shippingAddress = $quote->getShippingAddress();
        $billingAddress = $quote->getBillingAddress();

        $shippingTotal = QuoteAddressTotal::fromQuoteAddress($shippingAddress, "TEST", $mockProvider);

        // We have it including tax in the total
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Currency(6.17, true, 1.23, true),
            QuoteAddressTotal::itemKey($virtual) => new Currency(4.99, true, 1.0, true),
            // Discount is not applied to shipping, and magento is being magento
            "shipping" => new Currency(13.379999999999999, true, 2.68, true),
        ], $shippingTotal->getCurrency());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(666, true, 133),
            QuoteAddressTotal::itemKey($virtual) => new Points(999, true, 200),
            "shipping" => new Points(99, true, 20),
        ], $shippingTotal->getPointsValue());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(0, true, 0),
            QuoteAddressTotal::itemKey($virtual) => new Points(250, true, 50),
            "shipping" => new Points(0, true, 0),
        ], $shippingTotal->getPointsMin());
        $this->assertEquals([
            QuoteAddressTotal::itemKey($simple) => new Points(666, true, 133),
            QuoteAddressTotal::itemKey($virtual) => new Points(749, true, 150),
            "shipping" => new Points(99, true, 20),
        ], $shippingTotal->getPointsMax());

        $billingTotal = QuoteAddressTotal::fromQuoteAddress($billingAddress, "TEST", $mockProvider);
        $this->assertEquals([], $billingTotal->getCurrency());
        $this->assertEquals([], $billingTotal->getPointsValue());
        $this->assertEquals([], $billingTotal->getPointsMin());
        $this->assertEquals([], $billingTotal->getPointsMax());
        // TODO: This is off by one comparfed to currency
        $this->assertEquals(24.54, $quote->getGrandTotal());
    }
}
