<?php

declare(strict_types=1);

namespace Points\Core;

use Exception;
use Mage;

/**
 * Function wrapping time, which can be overriden in tests.
 */
function getTime(): int {
    return time();
}

/**
 * @param Array<string, float> $amounts
 * @return Array<string, int>
 */
function calculateRoundingSpread(array $amounts): array {
    // Round to 3 decimal digits to avoid tiny float errors,
    // could possibly be solved by using a diff vs ceil/floor with 0.01
    $amounts = array_map(function(float $a): float {
        return round($a, 3);
    }, $amounts);
    $total = (int)ceil(array_sum($amounts));
    $totalLow = array_sum($amounts);
    $totalErr = $total - $totalLow;

    // Sort them with the lowest rouding error vs floor first
    uasort($amounts, function(float $a, float $b) {
        $errA = $a - floor($a);
        $errB = $b - floor($b);
        $diff = $errA - $errB;

        return $diff > 0 ? 1 : ($diff < 0 ? -1 : 0);
    });

    // If we have error to correct we want to start to round upwards on
    // the largest error vs floor first
    if($totalErr > 0) {
        $amounts = array_reverse($amounts);
    }

    // Walk over the existing values and round them up as long as
    // we have remaining error
    $amountsRounded = array_map(function(float $a) use(&$totalErr) {
        // Correct error upwards if we still have some of the total error to
        // spend, take the float error into account
        if($totalErr > 0) {
            $rounded = ceil($a);
            $totalErr -= $rounded - $a;
        }
        else {
            // We have to increase the error since we are flooring
            $rounded = floor($a);
            $totalErr -= $rounded - $a;
        }

        return (int)$rounded;
    }, $amounts);

    if(abs($totalErr) >= 0.01) {
        $msg = sprintf(
            "%s: Failed to properly spread rounding error, totalErr: %f, on: %s",
            __FUNCTION__,
            $totalErr,
            json_encode($amounts)
        );

        Mage::log($msg);

        if(Mage::getIsDeveloperMode()) {
            throw new Exception($msg);
        }
    }

    return $amountsRounded;
}


/**
 * @param Array<string, Amount<float>> $points
 * @return Array<string, Points>
 */
function spreadPoints(array $points): array {
    $keys = array_keys($points);
    $amounts = calculateRoundingSpread(array_map("Points\\Core\\Amount::value", $points));
    $taxes = calculateRoundingSpread(array_map("Points\\Core\\Amount::tax", $points));

    return array_combine(
        // Preserve keys
        $keys,
        array_map(
            /**
             * @param string $k
             */
            function($k) use($amounts, $taxes, $points): Points {
                return new Points(
                    $amounts[$k],
                    $points[$k]->getValueIncludesTax(),
                    $taxes[$k]
                );
            },
            $keys
        )
    );
}

/**
 * Spreads the rounding into decimal numbers with specified precision.
 *
 * @param Array<string, float> $prices
 * @return Array<string, float>
 */
function spreadPrice(array $prices, int $precision = 2): array {
    $multiplier = pow(10, $precision);

    return array_map(function(int $a) use($multiplier, $precision): float {
        return round($a / $multiplier, $precision);
    }, calculateRoundingSpread(array_map(
        /**
         * @param float $a
         * @return float
         */
        function($a) use($multiplier) {
            return $a * $multiplier;
        },
        $prices
    )));
}

/**
 * @template T
 * @param Array<T, Currency> $currency
 * @return Array<T, Currency>
 */
function spreadCurrency(array $currency): array {
    $keys = array_keys($currency);
    $amounts = spreadPrice(array_map("Points\\Core\\Amount::value", $currency));
    $taxes = spreadPrice(array_map("Points\\Core\\Amount::tax", $currency));

    return array_combine(
        // Preserve keys
        $keys,
        array_map(
            /**
             * @param T $k
             */
            function($k) use($amounts, $taxes, $currency): Currency {
                return new Currency(
                    $amounts[$k],
                    $currency[$k]->getValueIncludesTax(),
                    $taxes[$k],
                    $currency[$k]->getIncluded()
                );
            },
            $keys
        )
    );
}
