<?php

declare(strict_types=1);

use GraphQL\Deferred;
use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use Points\Core\Extension\Order as OrderExt;
use Points\Core\Extension\Product as ProductExt;
use Points\Core\Extension\Quote as QuoteExt;
use Points\Core\Extension\QuoteAddress;
use Points\Core\Extension\QuoteItem as QuoteItemExt;
use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Points\Core\Schema\CustomerPointsBalance;
use Points\Core\Schema\OrderSelected;
use Points\Core\Schema\Quote;
use Points\Core\Schema\QuoteRejectionReason;
use Points\Core\Schema\QuoteRejectionReasonBalance;
use Points\Core\Schema\QuoteRejectionReasonTotalLimit;
use Points\Core\Schema\QuoteRejectionReasonOrderLimit;
use Points\Core\Schema\QuoteItem;
use Points\Core\Schema\QuoteItemSelected;
use Points\Core\Schema\QuoteSelected;
use Points\Core\Schema\TotalInterface;
use Points\Core\Total;
use Points\Core\UnionBuilder;

class Points_Core_Model_Schema extends MageQL_Core_Model_Schema_Abstract {
    const SUCCESS = "success";
    const NOT_MODIFIED = "notModified";
    const ERROR_INVALID_POINT_TYPE = "errorInvalidPointType";
    const ERROR_POINT_TYPE_DOES_NOT_APPLY = "errorPointTypeDoesNotApply";
    const ERROR_POINT_REDEMPTION_NOT_ALLOWED = "errorPointRedemptionNotAllowed";
    const ERROR_CUSTOMER_NOT_LOGGED_IN = "errorCustomerNotLoggedIn";

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "PointsCustomerBalance":
            return $this->object("A customer points balance")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsSpendingLimit":
            return $this->object("A point spending limit for a time window")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsTotal":
            return $this->object("A total in points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsProduct":
            return $this->object("A product price in points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsProductPoints":
            return $this->object("Point price of a product or item")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsProductCurrency":
            return $this->object("Currency price of a product or item")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuote":
            return $this->object("An available point payment method")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteItem":
            return $this->object("A quote item price in points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteItemSelected":
            return $this->object("A quote item price in points for the selected payment")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteItemPoints":
            return $this->object("Point price of a quote item")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteItemCurrency":
            return $this->object("Currency price of a quote item")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteItemSelectedPoints":
            return $this->object("Point price of a quote item")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteItemSelectedCurrency":
            return $this->object("Currency price of a quote item")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuotePoints":
            return $this->object("Points price of the quote")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteCurrency":
            return $this->object("Currency price of the quote")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteDiscount":
            return $this->object("Discount split on points and currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteDiscountPoints":
            return $this->object("Discount split on points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteDiscountCurrency":
            return $this->object("Discount split on currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteRejectionReason":
            return new UnionBuilder(
                "Reason to why a point payment cannot be used with the quote",
                [
                    "PointsQuoteRejectionReasonBalance",
                    "PointsQuoteRejectionReasonTotalLimit",
                    "PointsQuoteRejectionReasonOrderLimit",
                ],
                function(QuoteRejectionReason $object): string {
                    if($object instanceof QuoteRejectionReasonBalance) {
                        return "PointsQuoteRejectionReasonBalance";
                    }

                    if($object instanceof QuoteRejectionReasonTotalLimit) {
                        return "PointsQuoteRejectionReasonTotalLimit";
                    }

                    if($object instanceof QuoteRejectionReasonOrderLimit) {
                        return "PointsQuoteRejectionReasonOrderLimit";
                    }

                    throw new Exception("Unknown limit");
                }
            );

        case "PointsQuoteRejectionReasonBalance":
            return $this->object("Point payment is rejected because the customer balance is too low.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteRejectionReasonTotalLimit":
            return $this->object("Point payment is rejected because the customer has spent too much.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteRejectionReasonOrderLimit":
            return $this->object("Point payment is rejected because the quote has reached the order limit.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteShipping":
            return $this->object("Shipping price split on points and currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteShippingPoints":
            return $this->object("Shipping price split on points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteShippingCurrency":
            return $this->object("Shipping price split on currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelected":
            return $this->object("The selected payment method")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedPoints":
            return $this->object("Points price of the quote with selected point payment")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedCurrency":
            return $this->object("Currency price of the quote with selected point payment")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedDiscount":
            return $this->object("Discount split on points and currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedDiscountPoints":
            return $this->object("Discount split on points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedDiscountCurrency":
            return $this->object("Discount split on currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedShipping":
            return $this->object("Shipping price split on points and currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedShippingPoints":
            return $this->object("Shipping price split on points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsQuoteSelectedShippingCurrency":
            return $this->object("Shipping price split on currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsOrderSelected":
            return $this->object("Information about points spent on an order")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsOrderSelectedDiscount":
            return $this->object("Discount split on points and currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsOrderSelectedShipping":
            return $this->object("Shipping price split on points and currency")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "QuotePointsSetResult":
            return $this->object("Result of quotePointsSet")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuotePointsSetResultType":
            return $this->enum("Result type of quotePointsSet", [
                self::SUCCESS => [
                    "description" => "Successfully set the points",
                ],
                self::ERROR_INVALID_POINT_TYPE => [
                    "description" => "The supplied point type is invalid",
                ],
                self::ERROR_POINT_TYPE_DOES_NOT_APPLY => [
                    "description" => "The supplied point type does not apply to the quote",
                ],
                self::ERROR_POINT_REDEMPTION_NOT_ALLOWED => [
                    "description" => "The currently logged in customer is not allowed to redeem points of this type",
                ],
                self::ERROR_CUSTOMER_NOT_LOGGED_IN => [
                    "description" => "Customer must be logged in to be able to redeem points",
                ],
            ]);

        case "QuotePointsRemoveResult":
            return $this->object("Result of quotePointsRemove")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "QuotePointsRemoveResultType":
            return $this->enum("Result type of quotePointsRemove", [
                self::SUCCESS => [
                    "description" => "Successfully removed the points payment",
                ],
                self::NOT_MODIFIED => [
                    "description" => "No points payment to remove",
                ],
            ]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "ConfigurationOptionItem":
            return [
                "pointsPrices" => $this->field("[PointsProduct!]!", "")
                    ->setResolver(function(
                        MageQL_Catalog_Model_Product_Configurable_Option $option,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): Deferred {
                        $customerGroupId = (int)Mage::getSingleton("customer/session")->getCustomerGroupId();
                        $collector = Points_Core_Model_Product_Price_Collector::instance($ctx->getStore(), $customerGroupId);
                        $product = $option->getProduct();

                        $collector->queue($product);

                        return new Deferred(function() use($product, $collector) {
                            return $collector->getPrices($product);
                        });
                    }),
            ];

        case "Customer":
            return [
                "points" => $this->field("[PointsCustomerBalance!]!", "List of point currencies which are available for the customer")
                    ->setResolver(function(
                        Mage_Customer_Model_Customer $customer,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): array {
                        $helper = Mage::helper("points_core");
                        $providers = $helper->getEnabledTypeProviders($ctx->getStore());

                        // TODO: Refactor
                        return array_map(function(
                            string $key,
                            ProviderInterface $provider
                        ) use($customer): CustomerPointsBalance {
                            return new CustomerPointsBalance($key, $provider, $customer);
                        }, array_keys($providers), $providers);
                    }),
            ];

        case "PointsCustomerBalance":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "redemptionAllowed" => $this->field("Boolean!", "If this customer is allowed to redeem this point currency"),
                // TODO: Merge the two points fields?
                "points" => $this->field("Int!", "Total amount of points the customer has available for purchase"),
                "pointsIncludeVat" => $this->field("Boolean!", "If the customer points balance includes VAT"),
                "pointsSpent" => $this->field("Int!", "Total amount of points spent by the customer in this shop"),
                "spendingLimit" => $this->field("PointsSpendingLimit", "Limit to the number of points spent in a time window if any"),
            ];

        case "PointsSpendingLimit":
            return [
                "spent" => $this->field("Int!", "Amount of points spent in the current time window"),
                "limit" => $this->field("Int!", "Point limit per time window"),
                "remaining" => $this->field("Int!", "Amount of points spent in the current time window"),
                "includesVat" => $this->field("Boolean!", "If the limit is including VAT"),
                "resetsAt" => $this->field("String!", "Datetime when the current time window ends"),
            ];

        case "ListProduct":
        case "ProductDetail":
            return [
                "pointsPrices" => $this->field("[PointsProduct!]!", "List of available point payment options for this product")
                    ->setResolver(function(
                        Mage_Catalog_Model_Product $product,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): Deferred {
                        $customerGroupId = (int)Mage::getSingleton("customer/session")->getCustomerGroupId();
                        $collector = Points_Core_Model_Product_Price_Collector::instance($ctx->getStore(), $customerGroupId);

                        $collector->queue($product);

                        return new Deferred(function() use($product, $collector) {
                            return $collector->getPrices($product);
                        });
                    }),
            ];

        case "PointsProduct":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "pointsOnly" => $this->field("Boolean!", "If the product requires point payment")
                    ->setResolver(
                        function(
                            Points_Core_Model_Product_Price_Item $item,
                            array $unusedArgs,
                            MageQL_Core_Model_Context $ctx
                        ): Deferred {
                            $product = $item->getProduct();
                            $collector = Points_Core_Model_Product_PointsPaymentRequired::instance($ctx->getStore());

                            $collector->queue($product);

                            return new Deferred(function() use($product, $collector): bool {
                                return $collector->getPointsPaymentRequired($product);
                            });
                        }
                    ),
                "currencyRequired" => $this->field("Boolean!", "If the product requires some amount of currency payment"),
                "points" => $this->field("PointsProductPoints!", "Price of the product in points"),
                "currency" => $this->field("PointsProductCurrency!", "Price of the product in currency, equivalent to price without points"),
            ];

        case "PointsProductPoints":
            return [
                "min" => $this->field("PointsTotal!", "Minimum number of points possible to spend"),
                "max" => $this->field("PointsTotal!", "Maximum number of points possible to spend"),
                "value" => $this->field("PointsTotal!", "Total value of the points"),
            ];

        case "PointsProductCurrency":
            return [
                "min" => $this->field("ProductPrice!", "Minimum amount of currency to spend on the product"),
                "max" => $this->field("ProductPrice!", "Maximum amount of currency to spend on the product"),
                "value" => $this->field("ProductPrice!", "Total value of the product in currency"),
            ];

        case "Mutation":
            return [
                "quotePointsSet" => $this->field("QuotePointsSetResult!", "Attempts to set a certain number of points of a specific type to spend on this quote")
                    ->addArgument("id", $this->argument("ID!", "Point type identifier"))
                    ->addArgument("points", $this->argument("Int!", "Amount of points to spend on this quote"))
                    ->addArgument("incVat", $this->argument("Boolean!", "If the amount of points to spend includes VAT"))
                    ->setResolver(
                        /**
                         * @param mixed $unusedSrc
                         * @param array{id:string, points:int, incVat:bool} $args
                         */
                        function($unusedSrc, array $args): Varien_Object {
                            return $this->setPoints($args["id"], function(Mage_Sales_Model_Quote $quote) use($args): void {
                                /**
                                 * @var QuoteExt $quote
                                 */
                                $quote->setPointsWanted($args["points"]);
                                $quote->setPointsWantedIncludesTax($args["incVat"]);
                            });
                        }
                ),
                "quotePointsSetToMaximum" => $this->field("QuotePointsSetResult!", "Attempts to set the maximum number of points available to spend for a specific type of points on the current quote")
                    ->addArgument("id", $this->argument("ID!", "Point type identifier"))
                    ->setResolver(
                        /**
                         * @param mixed $unusedSrc
                         * @param array{id:string} $args
                         */
                        function($unusedSrc, array $args): Varien_Object {
                            return $this->setPoints(
                                $args["id"],
                                function(
                                    Mage_Sales_Model_Quote $quote,
                                    Mage_Customer_Model_Customer $customer,
                                    ProviderInterface $provider
                                ): void {
                                    /**
                                     * @var QuoteExt $quote
                                     */
                                    $points = $provider->getCustomerPointsBalance($quote->getStore(), $customer);
                                    $incVat = $provider->getCustomerPointsBalanceIncludesTax($quote->getStore(), $customer);

                                    $quote->setPointsWanted($points);
                                    $quote->setPointsWantedIncludesTax($incVat);
                                }
                            );
                        }
                    ),
                "quotePointsRemove" => $this->field("QuotePointsRemoveResult!", "Removes the point payment from the quote, if any")
                    ->setResolver(function(): Varien_Object {
                        $model = Mage::getSingleton("mageql_sales/quote");
                        /**
                         * @var QuoteExt $quote
                         */
                        $quote = $model->getQuote();
                        $result = $quote->getPointsType() ? self::SUCCESS : self::NOT_MODIFIED;

                        $quote->setPointsType(null);
                        $quote->setPointsWanted(0);
                        $quote->setPointsWantedIncludesTax(true);
                        $quote->setTotalsCollectedFlag(false);

                        $model->saveSessionQuote();

                        return new Varien_Object([
                            "result" => $result,
                            "quote" => $quote,
                        ]);
                    }),
            ];

        case "PointsTotal":
            return [
                "exVat" => $this->field("Int!", "Price excluding VAT")
                    ->setResolver(function(TotalInterface $src, array $unusedArgs, MageQL_Core_Model_Context $ctx): int {
                        return $src->getExVat($ctx);
                    }),
                "incVat" => $this->field("Int!", "Price including VAT")
                    ->setResolver(function(TotalInterface $src, array $unusedArgs, MageQL_Core_Model_Context $ctx): int {
                        return $src->getIncVat($ctx);
                    }),
                "vat" => $this->field("Int!", "VAT amount")
                    ->setResolver(function(TotalInterface $src, array $unusedArgs, MageQL_Core_Model_Context $ctx): int {
                        return $src->getVat($ctx);
                    }),
            ];

        case "Order":
            return [
                "selectedPointPayment" => $this->field("PointsOrderSelected", "The selected quote point payment if any")
                    ->setResolver(function(
                        Mage_Sales_Model_Order $order
                    ) {
                        /**
                         * @var OrderExt $order
                         */
                        $helper = Mage::helper("points_core");
                        $type = $order->getPointsType();

                        if( ! $type) {
                            return null;
                        }

                        $provider = $helper->getTypeProvider($type);

                        return new OrderSelected($order, $type, $provider);
                    }),
            ];

        case "Quote":
            return [
                "requiresPointPayment" => $this->field("Boolean!", "If this quote requires a point payment to be used")
                    ->setResolver(
                        /**
                         * @param QuoteExt $src
                         */
                        function(
                            Mage_Sales_Model_Quote $src
                        ): bool {
                            foreach($src->getAllItems() as $item) {
                                /**
                                 * @var ProductExt $product
                                 */
                                $product = $item->getProduct();

                                if($product->getPointsPaymentRequired()) {
                                    return true;
                                }
                            }

                            return false;
                        }
                    ),
                "availablePointPayments" => $this->field("[PointsQuote!]!", "List of available point payment methods which can be used in addition to the standard payment method")
                    ->setResolver(
                    /**
                     * @param QuoteExt $src
                     */
                    function(
                        Mage_Sales_Model_Quote $src
                    ): array {
                        $cache = Mage::getSingleton("points_core/cache_quote");

                        return array_map(function($v) use($src) {
                            return new Quote($v["type"], $v["provider"], $src, $v["total"]);
                        }, $cache->getAvailableTotals($src));
                    }),
                "selectedPointPayment" => $this->field("PointsQuoteSelected", "The selected quote point payment if any")
                    ->setResolver(function(
                        Mage_Sales_Model_Quote $quote
                    ) {
                        /**
                         * @var QuoteExt $quote
                         */
                        $cache = Mage::getSingleton("points_core/cache_quote");
                        $selected = $cache->getSelectedTotal($quote);

                        if( ! $selected) {
                            return null;
                        }

                        return new QuoteSelected($selected["type"], $selected["provider"], $quote, $selected["total"]);
                    }),
            ];

        case "QuoteItem":
            return [
                "requiresPointPayment" => $this->field("Boolean!", "If this quote item requires a point payment to be used")
                    ->setResolver(
                        function(
                            Mage_Sales_Model_Quote_Item $src
                        ): bool {
                            /**
                             * @var ProductExt $product
                             */
                            $product = $src->getProduct();

                            return (bool)$product->getPointsPaymentRequired();
                        }
                    ),
                "availablePointPayments" => $this->field("[PointsQuoteItem!]!", "List of available point payment methods which are available for this quote item")
                    ->setResolver(function(
                        Mage_Sales_Model_Quote_Item $item
                    ): array {
                        /**
                         * @var QuoteItemExt $item
                         */
                        $cache = Mage::getSingleton("points_core/cache_quote");

                        return array_map(function($v) use($item) {
                            return new QuoteItem($v["type"], $v["provider"], $v["total"], $item);
                        }, $cache->getAvailableTotals($item->getQuote()));
                    }),
                "selectedPointPayment" => $this->field("PointsQuoteItemSelected", "Information about the currently selected point payment for this quote item, if any is selected and available for this item")
                    ->setResolver(function(
                        Mage_Sales_Model_Quote_Item $item
                    ): ?QuoteItemSelected {
                        /**
                         * @var QuoteItemExt $item
                         */
                        $cache = Mage::getSingleton("points_core/cache_quote");
                        $selected = $cache->getSelectedTotal($item->getQuote());

                        if( ! $selected) {
                            return null;
                        }

                        return new QuoteItemSelected($selected["type"], $selected["provider"], $selected["total"], $item);
                    }),
            ];

        case "PointsQuote":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "rejectionReasons" => $this->field("[PointsQuoteRejectionReason!]!", "List of current reasons to why point payment is rejected"),
                "currencyRequired" => $this->field("Boolean!", "If the product requires some amount of currency payment"),
                "points" => $this->field("PointsQuotePoints!", "Price of the quote in points"),
                "currency" => $this->field("PointsQuoteCurrency!", "Price of the quote in currency, equivalent to price without points"),
                "shipping" => $this->field("PointsQuoteShipping", "Price of shipping, split into points and currency"),
                "discount" => $this->field("PointsQuoteDiscount", "Discount, split into points and currency"),
            ];

        case "PointsQuotePoints":
            return [
                "min" => $this->field("PointsTotal!", "Minimum number of points possible to spend"),
                "max" => $this->field("PointsTotal!", "Maximum number of points possible to spend"),
                "value" => $this->field("PointsTotal!", "Total value of the points"),
                "available" => $this->field("PointsTotal!", "Maximum available points to spend"),
            ];

        case "PointsQuoteCurrency":
            return [
                "min" => $this->field("ProductPrice!", "Minimum amount of currency to spend on the quote"),
                "max" => $this->field("ProductPrice!", "Maximum amount of currency to spend on the quote"),
                "value" => $this->field("ProductPrice!", "Total value of the quote in currency"),
            ];

        case "PointsQuoteRejectionReasonBalance":
            return [
                "requested" => $this->field("Int!", "Requested amount of points"),
                "includesVat" => $this->field("Boolean!", "If the requested amount includes VAT."),
                "balance" => $this->field("Int!", "Customer balance available"),
            ];

        case "PointsQuoteRejectionReasonTotalLimit":
            return [
                "requested" => $this->field("Int!", "Requested amount of points"),
                "includesVat" => $this->field("Boolean!", "If the requested amount includes VAT."),
                "remaining" => $this->field("Int!", "Amount of points spent in the current time window"),
                "limit" => $this->field("Int!", "Point limit per time window"),
                "resetsAt" => $this->field("String!", "Datetime when the current time window ends"),
            ];

        case "PointsQuoteRejectionReasonOrderLimit":
            return [
                "requested" => $this->field("Int!", "Requested amount of points"),
                "includesVat" => $this->field("Boolean!", "If the requested amount includes VAT."),
                "limit" => $this->field("Int!", "Order total limit"),
            ];

        case "PointsQuoteShipping":
            return [
                "points" => $this->field("PointsQuoteShippingPoints", "Points value of shipping, if any"),
                "currency" => $this->field("PointsQuoteShippingCurrency!", "Currency value of shipping"),
            ];

        case "PointsQuoteShippingPoints":
            return [
                "min" => $this->field("PointsTotal!", "Minimum number of points possible to spend"),
                "max" => $this->field("PointsTotal!", "Maximum number of points possible to spend"),
                "value" => $this->field("PointsTotal!", "Total value of the points"),
            ];

        case "PointsQuoteShippingCurrency":
            return [
                "min" => $this->field("ProductPrice!", "Minimum amount of currency to spend on shipping"),
                "max" => $this->field("ProductPrice!", "Maximum amount of currency to spend on shipping"),
                "value" => $this->field("ProductPrice!", "Total value of the quote in currency"),
            ];

        case "PointsQuoteDiscount":
            return [
                "points" => $this->field("PointsQuoteDiscountPoints!", "Currency value of shipping, if any"),
                "currency" => $this->field("PointsQuoteDiscountCurrency!", "Currency value of shipping"),
            ];

        case "PointsQuoteDiscountPoints":
            return [
                "min" => $this->field("PointsTotal!", "Minimum amount of discount in points"),
                "max" => $this->field("PointsTotal!", "Maximum amount of discount in points"),
                "value" => $this->field("PointsTotal!", "Total value of the discount in points, might not cover the full discount if there are items which do not have a point price"),
            ];

        case "PointsQuoteDiscountCurrency":
            return [
                "min" => $this->field("ProductPrice!", "Minimum amount of currency to spend on shipping"),
                "max" => $this->field("ProductPrice!", "Maximum amount of currency to spend on shipping"),
                "value" => $this->field("ProductPrice!", "Total value of the quote in currency"),
            ];

        case "PointsQuoteItem":
            return array_merge($registry->getFieldBuilders("PointsProduct"), [
                "points" => $this->field("PointsProductPoints!", "Price of the quote item in points"),
                "currency" => $this->field("PointsProductCurrency!", "Price of the quote item in currency"),
            ]);

        case "PointsQuoteItemSelected":
            return array_merge($registry->getFieldBuilders("PointsQuoteItem"), [
                "points" => $this->field("PointsQuoteItemSelectedPoints!", "Price of the quote item in points"),
                "currency" => $this->field("PointsQuoteItemSelectedCurrency!", "Price of the quote item in currency"),
            ]);

        case "PointsQuoteItemPoints":
            return $registry->getFieldBuilders("PointsProductPoints");

        case "PointsQuoteItemCurrency":
            return $registry->getFieldBuilders("PointsProductCurrency");

        case "PointsQuoteItemSelectedPoints":
            return array_merge($registry->getFieldBuilders("PointsProductPoints"), [
                "selected" => $this->field("PointsTotal!", "Amount of points to spend on the quote item"),
            ]);

        case "PointsQuoteItemSelectedCurrency":
            return array_merge($registry->getFieldBuilders("PointsProductCurrency"), [
                "remaining" => $this->field("ProductPrice!", "Amount of currency to spend on the quote item"),
            ]);

        case "PointsQuoteSelected":
            return array_merge($registry->getFieldBuilders("PointsQuote"), [
                "points" => $this->field("PointsQuoteSelectedPoints!", "Price of the quote in points"),
                "currency" => $this->field("PointsQuoteSelectedCurrency!", "Price of the quote in currency, equivalent to price without points"),
                "shipping" => $this->field("PointsQuoteSelectedShipping", "Pirice of shipping, split into points and currency"),
                "discount" => $this->field("PointsQuoteSelectedDiscount", "Discount, split into points and currency"),
            ]);

        case "PointsQuoteSelectedPoints":
            return array_merge($registry->getFieldBuilders("PointsQuotePoints"), [
                "selected" => $this->field("PointsTotal!", "Amount of points to spend on the quote"),
            ]);

        case "PointsQuoteSelectedCurrency":
            return array_merge($registry->getFieldBuilders("PointsQuoteCurrency"), [
                "remaining" => $this->field("ProductPrice!", "Remaining currency required to spend with the current selected amount of points"),
            ]);

        case "PointsQuoteSelectedDiscount":
            return array_merge($registry->getFieldBuilders("PointsQuoteDiscount"), [
                "points" => $this->field("PointsQuoteSelectedDiscountPoints!", "Points value of discount, if any"),
                "currency" => $this->field("PointsQuoteSelectedDiscountCurrency!", "Currency value of discount"),
            ]);

        case "PointsQuoteSelectedDiscountPoints":
            return array_merge($registry->getFieldBuilders("PointsQuoteDiscountPoints"), [
                "selected" => $this->field("PointsTotal!", "Discount in points"),
            ]);

        case "PointsQuoteSelectedDiscountCurrency":
            return array_merge($registry->getFieldBuilders("PointsQuoteDiscountCurrency"), [
                "remaining" => $this->field("ProductPrice!", "Discount in currency"),
            ]);

        case "PointsQuoteSelectedShipping":
            return array_merge($registry->getFieldBuilders("PointsQuoteShipping"), [
                "points" => $this->field("PointsQuoteSelectedShippingPoints!", "Currency value of shipping, if any"),
                "currency" => $this->field("PointsQuoteSelectedShippingCurrency!", "Currency value of shipping"),
            ]);

        case "PointsQuoteSelectedShippingPoints":
            return array_merge($registry->getFieldBuilders("PointsQuoteDiscountPoints"), [
                "selected" => $this->field("PointsTotal!", "Shipping in points"),
            ]);

        case "PointsQuoteSelectedShippingCurrency":
            return array_merge($registry->getFieldBuilders("PointsQuoteDiscountCurrency"), [
                "remaining" => $this->field("ProductPrice!", "Shipping in currency"),
            ]);

        case "PointsOrderSelected":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String", "Label of the point currency type"),
                "points" => $this->field("PointsTotal!", "Price of the quote in points"),
                "currency" => $this->field("ProductPrice!", "Price of the quote in currency, equivalent to price without points"),
                "shipping" => $this->field("PointsOrderSelectedShipping", "Price of shipping, split into points and currency"),
                "discount" => $this->field("PointsOrderSelectedDiscount", "Discount, split into points and currency"),
            ];

        case "PointsOrderSelectedDiscount":
            return [
                "points" => $this->field("PointsTotal!", "Points value of discount"),
                "currency" => $this->field("ProductPrice!", "Currency value of discount"),
            ];

        case "PointsOrderSelectedShipping":
            return [
                "points" => $this->field("PointsTotal!", "Points value of shipping"),
                "currency" => $this->field("ProductPrice!", "Currency value of shipping"),
            ];

        case "QuotePointsRemoveResult":
            return [
                "result" => $this->field("QuotePointsRemoveResultType!", "Type of result from quotePointsRemove"),
                "quote" => $this->field("Quote!", "Current session quote"),
            ];

        case "QuotePointsSetResult":
            return [
                "result" => $this->field("QuotePointsSetResultType!", "Type of result from quotePointsSet"),
                "quote" => $this->field("Quote!", "Current session quote"),
            ];
        }

        return [];
    }

    /**
     * Fetches and validates the point type provider against the currenti quote
     * and customer, then calls setProps to modify the requested point values.
     *
     * It will automatically set points type on the quote before the closure is
     * called, as well as set the totals collected flag to false.
     *
     * @param callable(
     *   QuoteExt,
     *   Mage_Customer_Model_Customer,
     *   ProviderInterface
     * ):void $setProps
     */
    public function setPoints(string $id, $setProps): Varien_Object {
        $helper = Mage::helper("points_core");
        $model = Mage::getSingleton("mageql_sales/quote");
        /**
         * @var QuoteExt $quote
         */
        $quote = $model->getQuote();
        $store = $quote->getStore();
        $customer = Mage::getSingleton("customer/session")->getCustomer();

        $provider = $helper->getTypeProvider($id);

        if( ! $provider) {
            return new Varien_Object([
                "result" => self::ERROR_INVALID_POINT_TYPE,
                "quote" => $quote,
            ]);
        }

        if( ! $provider->appliesTo($quote)) {
            return new Varien_Object([
                "result" => self::ERROR_POINT_TYPE_DOES_NOT_APPLY,
                "quote" => $quote,
            ]);
        }

        if( ! $customer->getId()) {
            return new Varien_Object([
                "result" => self::ERROR_CUSTOMER_NOT_LOGGED_IN,
                "quote" => $quote,
            ]);
        }

        if( ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            return new Varien_Object([
                "result" => self::ERROR_POINT_REDEMPTION_NOT_ALLOWED,
                "quote" => $quote,
            ]);
        }

        $quote->setPointsType($id);

        $setProps($quote, $customer, $provider);

        $quote->setTotalsCollectedFlag(false);

        $model->saveSessionQuote();

        // TODO: Verify it got set
        return new Varien_Object([
            "result" => self::SUCCESS,
            "quote" => $quote,
        ]);
    }
}
