<?php

declare(strict_types=1);

use Points\Core\Extension\Creditmemo;
use Points\Core\Extension\Invoice;
use Points\Core\Extension\Order;
use Points\Core\Extension\Quote;
use Points\Core\Extension\Product;
use Points\Core\Extension\QuoteAddress;
use Points\Core\QuoteAddressTotal;
use Points\Core\ProviderInterface;

class Points_Core_Model_Observer extends Mage_Core_Model_Abstract {
    public function salesQuoteCollectTotalsBefore(Varien_Event_Observer $observer): void {
        /**
         * @var Quote $quote
         */
        $quote = $observer->getQuote();
        $helper = Mage::helper("points_core");
        $store = $quote->getStore();
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);

        if($quote->getPointsType() && $quote->getPointsPoints() && ! $quote->hasPointsWanted()) {
            if($pricesIncludeTax) {
                $quote->setPointsWanted($quote->getPointsPoints() + $quote->getPointsTaxPoints());
                $quote->setPointsWantedIncludesTax(true);
            }
            else {
                $quote->setPointsWanted($quote->getPointsPoints());
                $quote->setPointsWantedIncludesTax(false);
            }
        }

        // TODO: Validate points type, drop if not valid
        // Make sure we reset in case we fail the calculation
        $helper->resetQuotePoints($quote);
    }

    public function salesQuoteCollectTotalsAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        $helper = Mage::helper("points_core");

        $helper->resetQuotePoints($quote);

        if( ! $quote->getPointsType()) {
            return;
        }

        /**
         * @var QuoteAddress $addr
         */
        foreach($quote->getAllAddresses() as $addr) {
            $quote->setPointsValue($quote->getPointsValue() + (int)$addr->getPointsValue());
            $quote->setPointsTaxValue($quote->getPointsTaxValue() + (int)$addr->getPointsTaxValue());
            $quote->setPointsPoints($quote->getPointsPoints() + (int)$addr->getPointsPoints());
            $quote->setPointsTaxPoints($quote->getPointsTaxPoints() + (int)$addr->getPointsTaxPoints());
            $quote->setPointsAmount($quote->getPointsAmount() + (float)$addr->getPointsAmount());
            $quote->setPointsTaxAmount($quote->getPointsTaxAmount() + (float)$addr->getPointsTaxAmount());
            $quote->setBasePointsAmount($quote->getBasePointsAmount() + (float)$addr->getBasePointsAmount());
            $quote->setBasePointsTaxAmount($quote->getBasePointsTaxAmount() + (float)$addr->getBasePointsTaxAmount());

            $quote->setPointsAmountIncluded($quote->getPointsAmountIncluded() + (float)$addr->getPointsAmountIncluded());
            $quote->setPointsTaxAmountIncluded($quote->getPointsTaxAmountIncluded() + (float)$addr->getPointsTaxAmountIncluded());
            $quote->setPointsAmountExcluded($quote->getPointsAmountExcluded() + (float)$addr->getPointsAmountExcluded());
            $quote->setPointsTaxAmountExcluded($quote->getPointsTaxAmountExcluded() + (float)$addr->getPointsTaxAmountExcluded());

            // Only set if shipping actually has a value
            if($addr->getPointsShippingValue() !== null) {
                $quote->setPointsShippingValue(
                    ($quote->getPointsShippingValue() ?: 0) + $addr->getPointsShippingValue()
                );
            }

            if($addr->getPointsShippingTaxValue() !== null) {
                $quote->setPointsShippingTaxValue(
                    ($quote->getPointsShippingTaxValue() ?: 0) + $addr->getPointsShippingTaxValue()
                );
            }

            $quote->setPointsShippingPoints($quote->getPointsShippingPoints() + $addr->getPointsShippingPoints());
            $quote->setPointsShippingTaxPoints($quote->getPointsShippingTaxPoints() + $addr->getPointsShippingTaxPoints());
            $quote->setPointsShippingTotal($quote->getPointsShippingTotal() + $addr->getPointsShippingTotal());
            $quote->setPointsShippingTaxTotal($quote->getPointsShippingTaxTotal() + $addr->getPointsShippingTaxTotal());
            $quote->setBasePointsShippingTotal($quote->getBasePointsShippingTotal() + $addr->getBasePointsShippingTotal());
            $quote->setBasePointsShippingTaxTotal($quote->getBasePointsShippingTaxTotal() + $addr->getBasePointsShippingTaxTotal());

            $quote->setPointsDiscountValue($quote->getPointsDiscountValue() + $addr->getPointsDiscountValue());
            $quote->setPointsDiscountTaxValue($quote->getPointsDiscountTaxValue() + $addr->getPointsDiscountTaxValue());
            $quote->setPointsDiscountPoints($quote->getPointsDiscountPoints() + $addr->getPointsDiscountPoints());
            $quote->setPointsDiscountTaxPoints($quote->getPointsDiscountTaxPoints() + $addr->getPointsDiscountTaxPoints());
            $quote->setPointsDiscountTotal($quote->getPointsDiscountTotal() + $addr->getPointsDiscountTotal());
            $quote->setPointsDiscountTaxTotal($quote->getPointsDiscountTaxTotal() + $addr->getPointsDiscountTaxTotal());
            $quote->setBasePointsDiscountTotal($quote->getBasePointsDiscountTotal() + $addr->getBasePointsDiscountTotal());
            $quote->setBasePointsDiscountTaxTotal($quote->getBasePointsDiscountTaxTotal() + $addr->getBasePointsDiscountTaxTotal());
        }

        $quote->unsPointsWanted();
        $quote->unsPointsWantedIncludesTax();
    }

    public function salesQuoteSubmitBefore(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        /**
         * @var Order
         */
        $order = $observer->getOrder();
        $basePoints = abs($quote->getBasePointsAmount()) + abs($quote->getBasePointsTaxAmount());
        $pointsPaymentRequired = $basePoints >= 0.01;
        $customer = $quote->getCustomer();
        $store = $quote->getStore();
        $type = $quote->getPointsType();

        foreach($quote->getAllItems() as $item) {
            /**
             * @var Product $product
             */
            $product = $item->getProduct();

            if($product->getPointsPaymentRequired()) {
                $pointsPaymentRequired = true;
            }
        }

        if( ! $pointsPaymentRequired) {
            return;
        }

        if($basePoints < 0.01 || ! $type) {
            throw new Points_Core_PointPaymentRequiredException();
        }

        // Ok, we have some amount of spent points, verify

        $provider = $this->getQuoteProvider($quote);

        if( ! $provider->appliesTo($quote)) {
            throw new Exception(sprintf(
                "%s: Provider for points type '%s' does not apply to quote %d",
                __METHOD__,
                $type,
                $quote->getId()
            ));
        }

        if( ! $customer->getId()) {
            throw new Exception(sprintf(
                "%s: No customer for quote %d",
                __METHOD__,
                $quote->getId()
            ));
        }

        if( ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            throw new Exception(sprintf(
                "%s: Customer '%d' is not allowed to redeem points of type '%s' for quote %d",
                __METHOD__,
                $customer->getId(),
                $type,
                $quote->getId()
            ));
        }

        // Verify product minimum/maximum
        $totals = array_map(function(Mage_Sales_Model_Quote_Address $a) use($type, $provider): QuoteAddressTotal {
            return QuoteAddressTotal::fromQuoteAddress($a, $type, $provider);
        }, $quote->getAllAddresses());

        // We only check inc-tax, ex-tax should follow
        $pointsMin = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMin()));
        }, $totals));
        $pointsMax = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMax()));
        }, $totals));

        if($pointsMin > $quote->getPointsPoints()) {
            throw new Exception(sprintf(
                "%s: Quote '%d' does not have enough points, needs %d, saw %d",
                __METHOD__,
                $quote->getId(),
                $pointsMin,
                $quote->getPointsPoints()
            ));
        }

        if($pointsMax < $quote->getPointsPoints()) {
            throw new Exception(sprintf(
                "%s: Quote '%d' has too many '%s' points, at most %d, saw %d (all ex tax)",
                __METHOD__,
                $quote->getId(),
                $pointsMax,
                $type,
                $quote->getPointsPoints()
            ));
        }

        $customerPoints = $provider->getUpdatedCustomerPointsBalance($store, $customer);
        $customerPointsIncludesTax = $provider->getCustomerPointsBalanceIncludesTax($store, $customer);

        $requestedPoints = $customerPointsIncludesTax ?
            $quote->getPointsPoints() + $quote->getPointsTaxPoints() :
            $quote->getPointsPoints();

        if($customerPoints < $requestedPoints) {
            throw new Exception(sprintf(
                "%s: Quote '%d' requests %d points %s, customer %d has %d available",
                __METHOD__,
                $quote->getId(),
                $requestedPoints,
                $customerPointsIncludesTax ? "including tax" : "excluding tax",
                $customer->getId(),
                $customerPoints
            ));
        }

        $provider->onQuoteSubmitBefore($order, $quote);
    }

    /**
     * @param Order|Quote $obj
     */
    protected function getQuoteProvider($obj): ProviderInterface {
        $type = $obj->getPointsType();
        $helper = Mage::helper("points_core");

        if( ! $type) {
            throw new Exception(sprintf(
                "%s: Amount with no type on %s %d",
                __METHOD__,
                get_class($obj),
                $obj->getId()
            ));
        }

        $provider = $helper->getTypeProvider($type);

        if( ! $provider) {
            throw new Exception(sprintf(
                "%s: Provider for points type '%s' not found for %s %d",
                __METHOD__,
                $type,
                get_class($obj),
                $obj->getId()
            ));
        }

        return $provider;
    }

    public function salesOrderPlaceBefore(Varien_Event_Observer $observer): void {
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($order->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($order);

        $provider->onOrderPlaceBefore($order);
    }

    public function salesOrderPlaceAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($order->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($order);

        $provider->onOrderPlaceEnd($order);
    }

    public function salesQuoteSubmitFailure(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($quote->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($quote);

        $provider->onQuoteSubmitFailure($order, $quote);
    }

    public function salesQuoteSubmitAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($order->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($order);

        $provider->onQuoteSubmitAfter($order, $quote);
    }

    /**
     * Observer listening to `sales_order_invoice_save_after` to update deposit amount invoiced in order.
     */
    public function invoiceSaveAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Invoice
         */
        $invoice = $observer->getInvoice();
        /**
         * @var Order
         */
        $order = $invoice->getOrder();

        if($invoice->getBasePointsAmount()) {
            $order->setPointsPointsInvoiced($order->getPointsPointsInvoiced() + $invoice->getPointsPoints());
            $order->setPointsTaxPointsInvoiced($order->getPointsTaxPointsInvoiced() + $invoice->getPointsTaxPoints());
            $order->setPointsAmountInvoiced($order->getPointsAmountInvoiced() + $invoice->getPointsAmount());
            $order->setPointsTaxAmountInvoiced($order->getPointsTaxAmountInvoiced() + $invoice->getPointsTaxAmount());
            $order->setBasePointsAmountInvoiced($order->getBasePointsAmountInvoiced() + $invoice->getBasePointsAmount());
            $order->setBasePointsTaxAmountInvoiced($order->getBasePointsTaxAmountInvoiced() + $invoice->getBasePointsTaxAmount());
        }
    }

    /**
     * Observer listening to `sales_order_creditmemo_save_after` to update deposit amount refunded in order.
     */
    public function creditmemoSaveAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Creditmemo
         */
        $creditmemo = $observer->getCreditmemo();
        /**
         * @var Order
         */
        $order = $creditmemo->getOrder();

        if($creditmemo->getBasePointsAmount()) {
            $order->setPointsPointsRefunded($order->getPointsPointsRefunded() + $creditmemo->getPointsPoints());
            $order->setPointsTaxPointsRefunded($order->getPointsTaxPointsRefunded() + $creditmemo->getPointsTaxPoints());
            $order->setPointsAmountRefunded($order->getPointsAmountRefunded() + $creditmemo->getPointsAmount());
            $order->setPointsTaxAmountRefunded($order->getPointsTaxAmountRefunded() + $creditmemo->getPointsTaxAmount());
            $order->setBasePointsAmountRefunded($order->getBasePointsAmountRefunded() + $creditmemo->getBasePointsAmount());
            $order->setBasePointsTaxAmountRefunded($order->getBasePointsTaxAmountRefunded() + $creditmemo->getBasePointsTaxAmount());
        }
    }
}
