<?php

declare(strict_types=1);

namespace Points\Core;

use Mage;
use Mage_Customer_Model_Customer;
use Points_Core_Model_Limit_Total;
use Throwable;

use Points\Core\ProviderInterface;
use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Spatie\Snapshots\MatchesSnapshots;

/**
 * @var ?int
 */
$staticTime = null;

function time(): int {
    global $staticTime;

    return $staticTime ?: \time();
}

class CustomerTest extends TestCase {
    use MatchesSnapshots;

    /**
     * Psalm hasn't figured out that $staticTime is referenced outside.
     *
     * @psalm-suppress UnusedVariable/
     */
    public function setUp(): void {
        global $staticTime;

        $staticTime = strtotime("2020-01-15");

        try {
            MagentoManager::reset();
            MagentoManager::initAdmin();

            $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

            if($conn) {
               $conn->query("DELETE FROM points_limit_total")->execute();
               $conn->query("DELETE FROM points_limit_order")->execute();
               $conn->query("UPDATE sales_flat_order SET points_type = 'TEST_RENAMED' WHERE points_type = 'TEST'")->execute();
            }
        }
        finally {
            MagentoManager::reset();
        }
    }

    /**
     * Psalm hasn't figured out that $staticTime is referenced outside.
     *
     * @psalm-suppress UnusedVariable/
     */
    public function tearDown(): void {
        global $staticTime;

        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $staticTime = null;
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    public function loginCustomer(): Mage_Customer_Model_Customer {
        MagentoManager::init();

        $store = Mage::app()->getStore();

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        return $customer;
    }

    public function implementLimits(): void {
        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $limit = Mage::getModel("points_core/limit_total");
        $order = Mage::getModel("sales/order");

        $limit->loadByStoreTypeCustomerGroupId($store, "TEST", 0);
        $limit->addData([
            "store_id" => $store->getId(),
            "type" => "TEST",
            "customer_group_id" => 0,
            "time_interval" => Points_Core_Model_Limit_Total::INTERVAL_MONTH,
            "limit" => 10000,
        ]);
        $limit->save();

        $order->loadByIncrementId("TEST-1");
        $order->addData([
            "points_type" => "TEST",
            "points_points" => 1337,
        ]);
        $order->save();

        MagentoManager::reset();
    }

    public function testNoProviders(): void {
        $this->loginCustomer();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProviderNotEnabled(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(false);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProvider(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(12345);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProviderLimited(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(12345);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
