<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Mage;
use Mage_Tax_Model_Config;
use MageQL_Core_Model_Context;
use Mage_Sales_Model_Quote;

use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\Quote as QuoteExt;

class QuotePoints {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var QuoteAddressTotal
     */
    protected $total;

    /**
     * @var QuoteExt
     */
    protected $quote;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @param QuoteExt $quote
     */
    public function __construct(
        string $type,
        QuoteAddressTotal $total,
        Mage_Sales_Model_Quote $quote,
        ProviderInterface $provider
    ) {
        $this->type = $type;
        $this->total = $total;
        $this->quote = $quote;
        $this->provider = $provider;
    }

    public function getMin(): TotalInterface {
        $min = (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $this->total->getPointsMin()));
        $tax = array_sum(array_map("Points\Core\Amount::tax", $this->total->getPointsMin()));

        return new QuoteItemPointsTotal(
            $min,
            $tax
        );
    }

    public function getMax(): TotalInterface {
        $max = (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $this->total->getPointsMax()));
        $tax = (int)array_sum(array_map("Points\Core\Amount::tax", $this->total->getPointsMax()));

        return new QuoteItemPointsTotal(
            $max,
            $tax
        );
    }

    public function getValue(): TotalInterface {
        $value = (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $this->total->getPointsValue()));
        $tax = (int)array_sum(array_map("Points\Core\Amount::tax", $this->total->getPointsValue()));

        return new QuoteItemPointsTotal(
            $value,
            $tax
        );
    }

    public function getAvailable(): TotalInterface {
        $model = Mage::getResourceModel("points_core/limit_total");
        $max = (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $this->total->getPointsMax()));
        $tax = (int)array_sum(array_map("Points\Core\Amount::tax", $this->total->getPointsMax()));

        $balance = $this->provider->getCustomerPointsBalance($this->quote->getStore(), $this->quote->getCustomer());
        $balanceInclTax = $this->provider->getCustomerPointsBalanceIncludesTax($this->quote->getStore(), $this->quote->getCustomer());
        $limit = $model->getMatchingLimits($this->quote->getStore(), $this->quote->getCustomerGroupId(), [$this->type])[$this->type] ?? null;
        $remaining = $limit ? max(0, $limit->getLimit() - $limit->getSpentDuringLastWindow($this->quote->getCustomer())) : null;

        $taxFrac = ($max + $tax) > 0 ? $tax / ($max + $tax) : 0;
        $balanceTax = $balanceInclTax ? (int)ceil($balance * $taxFrac) : (int)floor($balance * $taxFrac / (1 + $taxFrac));
        $remainingTax = $remaining !== null ?
            ($balanceInclTax ? (int)ceil($remaining * $taxFrac) : (int)floor($remaining * $taxFrac / (1 + $taxFrac))) :
            $tax;

        $balanceExTax = $balanceInclTax ? $balance - $balanceTax : $balance;
        $remainingExTax = $remaining !== null ?
            ($balanceInclTax ? $remaining - $remainingTax : $remaining) :
            $max;

        // All three scale using the same tax, so the tax is proportional to the main value
        return new QuoteItemPointsTotal(
            max(0, min($max, $balanceExTax, $remainingExTax)),
            max(0, min($tax, $balanceTax, $remainingTax))
        );
    }
}
