<?php

declare(strict_types=1);

/**
 * @psalm-type ProductFilterInput array{
 *   code: string,
 *   value?: ?string,
 *   minValue?: ?float,
 *   maxValue?: ?float,
 *   incVat?: ?bool,
 * }
 */
class Points_Core_Model_Product_Filter
    implements MageQL_Catalog_Model_Filter_Product_Interface {
    /**
     * @return Array<MageQL_Catalog_Model_Product_Filter_Abstract>
     */
    public function getFilterableBy(
        Mage_Catalog_Model_Resource_Product_Collection $collection
    ): array {
        $data = [];
        $helper = Mage::helper("points_core");
        $store = $helper->getStore($collection->getStoreId());
        $providers = $helper->getEnabledTypeProviders($store);

        foreach($providers as $key => $provider) {
            $data[$key] = new Points_Core_Model_Product_Filter_Points($key, $provider, $collection);
        }

        return $data;
    }

    /**
     * Checks if a filter can be applied by this instance, if it can apply it
     * is applied to the collection and true is returned
     *
     * @param ProductFilterInput $filter
     */
    public function applyFilter(
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $filter
    ): bool {
        $helper = Mage::helper("points_core");

        if(strpos($filter["code"], "points:") === false) {
            return false;
        }

        $type = substr($filter["code"], strlen("points:"));
        $provider = $helper->getTypeProvider($type);

        if( ! $provider) {
            return false;
        }

        $minValue = $filter["minValue"] ?? null;
        $maxValue = $filter["maxValue"] ?? null;
        $taxSetting = Mage::getStoreConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX) ? true : false;
        $incVat = $filter["incVat"] ?? $taxSetting;
        $select = $collection->getSelect();

        $customerGroupId = Mage::getSingleton('customer/session')->getCustomerGroupId();
        $websiteId = $helper->getStore($collection->getStoreId())->getWebsiteId();

        $select->join(
            ["points_index" => $collection->getTable("points_core/product_index")],
            "points_index.product_id = e.entity_id",
            []
        );

        $select->where("points_index.website_id = ?", $websiteId);
        $select->where("points_index.customer_group_id = ?", $customerGroupId);
        $select->where("points_index.type = ?", $type);

        if($taxSetting) {
            // Magento instance use prices including tax
            if($incVat) {
                // User query includes tax
                if($minValue !== null) {
                    $select->where("points_index.price >= ?", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("points_index.price <= ?", $maxValue);
                }
            }
            else {
                // User query excludes tax
                $select->join(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
                $select->join(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

                if($minValue !== null) {
                    $select->where("points_index.price >= ? * (1 + (tcr.rate / 100))", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("points_index.price <= ? * (1 + (tcr.rate / 100))", $maxValue);
                }
            }
        }
        else {
            // Magento instance use prices excluding tax
            if ($incVat) {
                // User query includes tax
                $select->join(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
                $select->join(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

                if($minValue !== null) {
                    $select->where("points_index.price >= ? / (1 + (tcr.rate / 100))", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("points_index.price <= ? / (1 + (tcr.rate / 100))", $maxValue);
                }
            } else {
                // User query excludes tax
                if($minValue !== null) {
                    $select->where("points_index.price >= ?", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("points_index.price <= ?", $maxValue);
                }
            }
        }

        return true;
    }
}
