<?php

declare(strict_types=1);

use Points\Core\Total;

/**
 * A limit for the amount of points spent in a single order, both lower and upper limits.
 *
 * @method int getStoreId()
 * @method void setStoreId(int $storeId)
 * @method int getCustomerGroupId()
 * @method void setCustomerGroupId(int $customerGroupId)
 * @method string getType()
 * @method void setType(string $type)
 *
 * @method int getMinValue()
 * @method void setMinValue(int $value)
 * @method int getMinPercent()
 * @method void setMinPercent(int $percent)
 * @method ?int getMaxValue()
 * @method void setMaxValue(?int $value)
 * @method ?int getMaxPercent()
 * @method void setMaxPercent(?int $percent)
 * @method bool getIncludesTax()
 * @method void setIncludesTax(bool $isIncVat)
 *
 * @method string getCreatedAt()
 * @method string getUpdatedAt()
 */
class Points_Core_Model_Limit_Order extends Mage_Core_Model_Abstract {
    public function _construct(): void {
        parent::_construct();

        $this->_init("points_core/limit_order");
    }

    /**
     * @param int $customerGroupId
     * @return $this
     */
    public function loadByStoreTypeCustomerGroupId(
        Mage_Core_Model_Store $store,
        string $type,
        $customerGroupId = Mage_Customer_Model_Group::NOT_LOGGED_IN_ID
    ) {
        /**
         * @var Points_Core_Model_Resource_Limit_Order
         */
        $resource = $this->_getResource();

        $resource->loadByStoreTypeCustomerGroupId($this, $store, $type, $customerGroupId);

        return $this;
    }

    public function setStore(Mage_Core_Model_Store $store): void {
        $this->setStoreId($store->getId());
    }

    public function setCustomerGroup(Mage_Customer_Model_Group $group): void {
        $this->setCustomerGroupId((int)$group->getId());
    }

    public function getTotalMin(int $total): int {
        $minPercent = $this->getMinPercent();
        $minValue = $this->getMinValue();
        $percentMin = (int)ceil($total * $minPercent / 100);

        return max((int)$minValue, $percentMin);
    }

    public function getTotalMax(int $total): int {
        $maxPercent = $this->getMaxPercent();
        $maxValue = $this->getMaxValue();
        $percentMax = $maxPercent !== null ? (int)ceil($total * $maxPercent / 100) : $total;
        $valueMax = $maxValue !== null ? (int)$maxValue : $total;

        return min($total, $percentMax, $valueMax);
    }

    protected function _beforeSave() {
        parent::_beforeSave();

        $this->setData("updated_at", date("Y-m-d\\TG:i:s"));

        return $this;
    }
}

