<?php

declare(strict_types=1);

namespace Points\Core\Total;

use Mage;
use Points\Core\ProviderInterface;
use Points\Core\Amount;
use Mage_Sales_Model_Quote_Address;
use Mage_Sales_Model_Quote_Address_Item;
use Points_Core_Model_Product_Price;

use function Points\Core\spreadAmount;

class Calculator {
    public function __construct() {
    }

    public function fromQuoteAddress(
        Mage_Sales_Model_Quote_Address $address,
        string $type,
        ProviderInterface $provider
    ): QuoteAddress {
        // TODO: getAllNonNominalItems
        $addressItems = $address->getAllItems();
        $quote = $address->getQuote();
        $customerGroupId = $quote->getCustomerGroupId();
        $store = $quote->getStore();
        $productIds = $this->getProductIds($addressItems);
        $resource = Mage::getResourceModel("points_core/product_price");
        $prices = $resource->getMatchingPrices($store, $customerGroupId, $type, $productIds);
        $totals = $this->getItems($addressItems, $prices);

        if($address->getBaseShippingAmount() > 0) {
            $amount = $provider->getQuoteShippingPrice($address);

            $totals[] = new Shipping(
                $address,
                $amount !== null ?
                    new ShippingPoints($address, $amount) :
                    null
            );
        }

        $spreadTotals = $this->spreadTotals($totals);

        return new QuoteAddress($address, $spreadTotals);
    }

    /**
     * @param bool $incVat If the calculations are requested including VAT
     */
    public function calculateSelected(QuoteAddress $addressTotals, int $requestedPoints, bool $incVat): QuoteAddress {
        $pointsMin = 0;
        $pointsMax = 0;
        $reindexed = $addressTotals->getTotals();

        foreach($reindexed as $total) {
            $points = $total->getPoints();

            if($points) {
                $pointsMin += $points->getMin()->getTotalAndTax($incVat);
                $pointsMax += $points->getMax()->getTotalAndTax($incVat);
            }
        }

        $pointsNonMinMax = max($pointsMax - $pointsMin, 0);
        // Fraction of points between min and max to allocate
        $fracNonMinMax = $pointsNonMinMax > 0 ? max($requestedPoints - $pointsMin, 0) / $pointsNonMinMax : 0;

        $rowPoints = $this->spreadTotal(0, $reindexed, function(PointsInterface $points) use($incVat, $fracNonMinMax) {
            $rowMin = $points->getMin();
            $rowMax = $points->getMax();
            $rowMaxPoints = $rowMax->getTotalAndTax($incVat);
            $rowMinPoints = $rowMin->getTotalAndTax($incVat);

            return new Amount(
                $rowMinPoints + $fracNonMinMax * max($rowMaxPoints - $rowMinPoints, 0),
                $incVat,
                $rowMin->getTax() + $fracNonMinMax * max($rowMax->getTax() - $rowMin->getTax(), 0)
            );
        });

        foreach($reindexed as $k => $item) {
            $oldPoints = $item->getPoints();

            if($oldPoints) {
                assert(array_key_exists($k, $rowPoints));

                /**
                 * @var TotalInterface<int>
                 */
                $reindexed[$k] = $item->withPoints(new SelectedPoints($oldPoints, $rowPoints[$k]));
            }
        }

        $rowTotal = $this->spreadTotal(2, $reindexed, function(PointsInterface $points, $total) use($incVat) {
            assert($points instanceof SelectedPoints);

            $rowCurrency = $total->getPrice();
            $rowValuePoints = $points->getValue()->getTotalAndTax($incVat);
            $rowFracPoints = $rowValuePoints > 0 ? $points->getSelected()->getTotalAndTax($incVat) / $rowValuePoints : 0;

            return new Amount(
                $rowFracPoints * $rowCurrency->getTotalAndTax($incVat),
                $incVat,
                $rowFracPoints * $rowCurrency->getTax(),
            );
        });

        foreach($reindexed as $k => $item) {
            $oldPoints = $item->getPoints();

            if($oldPoints) {
                assert(array_key_exists($k, $rowTotal));

                /**
                 * @var TotalInterface<int>
                 */
                $reindexed[$k] = $item->withPointTotalAmount($rowTotal[$k]);
            }
        }

        return new QuoteAddress($addressTotals->getAddress(), $reindexed);
    }

    /**
     * @param list<Mage_Sales_Model_Quote_Address_Item> $items
     * @return list<int>
     */
    private function getProductIds(array $items): array {
        $productIds = [];

        foreach($items as $i) {
            if($i->isDeleted()) {
                continue;
            }

            // Always fetch all product point-prices, they might be part of child-items
            $productIds[] = $i->getProductId();

            // FIXME: Handle Bundle?
        }

        return $productIds;
    }

    /**
     * @param Array<Mage_Sales_Model_Quote_Address_Item> $items
     * @param Array<Points_Core_Model_Product_Price> $prices
     * @return list<Item<float>>
     */
    private function getItems(array $items, array $prices): array {
        $pricedItems = [];

        foreach($items as $item) {
            $productId = $item->getProductId();

            if($item->getParentItem()) {
                // Only consider parents, since children do not have any amounts
                continue;
            }

            if($item->getProductType() === "configurable") {
                // We have a configurable item, those store their price on the
                // child product but report the price on the parent row, we need
                // to use the child item product id to find the correct price:

                /**
                 * @var Mage_Sales_Model_Quote_Address_Item $i
                 */
                foreach($item->getChildren() as $i) {
                    $productId = $i->getProductId();

                    break;
                }
            }

            $pricedItems[] = new Item(
                $item,
                array_key_exists($productId, $prices) ?
                    new ItemPoints($item, $prices[$productId]) :
                    null
            );
        }

        return $pricedItems;
    }

    /**
     * @param list<TotalInterface<float>> $totals
     * @return list<TotalInterface<int>>
     */
    private function spreadTotals(array $totals): array {
        $pointsValue = $this->spreadTotal(
            0,
            $totals,
            function(PointsInterface $points) { return $points->getValue(); }
        );
        $pointsMin = $this->spreadTotal(
            0,
            $totals,
            function(PointsInterface $points) { return $points->getMin(); }
        );
        $pointsMax = $this->spreadTotal(
            0,
            $totals,
            function(PointsInterface $points) { return $points->getMax(); }
        );
        $pointsDiscount = $this->spreadTotal(
            0,
            $totals,
            function(PointsInterface $points) { return $points->getDiscount(); }
        );

        foreach($totals as $k => $v) {
            $oldPoints = $v->getPoints();

            if($oldPoints) {
                assert(array_key_exists($k, $pointsValue));
                assert(array_key_exists($k, $pointsMin));
                assert(array_key_exists($k, $pointsMax));
                assert(array_key_exists($k, $pointsDiscount));

                $points = new SpreadPoints(
                    $pointsValue[$k],
                    $pointsMin[$k],
                    $pointsMax[$k],
                    $pointsDiscount[$k],
                );

                $totals[$k] = $v->withPoints($points);
            }
        }

        /**
         * @var list<Item<int>|Shipping<int>>
         */
        return $totals;
    }

    /**
     * @template P as int
     * @param P $precision
     * @param Array<TotalInterface> $totals
     * @param callable(PointsInterface, TotalInterface): Amount<float> $fn
     * @return Array<Amount<float|int>>
     * @psalm-return Array<Amount<(P is 0 ? int : float)>>
     */
    private function spreadTotal(int $precision, array $totals, $fn): array {
        return spreadAmount(array_filter(array_map(function($total) use($fn) {
            $points = $total->getPoints();

            if( ! $points) {
                return null;
            }

            return $fn($points, $total);
        }, $totals)), $precision);
    }
}
