<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Mage;
use Mage_Customer_Model_Customer;
use Points_Core_Model_Limit_Total;
use Throwable;
use Varien_Object;

use Points\Core\ProviderInterface;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/Base.php";
require_once __DIR__."/../Fixtures/Points.php";

/**
 * @var ?int
 */
$staticTime = null;

function time(): int {
    global $staticTime;

    return $staticTime ?: \time();
}

class CustomerTest extends Base {
    use MatchesSnapshots;

    /**
     * Psalm hasn't figured out that $staticTime is referenced outside.
     *
     * @psalm-suppress UnusedVariable/
     */
    public function setUp(): void {
        global $staticTime;

        $staticTime = strtotime("2020-01-15");

        $this->clearQuotes();
        $this->clearLimits();

        loadPoints();
    }

    /**
     * Psalm hasn't figured out that $staticTime is referenced outside.
     *
     * @psalm-suppress UnusedVariable/
     */
    public function tearDown(): void {
        global $staticTime;

        $staticTime = null;

        MagentoManager::logQueries();
        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();
        $this->clearLimits();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    public function implementLimits(bool $includesTax, int $totalLimit = 10000): void {
        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $limit = Mage::getModel("points_core/limit_total");
        $order = Mage::getModel("sales/order");

        $limit->loadByStoreTypeCustomerGroupId($store, "TEST", 0);
        $limit->addData([
            "store_id" => $store->getId(),
            "type" => "TEST",
            "customer_group_id" => 0,
            "time_interval" => Points_Core_Model_Limit_Total::INTERVAL_MONTH,
            "limit" => $totalLimit,
            "includes_tax" => $includesTax,
        ]);
        $limit->save();

        $order->loadByIncrementId("TEST-1");
        $order->addData([
            "points_type" => "TEST",
            "points_points" => 1337,
            "points_tax_points" => 335,
        ]);
        $order->save();

        MagentoManager::reset();
    }

    public function testNoProviders(): void {
        $this->loginCustomer();

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsIncludeVat
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        includesVat
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProviderNotEnabled(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(false);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsIncludeVat
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        includesVat
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProvider(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(12345);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsIncludeVat
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        includesVat
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProviderLimited(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits(false);
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(12345);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsIncludeVat
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        includesVat
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testProviderLimitedIncludesTax(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits(true);
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(12345);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
            customer {
                points {
                    id
                    label
                    points
                    pointsIncludeVat
                    pointsSpent
                    redemptionAllowed
                    spendingLimit {
                        spent
                        limit
                        remaining
                        includesVat
                        resetsAt
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testQuoteLimited(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits(false);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 10,
        ]));
        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 10,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteLimitedIncVat(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits(true);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 10,
        ]));
        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 10,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteLimitedBalance(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits(false);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(1);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 10,
        ]));
        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 10,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteLimitedOrder(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementOrderLimits([
            "min_value" => 0,
            "min_percent" => 50,
            "max_percent" => 55,
            "max_value" => null,
        ]);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 10,
        ]));
        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 10,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteLimitedOrder2(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementOrderLimits([
            "min_value" => 1200,
            "min_percent" => 0,
            "max_percent" => null,
            "max_value" => 5000,
        ]);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 10,
        ]));
        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 10,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteLimitedOrder3(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementOrderLimits([
            "min_value" => 1200,
            "min_percent" => 0,
            "max_percent" => null,
            "max_value" => 4000,
        ]);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 11,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testQuoteLimitedTotal(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->implementLimits(true, 1000);
        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");
        $mockProvider = $this->createMock(ProviderInterface::class);
        $quote = $checkoutSession->getQuote();

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 3,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    availablePointPayments {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    rejectionReasons {
                        __typename
                        ... on PointsQuoteRejectionReasonBalance {
                            requested
                            includesVat
                            balance
                        }
                        ... on PointsQuoteRejectionReasonTotalLimit {
                            requested
                            includesVat
                            remaining
                            limit
                            resetsAt
                        }
                        ... on PointsQuoteRejectionReasonOrderLimit {
                            requested
                            includesVat
                            limit
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
