<?php

declare(strict_types=1);

namespace Points\Core;

/**
 * @template-covariant T as float|int
 * @immutable
 */
final class Amount {
    /**
     * @var T
     */
    private $value;

    /**
     * @var T
     */
    private $tax;

    /**
     * If set to true, $value includes tax and total is $value, otherwise
     * $value + $tax.
     *
     * @var bool
     */
    private $valueIncludesTax;

    /**
     * @param T $value
     * @param T $tax
     */
    public function __construct($value, bool $valueIncludesTax, $tax) {
        $this->value = $value;
        $this->valueIncludesTax = $valueIncludesTax;
        $this->tax = $tax;
    }

    /**
     * @return T
     */
    public function getValue() {
        return $this->value;
    }

    /**
     * @return T
     */
    public function getTax() {
        return $this->tax;
    }

    public function getValueIncludesTax(): bool {
        return $this->valueIncludesTax;
    }

    /**
     * @return T
     */
    public function getTotalInclTax() {
        /**
         * @var T
         */
        return $this->valueIncludesTax ? $this->value : $this->value + $this->tax;
    }

    /**
     * @return T
     */
    public function getTotalExclTax() {
        /**
         * @var T
         */
        return $this->valueIncludesTax ? $this->value - $this->tax : $this->value;
    }

    /**
     * @return T
     */
    public function getTotalAndTax(bool $incTax) {
        return $incTax ? $this->getTotalInclTax() : $this->getTotalExclTax();
    }

    /**
     * Adds this amount with the given amount and produces a new instance containing the sum.
     *
     * NOTE: Will truncate floats to 3 decimals.
     *
     * @param Amount<T> $b
     * @return Amount<T>
     */
    // TODO: Precision?
    // TODO: Separate type for floats?
    public function add(Amount $b) {
        $adjustment = 0;

        if($this->valueIncludesTax !== $b->getValueIncludesTax()) {
            // If they are not equal and B includes tax, we have to remove tax from value
            /**
             * @var int|float
             */
            $adjustment = $b->getValueIncludesTax() ? -$b->getTax() : $b->getTax();
        }

        $value = $this->value + $b->getValue() + $adjustment;
        $tax = $this->tax + $b->getTax();

        /**
         * @var Amount<T>
         */
        return new Amount(
            is_int($this->value) ? (int)$value : truncate3($value),
            $this->valueIncludesTax,
            is_int($this->value) ? (int)$tax : truncate3($tax),
        );
    }

    /**
     * @template P of int
     * @param P $precision
     * @return Amount<float|int>
     * @psalm-return Amount<(P is 0 ? int : float)>
     */
    public function multiply(float $m, int $precision) {
        if($this->valueIncludesTax) {
            $value = round((float)$this->value * $m, $precision);
            $tax = min($value, round((float)$this->tax * $m, $precision));
        }
        else {
            ["value" => $value, "tax" => $tax ] = spreadPrice([
                "value" => (float)$this->value * $m,
                "tax" => (float)$this->tax * $m,
            ], $precision);
        }

        return new Amount(
            $precision === 0 ? (int)$value : truncate3($value),
            $this->valueIncludesTax,
            $precision === 0 ? (int)$tax : truncate3($tax)
        );
    }

    /**
     * Scales the amount down to $other including or excluding VAT with given
     * precision.
     *
     * @template P as int
     * @param int|float $other
     * @param P $precision
     * @return Amount<float>
     * @psalm-return Amount<(P is 0 ? int : float)>
     */
    public function scaleTo($other, bool $incVat, int $precision): Amount {
        $total = $this->getTotalAndTax($incVat);

        return $this->multiply($total > 0 ? $other / $total : 0, $precision);
    }
}
