<?php

declare(strict_types=1);

namespace Points\Core\Issues;

use function Fixtures\loadPoints;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Fixtures\TaxRates;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Group;
use Points\Core\Base;
use Points\Core\ProviderInterface;
use Throwable;
use Varien_Object;

require_once __DIR__."/../Base.php";
require_once __DIR__."/../../Fixtures/Points.php";
require_once __DIR__."/../../../vendor/crossroads/mageql-catalog/test/Fixtures/TaxRates.php";

/**
 * item.base_points_amount = 1450.79
 * item.base_points_tax_amount = 362.70
 *
 * quote.points_amount_included = 1450.79
 * quote.points_tax_amount_included = 362.70
 * quote.points_amount_excluded = 162
 * quote.points_tax_amount_excluded = 0
 *
 * These two should match the ones above since excluded > 0:
 *
 * quote.base_points_amount = **1421.65**
 * quote.base_points_tax_amount = **355.41**
 */
class PrecisionLossTest extends Base {
    public function setUp(): void {
        loadPoints();

        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearLimits();
        $this->clearQuotes();

        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $simple = Mage::getModel("catalog/product");
        $virtual = Mage::getModel("catalog/product");
        $simplePrice = Mage::getModel("points_core/product_price");
        $virtualPrice = Mage::getModel("points_core/product_price");

        $simple->setStoreId($store->getId());
        $simple->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $simple->setPrice(1450.79 + 362.70);
        $simple->save();

        $virtual->setStoreId($store->getId());
        $virtual->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $virtual->addData([
            "price" => 162,
            "store_id" => $store->getId(),
            "tax_class_id" => 0,
        ]);
        $virtual->save();

        $simplePrice->loadByStoreProductTypeCustomerGroupId($store, $simple, "TEST", Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        $simplePrice->setPrice(77980);
        $simplePrice->setMinPrice(0);
        $simplePrice->setMaxPrice(null);
        $simplePrice->save();

        $virtualPrice->loadByStoreProductTypeCustomerGroupId($store, $virtual, "TEST", Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        $virtualPrice->delete();

        Mage::getSingleton("points_core/product_index")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();

        loadPoints();

        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $simple = Mage::getModel("catalog/product");
        $virtual = Mage::getModel("catalog/product");

        $simple->setStoreId($store->getId());
        $simple->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $simple->setPrice(12.34);
        $simple->save();

        $virtual->setStoreId($store->getId());
        $virtual->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $virtual->addData([
            "price" => 9.99,
            "store_id" => $store->getId(),
            "tax_class_id" => TaxRates::getTaxClass(TaxRates::PRODUCT_TAX_CLASS)->getId(),
        ]);
        $virtual->save();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();

        loadPoints();

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    public function testTest(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(0);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(999999);
        $quote->setPointsWantedIncludesTax(false);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testIncVatTest(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(0);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));

        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(999999);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);
    }
}
