<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Sales_Model_Order;
use Points_Core_PointPaymentRequiredException;
use Throwable;
use Varien_Object;

use Spatie\Snapshots\MatchesSnapshots;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;

require_once __DIR__."/Base.php";
require_once __DIR__."/../Fixtures/Points.php";

class OrderTest extends Base {
    public static function setUpBeforeClass(): void {
        loadPoints();
    }

    public function setUp(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearQuotes();

        MagentoManager::initAdmin();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData("carriers/freeshipping/active", 1, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    public function testSimple(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(9999);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        $quote->getPayment()->importData([
            "method" => "free",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->method("getUpdatedCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceEnd")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->never())
            ->method("onQuoteSubmitFailure");
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitAfter")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();

        $service = Mage::getModel("sales/service_quote", $quote);
        $service->submitAll();

        $order = $service->getOrder();

        $this->assertOrderMatchesSnapshot($order);
    }

    public function testVirtual(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(9999);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        $quote->getPayment()->importData([
            "method" => "checkmo",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->method("getUpdatedCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceEnd")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->never())
            ->method("onQuoteSubmitFailure");
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitAfter")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();

        $service = Mage::getModel("sales/service_quote", $quote);
        $service->submitAll();

        $order = $service->getOrder();

        $this->assertOrderMatchesSnapshot($order);
    }

    public function testVirtualNoPayment(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        $quote->getPayment()->importData([
            "method" => "checkmo",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();

        $service = Mage::getModel("sales/service_quote", $quote);

        // We don't expect this exception before this
        $this->expectException(Points_Core_PointPaymentRequiredException::class);

        $service->submitAll();
    }

    public function testVirtualNoPoints(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        $quote->getPayment()->importData([
            "method" => "checkmo",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();

        $service = Mage::getModel("sales/service_quote", $quote);

        // We don't expect this exception before this
        $this->expectException(Points_Core_PointPaymentRequiredException::class);

        $service->submitAll();
    }

    public function testVirtualNoPointsBypass(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        $quote->getPayment()->importData([
            "method" => "checkmo",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->collectTotals();

        $quote->setBypassPointRequirement(true);

        $service = Mage::getModel("sales/service_quote", $quote);

        $service->submitAll();

        $order = $service->getOrder();

        $this->assertOrderMatchesSnapshot($order);
    }
}
