<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use MageQL_Core_Model_Context;
use Mage_Sales_Model_Quote;

use Points\Core\Currency;
use Points\Core\Extension\Quote as QuoteExt;
use Points\Core\Extension\QuoteAddress as QuoteAddressExt;
use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;

class Quote {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @var QuoteExt
     */
    protected $quote;

    /**
     * @var QuoteAddressTotal
     */
    protected $total;

    /**
     * @param QuoteExt $quote
     */
    public function __construct(
        string $type,
        ProviderInterface $provider,
        Mage_Sales_Model_Quote $quote,
        QuoteAddressTotal $total
    ) {
        $this->type = $type;
        $this->provider = $provider;
        $this->quote = $quote;
        $this->total = $total;
    }

    public function getId(): string {
        return $this->type;
    }

    public function getLabel(array $args, MageQL_Core_Model_Context $ctx): string {
        return $this->provider->getLabel($ctx->getStore());
    }

    public function getCurrencyRequired(): bool {
        $excluded = array_sum(array_map("Points\Core\Amount::totalInclTax", array_filter($this->total->getCurrency(), "Points\Core\Currency::isNotIncluded")));

        if($excluded > 0) {
            return true;
        }

        $value = array_sum(array_map("Points\Core\Amount::totalInclTax", $this->total->getPointsValue()));
        $max = array_sum(array_map("Points\Core\Amount::totalInclTax", $this->total->getPointsMax()));

        return $max < $value;
    }

    public function getPoints(): QuotePoints {
        return new QuotePoints($this->total, $this->quote, $this->provider);
    }

    public function getCurrency(): QuoteCurrency {
        return new QuoteCurrency($this->total);
    }

    public function getShipping(): ?QuoteShipping {
        $amounts = $this->getShippingAmounts();

        if( ! $amounts) {
            return null;
        }

        return new QuoteShipping($this->total, $amounts);
    }

    public function getDiscount(): ?QuoteDiscount {
        $discount = array_values(array_filter($this->total->getDiscount()));
        $discountPoints = array_values(array_filter($this->total->getPointsDiscount()));

        if(empty($discount)) {
            return null;
        }

        return new QuoteDiscount($discount, $discountPoints);
    }

    /**
     * @return ?array{currency:Currency, discount:?Currency}
     */
    protected function getShippingAmounts(): ?array {
        $currency = $this->total->getShippingCurrency();
        $discount = $this->total->getDiscount()[QuoteAddressTotal::ROW_SHIPPING] ?? null;

        if( ! $currency) {
            return null;
        }

        return [
            "currency" => $currency,
            "discount" => $discount,
        ];
    }
}
